package com.xebialabs.xlrelease.repository.sql.persistence.configuration

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.id.CiUid
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.repository.sql.persistence.FolderPersistence
import com.xebialabs.xlrelease.repository.sql.persistence.Schema._
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.configuration.ConfigurationPersistence.ConfigurationRow
import com.xebialabs.xlrelease.repository.sql.persistence.data.ConfigurationReferenceRow
import com.xebialabs.xlrelease.utils.FolderId
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.sql.ResultSet

@IsTransactional
class TriggerConfigurationReferencePersistence(override val folderPersistence: FolderPersistence,
                                               override val jdbcTemplate: JdbcTemplate,
                                               override implicit val dialect: Dialect)
  extends ConfigurationReferencePersistenceCommon {

  override protected val table: ConfigurationRefs = TRIGGER_CONFIGURATION_REFS

  override protected def STMT_GET_REFS: String =
    s"""|SELECT
        |   entity.${TRIGGERS.TRIGGER_TITLE},
        |   entity.${TRIGGERS.ID},
        |   folders.${FOLDERS.FOLDER_ID}
        | FROM
        |   ${table.TABLE} refs
        |   JOIN ${TRIGGERS.TABLE} entity ON refs.${table.USED_BY_UID} = entity.${TRIGGERS.CI_UID}
        |   JOIN ${FOLDERS.TABLE} folders ON folders.${FOLDERS.CI_UID} = entity.${TRIGGERS.FOLDER_UID}
        | WHERE
        |   refs.${table.CONFIGURATION_UID} = :configurationCiUid""".stripMargin

  override def findAllByUid(uid: CiUid): Seq[ConfigurationRow] = {
    logger.trace(s"Eager fetching configurations for a trigger with uid: $uid")
    sqlQuery(STMT_ALL_CONFIGURATIONS_BY_UID, params("uid" -> uid), configurationBinaryStreamRowMapper(uid)).toSeq
  }

  override def findAllNonInheritedReferences(folderId: String, queryParameters: Seq[String]): Seq[String] = {
    val sql =
      s"""
         |SELECT DISTINCT
         |  cfg.${CONFIGURATIONS.ID}
         |FROM
         |  ${TRIGGERS.TABLE} tr
         |  INNER JOIN ${FOLDERS.TABLE} folder ON folder.${FOLDERS.CI_UID} = tr.${TRIGGERS.FOLDER_UID}
         |  INNER JOIN ${table.TABLE} refs ON refs.${table.USED_BY_UID} = tr.${TRIGGERS.CI_UID}
         |  INNER JOIN ${CONFIGURATIONS.TABLE} cfg ON cfg.${CONFIGURATIONS.CI_UID} = refs.${table.CONFIGURATION_UID}
         |    AND cfg.${CONFIGURATIONS.FOLDER_CI_UID} IS NOT NULL
         |    AND cfg.${CONFIGURATIONS.FOLDER_CI_UID} != folder.${FOLDERS.CI_UID}
         |WHERE folder.${FOLDERS.FOLDER_ID} = :folderId
         |  OR folder.${FOLDERS.FOLDER_PATH} LIKE :folderPath
         |""".stripMargin
    val folderPartialId = Ids.getName(folderId)
    val folderPath = s"%${Ids.SEPARATOR}$folderPartialId%"
    sqlQuery(sql, params("folderId" -> folderPartialId, "folderPath" -> folderPath), rs => rs.getString(CONFIGURATIONS.ID)).toSeq
  }

  override protected def configurationReferenceRowMapper: RowMapper[ConfigurationReferenceRow] = (rs: ResultSet, _: Int) => {
    val title = rs.getString(TRIGGERS.TRIGGER_TITLE)
    val id = rs.getString(TRIGGERS.ID)
    val folderId = FolderId(rs.getString(FOLDERS.FOLDER_ID))

    ConfigurationReferenceRow(
      title = title,
      id = (folderId / id).absolute
    )
  }


}
