package com.xebialabs.xlrelease.security

import com.xebialabs.xlrelease.domain.distributed.events.{DistributedGroupDeleted, DistributedUserGroupMembershipUpdated}
import com.xebialabs.xlrelease.security.sql.SecurityCacheConfigurationConstants._
import com.xebialabs.xlrelease.service.BroadcastService
import com.xebialabs.xlrelease.config.CacheManagementConstants.SECURITY_USER_GROUP_CACHE_MANAGER
import org.springframework.cache.annotation.{CacheConfig, Cacheable}

@CacheConfig(cacheManager = SECURITY_USER_GROUP_CACHE_MANAGER)
class SqlUserGroupRepositoryCache(userGroupRepository: UserGroupRepository,
                                  broadcastService: BroadcastService) extends UserGroupRepository {

  @Cacheable(cacheNames = Array(SECURITY_USER_GROUPS), key = "#username", unless = "#result.isEmpty")
  override def findGroupsForUser(username: String): Set[String] = {
    userGroupRepository.findGroupsForUser(username)
  }

  override def createGroup(groupName: String): String = {
    userGroupRepository.createGroup(groupName)
  }

  override def createGroups(groupNames: Set[String]): Set[String] = {
    userGroupRepository.createGroups(groupNames)
  }

  override def deleteGroup(groupName: String): Unit = {
    userGroupRepository.deleteGroup(groupName)
    broadcastService.broadcast(DistributedGroupDeleted(groupName), publishEventOnSelf = true)
  }

  override def addUserToGroup(username: String, groupName: String): Unit = {
    userGroupRepository.addUserToGroup(username, groupName)
    broadcastService.broadcast(DistributedUserGroupMembershipUpdated(username), publishEventOnSelf = true)
  }

  override def removeUserFromGroup(username: String, groupName: String): Unit = {
    userGroupRepository.removeUserFromGroup(username, groupName)
    broadcastService.broadcast(DistributedUserGroupMembershipUpdated(username), publishEventOnSelf = true)
  }

  override def updateGroupsMembershipForUser(username: String, groupNames: Set[String]): Unit = {
    userGroupRepository.updateGroupsMembershipForUser(username, groupNames)
    broadcastService.broadcast(DistributedUserGroupMembershipUpdated(username), publishEventOnSelf = true)
  }

  override def addUserToGroups(username: String, groupNames: Set[String]): Unit = {
    userGroupRepository.addUserToGroups(username, groupNames)
    broadcastService.broadcast(DistributedUserGroupMembershipUpdated(username), publishEventOnSelf = true)
  }

  override def removeUserFromGroups(username: String, groupNames: Set[String]): Unit = {
    userGroupRepository.removeUserFromGroups(username, groupNames)
    broadcastService.broadcast(DistributedUserGroupMembershipUpdated(username), publishEventOnSelf = true)
  }
}


