package com.xebialabs.xlrelease.upgrade.db

import com.xebialabs.deployit.security.PermissionEditor
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.deployit.server.api.upgrade.Upgrade
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.{ACTIVE_STATUSES, PLANNED, TEMPLATE}
import com.xebialabs.xlrelease.repository.Ids.ROOT_FOLDER_ID
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.RELEASES
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import com.xebialabs.xlrelease.repository.sql.persistence.{FolderPersistence, PersistenceSupport}
import com.xebialabs.xlrelease.security.XlrPermissionLabels.PermissionOperations
import com.xebialabs.xlrelease.upgrade.UpgradeSupport.{BatchSupport, ParallelSupport}
import grizzled.slf4j.Logging

import java.util
import scala.jdk.CollectionConverters._

trait BasePermissionUpgrade extends Upgrade with BatchSupport with ParallelSupport with PersistenceSupport with Logging {

  def folderPersistence: FolderPersistence

  def permissionEditor: PermissionEditor

  /**
   * Find folder and release Cis that contain the given existingPermission and apply the newPermission
   */
  def grantPermissionToRolesWithPermission(existingPermission: Permission, newPermission: Permission): Boolean = {
    logger.info(s"Adding permission ${newPermission.label()}")
    val securedUids = findSecuredFolderCis ++ findRootReleaseCis
    doInBatch(securedUids) { batch =>
      doInParallel(batch.items) { securedUid =>
        val existingRoleAssignments = permissionEditor.readPermissions(securedUid).asScala

        val rolesToAssignNewPermission = existingRoleAssignments.filter {
          case (_, permissions) => permissions.contains(existingPermission) && !permissions.contains(newPermission)
        }.keys.map(role => (role, Seq(newPermission).asJava)).toMap.asJava

        permissionEditor.updatePermissions(securedUid, rolesToAssignNewPermission, new util.HashMap())
      }
    }
    logger.info(s"Finished adding permission ${newPermission.label()}")
    true
  }

  /**
   * Find folders that contain the given team name and grant them the upgradePermission
   */
  def grantPermissionToRoleWithName(teamName: String, newPermissions: Permission*): Boolean = {
    logger.info(s"Adding permission ${newPermissions.map(_.label()).mkString(", ")} to $teamName")
    val securedUids = findSecuredFolderCis
    doInBatch(securedUids) { batch =>
      doInParallel(batch.items) { securedUid =>
        val existingRoleAssignments = permissionEditor.readPermissions(securedUid).asScala

        val rolesToAssignNewPermission = existingRoleAssignments
          .filter(_._1.getName == teamName)
          .map { case (role, existingPermissions) =>
            (role, newPermissions.filter(!existingPermissions.contains(_)).asJava)
          }.asJava

        permissionEditor.updatePermissions(securedUid, rolesToAssignNewPermission, new util.HashMap())
      }
    }
    logger.info(s"Finished adding permission ${newPermissions.map(_.label()).mkString(", ")} to $teamName")
    true
  }

  private def findSecuredFolderCis: Seq[String] = {
    folderPersistence.findDescendantsById(ROOT_FOLDER_ID).map(r => r.securityUid.toString).distinct.toSeq
  }

  private val STMT_FIND_RELEASE_CIS =
    s"""
       |SELECT
       |  r.${RELEASES.CI_UID}
       |FROM
       |  ${RELEASES.TABLE} r
       |WHERE
       |  r.${RELEASES.STATUS} IN (:statuses) AND
       |  r.${RELEASES.CI_UID} = r.${RELEASES.SECURITY_UID}
     """.stripMargin

  private def findRootReleaseCis: Seq[String] = {
    val statuses = (ACTIVE_STATUSES :+ PLANNED :+ TEMPLATE).toSeq.map(_.value()).asJava
    sqlQuery(STMT_FIND_RELEASE_CIS, params("statuses" -> statuses), rs => rs.getString(RELEASES.CI_UID)).distinct.toSeq
  }
}
