package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.repository.sql.SqlRepository
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.LICENSE
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.params
import grizzled.slf4j.Logging
import org.springframework.dao.DuplicateKeyException
import org.springframework.jdbc.core.{JdbcTemplate, RowMapper}

import java.nio.charset.{Charset, StandardCharsets}
import java.sql.ResultSet
import scala.util.Using


class LicensePersistence(override val jdbcTemplate: JdbcTemplate,
                         override implicit val dialect: Dialect)
  extends SqlRepository with PersistenceSupport with LimitOffset with Logging {

  val LICENSE_FILE_ENCODING: Charset = StandardCharsets.UTF_8

  private val STMT_INSERT_LICENSE_CONFIGURATION =
    s"""|INSERT INTO ${LICENSE.TABLE}
        |   ( ${LICENSE.ID}
        |   , ${LICENSE.CONTENT}
        |   )
        | VALUES
        |   ( :licenseId
        |   , :content
        |   )
        """.stripMargin


  def insert(id: String, data: String): Unit = {
    try {
      val parmap = params("licenseId" -> id)
      sqlExecWithContent(STMT_INSERT_LICENSE_CONFIGURATION,
        parmap,
        "content" -> data,
        _ => ())
    } catch {
      case ex: DuplicateKeyException =>
        throw new IllegalArgumentException(s"License id '$id' already exists", ex)
    }
  }

  private val STMT_UPDATE_LICENSE_CONFIGURATION =
    s"""|UPDATE ${LICENSE.TABLE}
        | SET
        |  ${LICENSE.CONTENT} = :content
        | WHERE
        |  ${LICENSE.ID} = :licenseId
        """.stripMargin

  def update(id: String, data: String): Unit = {
    sqlExecWithContent(
      STMT_UPDATE_LICENSE_CONFIGURATION,
      params("licenseId" -> id),
      "content" -> data,
      checkCiUpdated(id)
    )
  }

  private val STMT_EXISTS: String = s"SELECT COUNT(*) FROM ${LICENSE.TABLE} WHERE ${LICENSE.ID} = :licenseId"

  def exists(licenseId: String): Boolean = {
    sqlQuery(STMT_EXISTS, params("licenseId" -> licenseId), _.getInt(1) > 0).head
  }

  private val licenseBinaryStreamRowMapper: RowMapper[String] = (rs: ResultSet, _: Int) => {
    Using.resource(rs.getBinaryStream(LICENSE.CONTENT))(decompress)
  }

  private val STMT_FIND_BY_ID =
    s"""|SELECT
        |   conf.${LICENSE.CONTENT}
        | FROM ${LICENSE.TABLE} conf
        | WHERE conf.${LICENSE.ID} = :licenseId""".stripMargin

  def findById(licenseId: String): Option[String] = {
    logger.trace(s"Loading License with id: $licenseId")
    sqlQuery(STMT_FIND_BY_ID, params("licenseId" -> licenseId), licenseBinaryStreamRowMapper).headOption
  }

}
