package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.status.PhaseStatus.PLANNED
import com.xebialabs.xlrelease.domain.status.TaskStatus
import com.xebialabs.xlrelease.domain.status.TaskStatus.SKIPPED_IN_ADVANCE
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.events.XLReleaseOperations.{publishEvents, runActionInterceptors}
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository.Ids.releaseIdFrom
import com.xebialabs.xlrelease.repository.RetryTitleGenerator.getNextTitle
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.persistence.DependencyPersistence
import com.xebialabs.xlrelease.security.PermissionChecker
import com.xebialabs.xlrelease.utils.CiHelper.{TO_ID, rewriteWithNewId}
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper.scanAndBuildNewVariables
import org.springframework.stereotype.Component

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Component
class SqlPhaseRestart(taskBackup: TaskBackup,
                      permissions: PermissionChecker,
                      releaseRepository: ReleaseRepository,
                      val phaseRepository: PhaseRepository,
                      ciIdService: CiIdService,
                      dependencyService: DependencyService,
                      taskRepository: TaskRepository,
                      dependencyPersistence: DependencyPersistence,
                      facetRepositoryDispatcher: FacetRepositoryDispatcher)
                     (implicit eventBus: XLReleaseEventBus)
  extends PhaseRestart
    with ChangeProcessor {

  import PhaseRestart._

  /**
   * Should only be called from Actor
   */
  override def restartPhases(releaseId: String,
                             phaseId: String,
                             taskId: String,
                             phaseVersion: PhaseVersion = PhaseVersion.ALL,
                             resumeRelease: Boolean = false,
                             release: Release): RestartPhaseResult = {
    // backup the ones that didnt execute yet as their variables are gonna be frozen in Phase.close and lost like tears in rain
    release.getAllTasks.asScala.filter(_.canBeBackup).foreach(task => taskBackup.backupTask(task, release))
    val original = cloneCi(release)
    checkCanRestartPhases(release, phaseId, taskId, phaseVersion, resumeRelease)
    val phasesToRestore: Seq[Phase] = getPhasesToRestore(release, phaseId, phaseVersion).asScala.toSeq

    if (phasesToRestore.isEmpty) {
      throw new RestartPhasesException(releaseId, phaseId, taskId, phaseVersion, resumeRelease, s"There are no $phaseVersion phases to restart from $phaseId")
    }

    checkIfPhaseCanBeRestartedFrom(release.getTask(taskId), phasesToRestore.head, permissions)

    val restoredPhases = restorePhases(phasesToRestore.asJava)
    val changes = release.restorePhases(restoredPhases)
    changes.addAll(release.getCurrentPhase.close)
    changes.addAll(skipTasksBefore(taskId, phasesToRestore.head, restoredPhases.get(0)))
    persistRelease(original, release, changes, restoredPhases)
    RestartPhaseResult(
      release,
      phasesToRestore.map(_.getId),
      restoredPhases.asScala.map(_.getId).toSeq
    )
  }

  /**
   * Should only be called from Actor
   */
  override def restorePhases(phasesToRestore: JList[Phase]): JList[Phase] = phasesToRestore.asScala.map(originalPhase => {
    val restoredPhase: Phase = restorePhase(originalPhase)
    restoreTasks(originalPhase.getTasks, restoredPhase.getTasks)
    restoredPhase
  }).asJava

  /**
   * Should only be called from Actor
   */
  private def restorePhase(phase: Phase): Phase = {
    val restoredPhaseId = ciIdService.getUniqueId(Type.valueOf(classOf[Phase]), releaseIdFrom(phase.getId))
    val copiedPhase = cloneCi(phase)
    rewriteWithNewId(copiedPhase, restoredPhaseId)
    val lastGeneratedTitle: String = phase.getRelease.getPhases.asScala.findLast(_.getOriginId == phase.getId).map(_.getTitle).getOrElse(phase.getTitle)
    copiedPhase.setTitle(getNextTitle(lastGeneratedTitle))
    copiedPhase.setStatus(PLANNED)
    copiedPhase.setStartDate(null)
    copiedPhase.setEndDate(null)
    copiedPhase.setOriginId(phase.getId)
    copiedPhase
  }

  /**
   * For each original task, replace recursively the copied task with the last planned version of the original task.
   * Should only be called from Actor
   */
  private def restoreTasks(originalTasks: JList[Task], newTasks: JList[Task]): Unit = {
    for (i <- originalTasks.asScala.indices) {
      val restoredTask: Task = taskBackup.restoreTask(newTasks.get(i), originalTasks.get(i).getId, inMemory = true)
      restoredTask.getAllTasks.forEach(subTask => subTask.setStatus(TaskStatus.PLANNED))
      restoredTask.clearComments()
      restoredTask.clearModificationAttributes()
      restoredTask.setStartDate(null)
      restoredTask.setEndDate(null)
      restoredTask match {
        case task: GateTask => task.getConditions.asScala.foreach(_.setChecked(false))
        case _ =>
      }
      newTasks.set(i, restoredTask)
    }
  }

  override def updateGatesReferencingPhases(release: Release, phasesToRestoreIds: Seq[String], restoredPhasesIds: Seq[String]): Unit = {
    val originToRestored: Map[String, String] = phasesToRestoreIds.zip(restoredPhasesIds).toMap

    val dependencies = dependencyService.findActiveIncomingDependencies(release.getId)

    val dependenciesToUpdate = dependencies.filter { dependency =>
      originToRestored.exists { case (originalPhaseId, restoredPhaseId) =>
        if (dependency.getTargetId != null && dependency.getTargetId.startsWith(originalPhaseId)) {
          dependency.setTargetId(dependency.getTargetId.replace(originalPhaseId, restoredPhaseId))
          true
        } else {
          false
        }
      }
    }

    dependencyPersistence.batchUpdateDependencyTargets(dependenciesToUpdate.toSet)
  }

  private def skipTasksBefore(taskId: String, originalPhase: Phase, restoredPhase: Phase): Changes = {
    val originalTasks = originalPhase.getAllTasks.asScala
    val restoredTasks = restoredPhase.getAllTasks.asScala
    val taskIndexToRestartFrom = originalTasks.map(TO_ID(_)).indexOf(taskId)
    val changes: Changes = new Changes
    if (restoredTasks.nonEmpty && taskIndexToRestartFrom != -1) {
      val restoredSkipTaskId = restoredTasks.take(taskIndexToRestartFrom + 1).map(_.getId).last

      restoredTasks.take(taskIndexToRestartFrom).foreach {
        case task: TaskGroup =>
          if (!restoredSkipTaskId.startsWith(task.getId)) {
            changes.addAll(task.markAsDone(task.getId, SKIPPED_IN_ADVANCE))
          }
        case task =>
          changes.addAll(task.markAsDone(task.getId, SKIPPED_IN_ADVANCE))
      }
    }
    
    changes
  }

  private def persistRelease(original: Release, release: Release, changes: Changes, restoredPhases: JList[Phase]): Unit = {
    runActionInterceptors(changes.getOperations)
    scanAndBuildNewVariables(release, release, ciIdService)
    releaseRepository.update(original, release)
    processPhases(changes)

    changes.getUpdatedItems.asScala
      .filter(_.isInstanceOf[Task])
      .foreach(task => taskRepository.updateTaskProperties(task.asInstanceOf[Task]))
    facetRepositoryDispatcher.liveRepository.createFromTasks(restoredPhases.asScala.flatMap(_.getAllTasks.asScala).toSeq)

    publishEvents(changes.getOperations)
  }
}
