package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.deployit.security.Permissions
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.api.v1.forms.{ReleaseOrderMode, RiskStatus, RiskStatusWithThresholds}
import com.xebialabs.xlrelease.db.sql.SqlBuilder._
import com.xebialabs.xlrelease.db.sql.{Sql, SqlBuilder}
import com.xebialabs.xlrelease.domain.ReleaseKind
import com.xebialabs.xlrelease.domain.status.ReleaseStatus.TEMPLATE
import com.xebialabs.xlrelease.domain.status.{FlagStatus, PhaseStatus, ReleaseStatus}
import com.xebialabs.xlrelease.reports.filters.ReportFilter
import com.xebialabs.xlrelease.repository.Ids.{SEPARATOR, getName}
import com.xebialabs.xlrelease.repository.sql.persistence.CiId._
import com.xebialabs.xlrelease.repository.sql.persistence.ReleasesSqlBuilder._
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{DEPLOYMENT_TASK_REPORTING_RECORD => DEPL, ITSM_TASK_REPORTING_RECORD => ITSM, _}
import com.xebialabs.xlrelease.repository.sql.persistence.Utils._
import com.xebialabs.xlrelease.utils.FolderId

import java.util.Date
import scala.jdk.CollectionConverters._

object ReleasesSqlBuilder {
  val alias: String = "rel"
  val dataAlias: String = "d"
  val folderAlias: String = "f"
  val templateAlias: String = "t"
  val phaseAlias: String = "ph"

  val CURRENT_PHASE_TITLE: String = "currentPhaseTitle"

  def selectReleaseFullPathSql(dialect: Dialect): String = selectReleaseFullPathSql(alias, folderAlias, RELEASES.RELEASE_ID, dialect)

  def selectReleaseFullPathSql(tableAlias: String, folderTableAlias: String, columnAlias: String, dialect: Dialect): String = dialect match {
    case DerbyDialect(_) =>
      s"$folderTableAlias.${FOLDERS.FOLDER_PATH} || '$SEPARATOR' || $folderTableAlias.${FOLDERS.FOLDER_ID} || '$SEPARATOR' || $tableAlias.${RELEASES.RELEASE_ID} AS $columnAlias"
    case CommonDialect(_) | MSSQLDialect(_) =>
      s"CONCAT($folderTableAlias.${FOLDERS.FOLDER_PATH}, '$SEPARATOR', $folderTableAlias.${FOLDERS.FOLDER_ID}, '$SEPARATOR', $tableAlias.${RELEASES.RELEASE_ID}) AS $columnAlias"
    case OracleDialect(_) | Db2Dialect(_) =>
      s"CONCAT(CONCAT(CONCAT(CONCAT($folderTableAlias.${FOLDERS.FOLDER_PATH}, '$SEPARATOR'), $folderTableAlias.${FOLDERS.FOLDER_ID}), '$SEPARATOR'), $tableAlias.${RELEASES.RELEASE_ID}) $columnAlias"
  }
}

// scalastyle:off number.of.methods
class ReleasesSqlBuilder(implicit dialect: Dialect) extends SqlBuilder[ReleasesSqlBuilder] with SecurableSqlBuilder {

  private val joinFolders = s"INNER JOIN ${FOLDERS.TABLE} $folderAlias ON $folderAlias.${FOLDERS.CI_UID} = $alias.${RELEASES.FOLDER_CI_UID}"
  private val joinReleaseData = s"INNER JOIN ${RELEASES_DATA.TABLE} $dataAlias ON $alias.${RELEASES.CI_UID} = $dataAlias.${RELEASES_DATA.CI_UID}"
  private val joinTemplates = s"LEFT JOIN ${RELEASES.TABLE} $templateAlias ON $alias.${RELEASES.ORIGIN_TEMPLATE_ID} = $templateAlias.${RELEASES.RELEASE_ID}"
  private var joinedTasks: Boolean = false
  private var joinedDeployments: Boolean = false

  private val selectReleasesDataColumns = Seq(
    s"$alias.${RELEASES.CI_UID}",
    s"$alias.${RELEASES.RELEASE_ID}",
    s"$alias.${RELEASES.SECURITY_UID}",
    s"$folderAlias.${FOLDERS.FOLDER_PATH}",
    s"$folderAlias.${FOLDERS.FOLDER_ID}",
    s"$alias.${RELEASES.RISK_SCORE}",
    s"$alias.${RELEASES.TOTAL_RISK_SCORE}",
    s"$dataAlias.${RELEASES_DATA.CONTENT}",
    s"$alias.${RELEASES.SCM_DATA}",
    s"$alias.${RELEASES.PLANNED_DURATION}",
    s"$alias.${RELEASES.STATUS}",
    s"$alias.${RELEASES.REAL_FLAG_STATUS}",
    s"$alias.${RELEASES.RELEASE_TITLE}",
    s"$alias.${RELEASES.START_DATE}",
    s"$alias.${RELEASES.END_DATE}",
    s"$alias.${RELEASES.IS_OVERDUE_NOTIFIED}",
    s"$alias.${RELEASES.KIND}"
  )

  private def joinPhasesWithAnyStatus(statuses: Set[PhaseStatus]): String = {
    val statusValues = statuses.map(_.value())

    s"""| LEFT OUTER JOIN ${PHASES.TABLE} $phaseAlias
        |   ON $phaseAlias.${PHASES.RELEASE_UID} = $alias.${RELEASES.CI_UID}
        |   AND $phaseAlias.${PHASES.STATUS} IN (${statusValues.mkString("'", "','", "'")})
        """.stripMargin
  }

  private def selectFromReleases(columns: String*) =
    s"""
       |SELECT ${columns.mkString(", ")}
       |FROM ${RELEASES.TABLE} $alias
     """.stripMargin

  def customSelect(columns: String*): ReleasesSqlBuilder = {
    super.select(selectFromReleases(columns: _*))
    this
  }

  def selectReleaseData(): ReleasesSqlBuilder = {
    super.select(selectFromReleases(selectReleasesDataColumns: _*))
    addSelectReleaseDataJoins()
    this
  }

  private def addSelectReleaseDataJoins(): Unit = {
    addJoin(joinReleaseData)
    addJoin(joinFolders)
  }

  def selectReleaseDataAndTemplateData(): ReleasesSqlBuilder = {
    super.select(selectFromReleases(
      selectReleasesDataColumns ++ Seq(
        s"$templateAlias.${RELEASES.RELEASE_ID} AS $TEMP_ORIGIN_TEMPLATE_ID",
        s"$templateAlias.${RELEASES.RELEASE_TITLE} AS $TEMP_ORIGIN_TEMPLATE_TITLE"
      ): _*
    ))
    addSelectReleaseDataJoins()
    addJoin(joinTemplates)
    this
  }

  def selectShortReleaseIdAndOrderCriterion(order: Option[ReleaseOrderMode]): ReleasesSqlBuilder = {
    val args: Seq[Option[String]] = Seq(
      Some(s"$alias.${RELEASES.RELEASE_ID}"),
      Some(s"$folderAlias.${FOLDERS.FOLDER_ID}"),
      order.map {
        case ReleaseOrderMode.risk => RELEASES.RISK_SCORE
        case ReleaseOrderMode.start_date => RELEASES.START_DATE
        case ReleaseOrderMode.end_date => RELEASES.END_DATE
        case ReleaseOrderMode.title => RELEASES.RELEASE_TITLE
        case ReleaseOrderMode.status => RELEASES.STATUS
      }.map(field => s"$alias.$field")
    )
    super.select(selectFromReleases(args.flatten: _*))
    addJoin(joinFolders)
    this
  }


  def selectReleaseId(): ReleasesSqlBuilder = {
    select(selectFromReleases(selectReleaseFullPathSql(dialect)))

    addJoin(joinFolders)
    this
  }

  def selectReleaseIdAndTitle(): ReleasesSqlBuilder = {
    select(selectFromReleases(selectReleaseFullPathSql(dialect), s"$alias.${RELEASES.RELEASE_TITLE}"))

    addJoin(joinFolders)
    this
  }

  def countByStatus(): ReleasesSqlBuilder = {
    select(selectFromReleases(s"$alias.${RELEASES.STATUS}", s"COUNT(DISTINCT $alias.${RELEASES.CI_UID})"))

    addJoin(joinFolders)

    groupBy(s"$alias.${RELEASES.STATUS}")
  }

  def dateRange(): ReleasesSqlBuilder = {
    select(selectFromReleases(s"MIN($alias.${RELEASES.START_DATE})", s"MAX($alias.${RELEASES.END_DATE})"))
  }

  def withReleaseIds(ids: Seq[CiId]): ReleasesSqlBuilder = {
    if (ids.nonEmpty) {
      val questionMarks = ids.map(_ => "?").mkString(",")
      conditions += Sql(s"$alias.${RELEASES.RELEASE_ID} IN ($questionMarks)", ids.map(getName))
    } else {
      nothingToBeFound = true
    }
    this
  }

  def withFolder(folderId: CiId): ReleasesSqlBuilder = {
    if (folderId != null && folderId.nonEmpty) {
      conditions += Sql(s"$folderAlias.${FOLDERS.FOLDER_ID} = ?", Seq(FolderId(folderId).id))
      addJoin(joinFolders)
    }
    this
  }

  def withAncestor(ancestorId: CiId): ReleasesSqlBuilder = {
    if (ancestorId != null && ancestorId.nonEmpty) {
      val folderId = FolderId(ancestorId.normalized).id
      conditions += Sql(s"($folderAlias.${FOLDERS.FOLDER_ID} = ? OR $folderAlias.${FOLDERS.FOLDER_PATH} LIKE ?) ",
        Seq(folderId, s"%$SEPARATOR$folderId%"))
    }
    this
  }

  def withTitle(title: String): ReleasesSqlBuilder = {
    if (title != null && !title.isEmpty) {
      conditions += Sql(s"LOWER($alias.${RELEASES.RELEASE_TITLE}) = ?", Seq(title.toLowerCase()))
    }
    this
  }

  def withTitleLike(titlePart: String): ReleasesSqlBuilder = {
    like(s"$alias.${RELEASES.RELEASE_TITLE}", titlePart)
  }

  def withRootReleaseId(rootReleaseId: CiId): ReleasesSqlBuilder = {
    if (rootReleaseId != null && !rootReleaseId.isEmpty) {
      // ## Technical Debt -- OR exists because of ticket D-22263 and that it should be removed after couple of releases
      conditions += Sql(s"($alias.${RELEASES.ROOT_RELEASE_ID} = ? OR $alias.${RELEASES.ROOT_RELEASE_ID} = ?)",
        Seq(rootReleaseId.normalized, rootReleaseId.shortId))
    }
    this
  }

  def withSCMData(releaseId: Integer): ReleasesSqlBuilder = {
    if (releaseId != null) {
      conditions += Sql(s"$alias.${RELEASES.SCM_DATA} = ?", Seq(releaseId))
    }
    this
  }

  def withStatus(status: ReleaseStatus): ReleasesSqlBuilder = {
    if (status != null) {
      conditions += Sql(s"$alias.${RELEASES.STATUS} = ?", Seq(status.value()))
    }
    this
  }

  def withOneOfStatuses(statuses: Seq[ReleaseStatus]): ReleasesSqlBuilder = {
    if (statuses.nonEmpty) {
      conditions += Sql(s"$alias.${RELEASES.STATUS} IN (${statuses.map(_ => "?").mkString(",")})",
        statuses.map(_.value()))
    }
    this
  }

  def withoutStatuses(statuses: Seq[ReleaseStatus]): ReleasesSqlBuilder = {
    if (statuses.nonEmpty) {
      conditions += Sql(s"$alias.${RELEASES.STATUS} NOT IN (${statuses.map(_ => "?").mkString(",")})",
        statuses.map(_.value()))
    }
    this
  }

  def withoutTemplates(): ReleasesSqlBuilder = {
    conditions += Sql(s"$alias.${RELEASES.STATUS} <> ?", Seq(TEMPLATE.value()))
    this
  }

  def withOneOfFlagStatuses(flagStatuses: Seq[FlagStatus]): ReleasesSqlBuilder = {
    if (flagStatuses.nonEmpty) {
      conditions += Sql(s"$alias.${RELEASES.REAL_FLAG_STATUS} IN (${flagStatuses.map(_ => "?").mkString(",")})",
        flagStatuses.map(s => Integer.valueOf(s.getRisk)))
    }
    this
  }

  def withAutoStart(): ReleasesSqlBuilder = {
    conditions += Sql(s"$alias.${RELEASES.AUTO_START} = ?", Seq(Integer.valueOf(1)))
    this
  }

  def withCalendarToken(calendarToken: String): ReleasesSqlBuilder = {
    conditions += Sql(s"$alias.${RELEASES.CALENDAR_TOKEN} = ?", Seq(calendarToken))
    this
  }

  def withOwner(owner: String): ReleasesSqlBuilder = {
    conditions += Sql(s"$alias.${RELEASES.RELEASE_OWNER} = ?", Seq(owner))
    this
  }

  def withKind(kind: ReleaseKind): ReleasesSqlBuilder = {
    conditions += Sql(s"$alias.${RELEASES.KIND} = ?", Seq(kind.value()))
    this
  }

  def withAnyOfKinds(kinds: Seq[ReleaseKind]): ReleasesSqlBuilder = {
    if (kinds != null && kinds.nonEmpty) {
      val condition = s"$alias.${RELEASES.KIND} IN (${kinds.map(_ => "?").mkString(",")})"
      conditions += Sql(condition, kinds.map(_.value()))
    }
    this
  }

  def withTags(tags: Set[String]): ReleasesSqlBuilder = {
    if (tags != null && tags.nonEmpty) {
      val orderedTags = tags.toSeq.map(_.toLowerCase)
      val tagsIndices = orderedTags.indices
      tagsIndices.map(joinTag).foreach(addJoin)
      conditions += Sql(
        tagsIndices.map(idx => s"tags$idx.${TAGS.VALUE} = ?").mkString(" AND "),
        orderedTags
      )
    }
    this
  }

  def withAnyOfTaskTags(tags: Set[String]): ReleasesSqlBuilder = {
    if (tags != null && tags.nonEmpty) {
      val condition = s"EXISTS ( SELECT 1 FROM ${TASKS.TABLE}" +
        s" INNER JOIN ${TASK_TAGS.TABLE} " +
        s" ON ${TASKS.TABLE}.${TASKS.CI_UID} = ${TASK_TAGS.TABLE}.${TASK_TAGS.CI_UID}" +
        s" WHERE ${TASKS.RELEASE_UID} = ${alias}.CI_UID AND " +
        s" ${TASK_TAGS.VALUE} IN (${tags.toSeq.map(_ => "?").mkString(",")}))"
      conditions += Sql(condition, tags.toSeq.map(_.toLowerCase))
    }
    this
  }

  def withAnyOfTags(tags: Set[String]): ReleasesSqlBuilder = {
    if (tags != null && tags.nonEmpty) {
      val condition =
        s"""$alias.${RELEASES.CI_UID} IN (
              SELECT DISTINCT tags0.${TAGS.CI_UID}
              FROM ${TAGS.TABLE} tags0
              WHERE tags0.${TAGS.VALUE} IN (${tags.toSeq.map(_ => "?").mkString(",")})
            )"""
      conditions += Sql(condition, tags.toSeq.map(_.toLowerCase))
    }
    this
  }

  def withAllTags(tags: Set[String]): ReleasesSqlBuilder = {
    if (tags != null && tags.nonEmpty) {
      val joins = tags.toSeq.zipWithIndex.filter(_._2 > 0).map {
        case (_, index) => s"INNER JOIN ${TAGS.TABLE} tags$index ON tags$index.${TAGS.CI_UID} = tags0.${TAGS.CI_UID}"
      }
      val wheres = tags.toSeq.zipWithIndex.map {
        case (_, index) => s"tags$index.${TAGS.VALUE} = ?"
      }
      val condition =
        s"""$alias.${RELEASES.CI_UID} IN (
              SELECT DISTINCT tags0.${TAGS.CI_UID}
              FROM ${TAGS.TABLE} tags0
              ${joins.mkString(" ")}
              WHERE ${wheres.mkString(" AND ")}
            )"""
      conditions += Sql(condition, tags.toSeq.map(_.toLowerCase))
    }
    this
  }

  private def joinTag(idx: Int): String = {
    s"INNER JOIN ${TAGS.TABLE} tags$idx ON $alias.${RELEASES.CI_UID} = tags$idx.${TAGS.CI_UID}"
  }

  def withDates(from: Date, to: Date): ReleasesSqlBuilder = {
    if (from != null && to != null) {
      conditions += Sql(s"? <= $alias.${RELEASES.END_DATE} AND $alias.${RELEASES.START_DATE} < ?",
        Seq(from, to))
    } else {
      if (from != null) {
        conditions += Sql(s"? <= $alias.${RELEASES.END_DATE}", Seq(from))
      }
      if (to != null) {
        conditions += Sql(s"$alias.${RELEASES.START_DATE} < ?", Seq(to))
      }
    }
    this
  }

  def withEndDateBefore(date: Date): ReleasesSqlBuilder = {
    conditions += Sql(s"$alias.${RELEASES.END_DATE} < ?", Seq(date))
    this
  }

  def withReleaseName(candidateName: String): ReleasesSqlBuilder = {
    if (candidateName != null && !candidateName.isEmpty) {
      conditions += Sql(s"$alias.${RELEASES.RELEASE_ID} LIKE ? ", Seq(s"%$candidateName"))
    }
    this
  }

  def withPermission(permission: Permission, principals: Iterable[String], roleIds: Iterable[String]): ReleasesSqlBuilder = {
    conditions += Sql(
      s"""$alias.${RELEASES.SECURITY_UID} IN (
         |  ${selectCiIdsWithPermission(principals, roleIds)}
         |)
      """.stripMargin,
      Seq(permission.getPermissionName) ++ principals.map(_.toLowerCase) ++ roleIds
    )
    this
  }

  def withPermissionOrOwner(permission: Permission, principals: Iterable[String], roleIds: Iterable[String]): ReleasesSqlBuilder = {
    val releaseOwner = Permissions.getAuthenticatedUserName
    if (releaseOwner != null && releaseOwner.nonEmpty) {
      conditions += Sql(
        s"""($alias.${RELEASES.RELEASE_OWNER} = ? OR
           |  $alias.${RELEASES.SECURITY_UID} IN (${selectCiIdsWithPermission(principals, roleIds)})
           |)
      """.stripMargin,
        Seq(releaseOwner, permission.getPermissionName) ++ principals.map(_.toLowerCase) ++ roleIds
      )
    } else {
      withPermission(permission, principals, roleIds)
    }
    this
  }

  def withPreArchived(preArchived: Boolean = true): ReleasesSqlBuilder = {
    conditions += Sql(s"$alias.${RELEASES.PRE_ARCHIVED} = ?", Seq(preArchived.asInteger))
    this
  }

  private def selectReleasesHavingTaskFacet(facetTable: Table with TaskRecordTable, facetColumn: String, recordAlias: String, values: Set[String]) = {
    val parameters = values.toSeq.map(_.toLowerCase)
    Sql(
      s"""|SELECT DISTINCT ${recordAlias}Rel.${RELEASES.CI_UID} FROM ${RELEASES.TABLE} ${recordAlias}Rel
          |INNER JOIN ${TASKS.TABLE} ${recordAlias}Task ON ${recordAlias}Task.${TASKS.RELEASE_UID} = ${recordAlias}Rel.${RELEASES.CI_UID}
          |INNER JOIN ${facetTable.TABLE} $recordAlias ON $recordAlias.${facetTable.TASK_UID} = ${recordAlias}Task.${TASKS.CI_UID}
          |WHERE LOWER($recordAlias.$facetColumn) IN ${parameters.map(_ => "?").mkString("(", ", ", ")")}""".stripMargin,
      parameters
    )
  }

  def withTotalRiskScoresBetween(fromScore: Integer, toScore: Integer): ReleasesSqlBuilder = {
    if (toScore == -1) {
      conditions += Sql(s" $alias.${RELEASES.TOTAL_RISK_SCORE} >= ?", Seq(fromScore))
    } else if (fromScore == toScore) {
      conditions += Sql(s" $alias.${RELEASES.TOTAL_RISK_SCORE} = ?", Seq(fromScore))
    } else {
      conditions += Sql(s" $alias.${RELEASES.TOTAL_RISK_SCORE} BETWEEN ? AND ? ", Seq(fromScore, toScore))
    }
    this
  }

  protected def whereReleaseHasTaskFacet(table: Table with TaskRecordTable, column: String, recordAlias: String, values: Set[String]): ReleasesSqlBuilder = {
    val parameters = values.filterNot(_.isEmpty)
    if (parameters.nonEmpty) {
      val Sql(subQuery, subParams) = selectReleasesHavingTaskFacet(table, column, recordAlias, parameters)
      conditions += Sql(s"$alias.${RELEASES.CI_UID} IN ($subQuery)", subParams)
    }
    this
  }

  def withChangeNumbers(changeNumbers: Set[String]): ReleasesSqlBuilder = whereReleaseHasTaskFacet(ITSM, ITSM.RECORD, "itsm", changeNumbers)

  def withApplicationNames(applicationNames: Set[String]): ReleasesSqlBuilder = whereReleaseHasTaskFacet(DEPL, DEPL.APPLICATION_NAME, "app", applicationNames)

  def withEnvironmentNames(environmentNames: Set[String]): ReleasesSqlBuilder = whereReleaseHasTaskFacet(DEPL, DEPL.ENVIRONMENT_NAME, "env", environmentNames)

  def orderBy(order: ReleaseOrderMode, forceAscOrDesc: Option[Boolean] = None): ReleasesSqlBuilder = {
    order match {
      case ReleaseOrderMode.start_date =>
        orderBy(s"$alias.${RELEASES.START_DATE} ${getOrderAscOrDesc(forceAscOrDesc, default = false)}")
        orderBy(s"$alias.${RELEASES.RELEASE_TITLE} ASC")

      case ReleaseOrderMode.end_date =>
        orderBy(s"$alias.${RELEASES.END_DATE} ${getOrderAscOrDesc(forceAscOrDesc, default = false)}")
        orderBy(s"$alias.${RELEASES.RELEASE_TITLE} ASC")

      case ReleaseOrderMode.risk =>
        orderBy(s"$alias.${RELEASES.RISK_SCORE} ${getOrderAscOrDesc(forceAscOrDesc, default = false)}")
        orderBy(s"$alias.${RELEASES.TOTAL_RISK_SCORE} DESC")
        orderBy(s"$alias.${RELEASES.RELEASE_TITLE} ASC")

      case ReleaseOrderMode.title =>
        orderBy(s"$alias.${RELEASES.RELEASE_TITLE} ${getOrderAscOrDesc(forceAscOrDesc, default = true)}")
        orderBy(s"$alias.${RELEASES.RELEASE_ID} ASC")
      case ReleaseOrderMode.status =>
        orderBy(s"$alias.${RELEASES.STATUS} ${getOrderAscOrDesc(forceAscOrDesc, default = true)}")
        orderBy(s"$alias.${RELEASES.RELEASE_TITLE} ASC")
    }
    this
  }

  def withRiskStatus(riskStatusWithThresholds: RiskStatusWithThresholds): ReleasesSqlBuilder = {
    val atRiskFrom = Integer.valueOf(riskStatusWithThresholds.getAtRiskFrom)
    val attentionNeededFrom = Integer.valueOf(riskStatusWithThresholds.getAttentionNeededFrom)

    val completedStatuses = Seq(ReleaseStatus.COMPLETED, ReleaseStatus.ABORTED)

    def notCompletedAndSql(sql: String, params: AnyRef*): Sql = {
      Sql(s"($alias.${RELEASES.STATUS} NOT IN(${completedStatuses.map(_ => "?").mkString(",")}) AND $sql)", completedStatuses.map(_.value()) ++ params)
    }

    riskStatusWithThresholds.getRiskStatus match {
      case RiskStatus.AT_RISK => conditions += notCompletedAndSql(s"$alias.${RELEASES.RISK_SCORE} >= ?", atRiskFrom)
      case RiskStatus.ATTENTION_NEEDED =>
        conditions += notCompletedAndSql(s"$alias.${RELEASES.RISK_SCORE} >= ? AND $alias.${RELEASES.RISK_SCORE} < ?", attentionNeededFrom, atRiskFrom)
      case RiskStatus.OK => conditions += notCompletedAndSql(s"$alias.${RELEASES.RISK_SCORE} < ?", attentionNeededFrom)
    }
    this
  }

  private def joinTasks(): Unit = {
    if (!joinedTasks) {
      addJoin(s"INNER JOIN ${TASKS.TABLE} task ON $alias.${RELEASES.CI_UID} = task.${TASKS.RELEASE_UID}")
      joinedTasks = true
    }
  }

  private def joinDeployments(): Unit = {
    if (!joinedDeployments) {
      joinTasks()
      addJoin(s"INNER JOIN ${DEPL.TABLE} depl ON depl.${DEPL.TASK_UID} = task.${TASKS.CI_UID}")
      joinedDeployments = true
    }
  }

  def addPhaseJoinWithAnyStatus(statuses: Set[PhaseStatus]): ReleasesSqlBuilder = {
    addJoin(joinPhasesWithAnyStatus(statuses))
    this
  }

  def addFolderJoin(): ReleasesSqlBuilder = {
    addJoin(joinFolders)
    this
  }

  override def newInstance: ReleasesSqlBuilder = new ReleasesSqlBuilder()

  def withFilters(filters: java.util.List[ReportFilter]): ReleasesSqlBuilder = {
    withFilters(filters.asScala.toSeq)
  }

  def withFilters(filters: Seq[ReportFilter]): ReleasesSqlBuilder = {
    if (filters != null) {
      filters.foreach(_.apply(this))
    }
    this
  }

  def withOriginTemplateId(templateId: String): ReleasesSqlBuilder = {
    if (templateId != null && !templateId.isEmpty) {
      conditions += Sql(s"$alias.${RELEASES.ORIGIN_TEMPLATE_ID} = ?", Seq(getName(templateId)))
    }
    this
  }

  def withCategories(categories: Seq[String]): ReleasesSqlBuilder = {
    if (categories != null && categories.nonEmpty) {

      val GET_RELEASE_UID_BY_CATEGORIES =
        s"""
           |SELECT xrf.${RELEASE_CATEGORY_REFS.sourceColumn}
           |  FROM
           |    ${RELEASE_CATEGORY_REFS.TABLE} xrf
           |    JOIN ${CATEGORIES.TABLE} xc on xrf.${RELEASE_CATEGORY_REFS.targetColumn} = xc.${CATEGORIES.CI_UID}
           |  WHERE
           |    xc.${CATEGORIES.CATEGORY} IN (${categories.map(_ => "?").mkString(",")})
           |""".stripMargin

      conditions += Sql(s"$alias.${RELEASES.CI_UID} IN ($GET_RELEASE_UID_BY_CATEGORIES)", categories)
    }
    this
  }
}

// scalastyle:on number.of.methods
