package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.deployit.checks.Checks.IncorrectArgumentException
import com.xebialabs.license.LicenseProperty
import com.xebialabs.license.LicenseVersion4.Edition4
import com.xebialabs.license.service.LicenseService
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.status.ReleaseStatus
import com.xebialabs.xlrelease.domain.{Release, ReleaseKind}
import com.xebialabs.xlrelease.exception.LicenseLimitReachedException
import com.xebialabs.xlrelease.limits.LimitEnforcer
import com.xebialabs.xlrelease.limits.LimitEnforcer.LimitType
import com.xebialabs.xlrelease.plugins.dashboard.domain.Dashboard
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.repository.sql.ReleasePersistenceInterceptor.{MAX_NO_WORKFLOW_TEMPLATES_ALLOWED, UNRESTRICTED_LICENSE_EDITIONS}
import com.xebialabs.xlrelease.repository.sql.persistence.TaskPersistence
import com.xebialabs.xlrelease.utils.TenantContext
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils
import org.springframework.util.StringUtils.hasText

import scala.jdk.CollectionConverters._

@Component
@IsTransactional
class ReleasePersistenceInterceptor(releaseRepository: ReleaseRepository,
                                    val taskPersistence: TaskPersistence,
                                    folderRepository: FolderRepository,
                                    licenseService: LicenseService,
                                    limitEnforcer: LimitEnforcer)
  extends PersistenceInterceptor[Release]
    with WorkflowTaskValidator
    with WorkflowInterceptorLogic
    with Logging {

  releaseRepository.registerPersistenceInterceptor(this)

  override def onCreate(ci: Release): Unit = {
    val tenantId = if (hasText(ci.getTenantId)) {
      ci.getTenantId
    } else {
      val tenantIdFromContext = TenantContext.getTenant()
      ci.setTenantId(tenantIdFromContext)
      tenantIdFromContext
    }
    ci.getStatus match {
      case ReleaseStatus.TEMPLATE =>
        limitEnforcer.enforceLimit(tenantId, LimitType.TEMPLATES, 1, () => tenantTemplateCount(tenantId))
      case ReleaseStatus.PLANNED | ReleaseStatus.IN_PROGRESS | ReleaseStatus.PAUSED | ReleaseStatus.FAILED | ReleaseStatus.FAILING =>
        limitEnforcer.enforceLimit(tenantId, LimitType.RELEASES, 1, () => tenantReleaseCount(tenantId))
      case _ => ()
    }
    onCreateOrUpdate(ci)
  }

  override def onUpdate(ci: Release): Unit = {
    onCreateOrUpdate(ci)
  }

  private def tenantTemplateCount(tenantId: String) = releaseRepository.tenantTemplateCountByKind(tenantId, ReleaseKind.RELEASE)

  private def tenantReleaseCount(tenantId: String) = releaseRepository.tenantReleaseCountByKind(tenantId, ReleaseKind.RELEASE)

  private def onCreateOrUpdate(release: Release): Unit = {
    validateTotalNoOfWorkflowTemplates(release)
    updateWorkflowProperties(release)
    validateDefaultTargetFolder(release)
    validateWorkflowProperties(release)
  }

  private def validateDefaultTargetFolder(release: Release): Unit = {
    val doesNotAllowTargetFolderOverride = !release.getAllowTargetFolderOverride
    val defaultTargetFolderId = release.getDefaultTargetFolderId
    val defaultTargetFolderIdIsEmpty = !StringUtils.hasText(defaultTargetFolderId)
    if (release.isTemplate && doesNotAllowTargetFolderOverride && defaultTargetFolderIdIsEmpty) {
      val msg = s"Override of the target folder is not allowed, but default target folder is not provided for release with ID [${release.getId}]"
      throw new IncorrectArgumentException(msg)
    }
    if (!defaultTargetFolderIdIsEmpty) {
      val optionalFolder = folderRepository.findById(defaultTargetFolderId, depth = 1)
      if (optionalFolder.isEmpty) {
        throw new IncorrectArgumentException(s"Cannot find folder with id [$defaultTargetFolderId]")
      }
    }
  }

  private def validateWorkflowProperties(release: Release): Unit = {
    validateNotARootWorkflow(release)
    validateNoDashboardForWorkflow(release)
    validateNoRiskProfileForWorkflow(release)
    validateNoLockTaskPresentForWorkflow(release)
    validateNoUnsupportedTaskPresentForWorkflow(release)
    validateWorkflowTaskCount(release)
    validateNoMultiLevelNestedTasksForWorkflow(release)
  }

  private def validateNotARootWorkflow(release: Release): Unit = {
    val parentFolder = Ids.getParentId(release.getId)
    if (release.isWorkflow && parentFolder == Ids.ROOT_FOLDER_ID) {
      throw new IncorrectArgumentException("Workflows can be created only inside a folder")
    }
  }

  private def validateNoDashboardForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      val hasDashboards = release.getExtensions.asScala.exists(_.isInstanceOf[Dashboard])
      if (hasDashboards) {
        throw new IncorrectArgumentException("Dashboards are not supported for workflows")
      }
    }
  }

  private def validateNoLockTaskPresentForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      validateNoLockTaskPresent(release.getAllTasks.asScala.toSeq)
    }
  }

  private def validateNoRiskProfileForWorkflow(release: Release): Unit = {
    if (release.isWorkflow && null != release.getRiskProfile) {
      throw new IncorrectArgumentException("Risk profile is not supported for workflows")
    }
  }

  private def validateNoUnsupportedTaskPresentForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      validateNoUnsupportedTaskPresent(release.getAllTasks.asScala.toSeq)
    }
  }

  private def validateNoMultiLevelNestedTasksForWorkflow(release: Release): Unit = {
    if (release.isWorkflow) {
      validateNoMultiLevelNestedTasks(release.getAllTasks.asScala.toSeq)
    }
  }

  private def validateTotalNoOfWorkflowTemplates(release: Release): Unit = {
    if (release.isTemplate && release.isWorkflow) {
      val licenseEdition = getLicenseEdition()
      if (!UNRESTRICTED_LICENSE_EDITIONS.contains(licenseEdition)) {
        val totalWorkflowTemplates = releaseRepository.countTemplatesByKind(ReleaseKind.WORKFLOW)
        if (totalWorkflowTemplates >= MAX_NO_WORKFLOW_TEMPLATES_ALLOWED) {
          throw new LicenseLimitReachedException("License limit reached. " +
            "You have exceeded the limit on the number of workflow templates supported with your license.")
        }
      }
    }
  }

  private def getLicenseEdition(): String = {
    val license = licenseService.getLicense()
    if (license.isDummyLicense) {
      Edition4.Unregistered.name()
    } else {
      // Edition is a required value for all license types
      licenseService.getLicense.getStringValue(LicenseProperty.EDITION)
    }
  }
}

object ReleasePersistenceInterceptor {
  // No of workflow templates allowed without pro or premium edition license
  private val MAX_NO_WORKFLOW_TEMPLATES_ALLOWED: Int = 100

  private val UNRESTRICTED_LICENSE_EDITIONS: Seq[String] = Seq(Edition4.Pro.name(), Edition4.Premium.name())
}
