package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.Tenant
import com.xebialabs.xlrelease.domain.status.TenantStatus
import com.xebialabs.xlrelease.repository.TenantRepository
import com.xebialabs.xlrelease.repository.sql.jpa.JpaTenantRepository
import com.xebialabs.xlrelease.repository.sql.jpa.entities.XlTenant
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.{FOLDERS, USER_PROFILE}
import com.xebialabs.xlrelease.repository.sql.persistence.{PersistenceSupport, Utils}
import grizzled.slf4j.Logging
import org.springframework.jdbc.core.JdbcTemplate

import java.util.UUID
import jakarta.persistence.OptimisticLockException
import scala.jdk.CollectionConverters._
import scala.jdk.OptionConverters._

@IsTransactional
class SqlTenantRepository(jpaTenantRepository: JpaTenantRepository,
                          implicit val dialect: Dialect,
                          implicit val jdbcTemplate: JdbcTemplate)
  extends TenantRepository with PersistenceSupport with Logging {

  private val BATCH_SIZE = 512

  @IsReadOnly
  override def findById(tenantId: String): Option[Tenant] = {
    jpaTenantRepository.findById(tenantId)
      .map(fromRow)
      .toScala
  }

  @IsReadOnly
  override def findByName(name: String): Option[Tenant] = {
    jpaTenantRepository.findByNameIgnoreCase(name)
      .map(fromRow)
      .toScala
  }

  @IsReadOnly
  override def findAll(): Seq[Tenant] = {
    jpaTenantRepository.findAll().asScala.map(fromRow).toSeq
  }

  override def create(tenantName: String, tenantStatus: TenantStatus): Tenant = {
    val newTenantRow = new XlTenant()
    newTenantRow.setTenantId(UUID.randomUUID().toString)
    newTenantRow.setName(tenantName)
    newTenantRow.setStatus(tenantStatus.value())

    jpaTenantRepository.save(newTenantRow)
    fromRow(newTenantRow)
  }

  override def update(tenant: Tenant): Unit = {
    val maybeTenantRow = jpaTenantRepository.findById(tenant.tenantId)
    if (maybeTenantRow.isPresent) {
      val existingTenant = maybeTenantRow.get()
      existingTenant.setName(tenant.tenantName)
      existingTenant.setStatus(tenant.tenantStatus.value())
      try {
        jpaTenantRepository.save(existingTenant)
      } catch {
        case e: OptimisticLockException =>
          throw new RuntimeException(s"Failed to update tenant with ID ${tenant.tenantId}", e)
      }

    } else {
      throw new IllegalArgumentException(s"Tenant with ID '${tenant.tenantId}' does not exist")
    }
  }

  override def delete(tenantId: String): Unit = {
    jpaTenantRepository.deleteById(tenantId)
  }

  override def linkFoldersToTenant(tenantId: String, folderIds: List[String]): Unit = {
    if (folderIds.nonEmpty) {
      val updateFolderSql =
        s"""
           |UPDATE ${FOLDERS.TABLE}
           |  SET ${FOLDERS.TENANT_ID} = :${FOLDERS.TENANT_ID}
           |WHERE ${FOLDERS.FOLDER_ID} = :${FOLDERS.FOLDER_ID}""".stripMargin

      folderIds.grouped(BATCH_SIZE).foreach { batch =>
        val parameters = batch.map { folderId =>
          Utils.params(FOLDERS.TENANT_ID -> tenantId, FOLDERS.FOLDER_ID -> folderId)
        }.toSet
        sqlBatch(updateFolderSql, parameters)
      }
    }
  }

  override def linkUserProfilesToTenant(tenantId: String, userProfileIds: List[String]): Unit = {
    if (userProfileIds.nonEmpty) {
      val updateUserProfileSql =
        s"""
           |UPDATE ${USER_PROFILE.TABLE}
           |  SET ${USER_PROFILE.TENANT_ID} = :${USER_PROFILE.TENANT_ID}
           |WHERE ${USER_PROFILE.USERNAME} = :${USER_PROFILE.USERNAME}""".stripMargin

      userProfileIds.grouped(BATCH_SIZE).foreach { batch =>
        val parameters = batch.map { userProfileId =>
          Utils.params(USER_PROFILE.TENANT_ID -> tenantId, USER_PROFILE.USERNAME -> userProfileId.toLowerCase)
        }.toSet
        sqlBatch(updateUserProfileSql, parameters)
      }
    }
  }

  private def fromRow(row: XlTenant): Tenant = {
    Tenant(row.getTenantId, row.getName, TenantStatus.fromString(row.getStatus))
  }
}
