package com.xebialabs.xlrelease.repository.sql.persistence

import com.xebialabs.xlrelease.db.sql.LimitOffset
import com.xebialabs.xlrelease.db.sql.SqlBuilder.Dialect
import com.xebialabs.xlrelease.db.sql.transaction.{IsReadOnly, IsTransactional}
import com.xebialabs.xlrelease.domain.UserProfile
import com.xebialabs.xlrelease.repository.sql.persistence.Schema.USER_PROFILE
import com.xebialabs.xlrelease.repository.sql.persistence.Utils.{params, _}
import com.xebialabs.xlrelease.serialization.json.utils.CiSerializerHelper.serialize
import org.springframework.beans.factory.annotation.{Autowired, Qualifier}
import org.springframework.jdbc.core.JdbcTemplate
import org.springframework.stereotype.Repository

import java.sql.ResultSet
import java.util.Date
import scala.collection.mutable.ArrayBuffer
import scala.util.Try

@Repository
@IsTransactional
class UserProfilePersistence @Autowired()(
                                           @Qualifier("xlrRepositoryJdbcTemplate") implicit val jdbcTemplate: JdbcTemplate,
                                           @Qualifier("xlrRepositorySqlDialect") implicit val dialect: Dialect
                                         )
  extends PersistenceSupport
    with LimitOffset
    with UserProfileMapper {

  private val STMT_INSERT_USER_PROFILE =
    s"""INSERT INTO ${USER_PROFILE.TABLE}
       | ( ${USER_PROFILE.USERNAME}
       | , ${USER_PROFILE.EMAIL}
       | , ${USER_PROFILE.ENABLED}
       | , ${USER_PROFILE.FULL_NAME}
       | , ${USER_PROFILE.CONTENT}
       | )
       | VALUES
       | ( :username
       | , :email
       | , :enabled
       | , :fullName
       | , :content
       | )
     """.stripMargin

  def insert(userProfile: UserProfile): Unit = {
    val content = serialize(userProfile)
    sqlExecWithContent(STMT_INSERT_USER_PROFILE, params(
      "username" -> userProfile.getCanonicalId,
      "email" -> userProfile.getEmail,
      "enabled" -> userProfile.isLoginAllowed.asInteger,
      "fullName" -> userProfile.getFullName
    ), "content" -> content,
      _ => ())
  }

  private val STMT_USER_PROFILE_EXISTS: String = s"SELECT COUNT(*) FROM ${USER_PROFILE.TABLE} WHERE ${USER_PROFILE.USERNAME} = :username"

  def exists(username: String): Boolean = {
    sqlQuery(STMT_USER_PROFILE_EXISTS, params("username" -> username), _.getInt(1) > 0).head
  }

  private val STMT_UPDATE_USER_PROFILE =
    s"""UPDATE ${USER_PROFILE.TABLE}
       | SET
       | ${USER_PROFILE.FULL_NAME} = :fullName,
       | ${USER_PROFILE.EMAIL} = :email,
       | ${USER_PROFILE.ENABLED} = :enabled,
       | ${USER_PROFILE.CONTENT} = :content
       | WHERE
       |   ${USER_PROFILE.USERNAME} = :username
     """.stripMargin

  def updateUserProfile(userProfile: UserProfile): UserProfile = {
    val content = serialize(userProfile)
    sqlExecWithContent(STMT_UPDATE_USER_PROFILE, params(
      "username" -> userProfile.getCanonicalId,
      "fullName" -> userProfile.getFullName,
      "email" -> userProfile.getEmail,
      "enabled" -> userProfile.isLoginAllowed.asInteger
    ), "content" -> content,
      checkCiUpdated(userProfile.getId)
    )
    userProfile
  }

  private val STMT_UPDATE_LAST_ACTIVE: String =
    s"""|UPDATE ${USER_PROFILE.TABLE}
        | SET ${USER_PROFILE.LAST_ACTIVE} = CASE
        |     WHEN ${USER_PROFILE.LAST_ACTIVE} < :lastActive OR ${USER_PROFILE.LAST_ACTIVE} IS NULL THEN :lastActive
        |     ELSE ${USER_PROFILE.LAST_ACTIVE}
        | END
        | WHERE ${USER_PROFILE.USERNAME} = :username """.stripMargin

  def updateLastActive(canonicalId: String, lastActive: Date): Boolean = {
    sqlUpdate(STMT_UPDATE_LAST_ACTIVE, params("username" -> canonicalId, "lastActive" -> lastActive), _ == 1)
  }

  def updateLastActiveBatch(entries: Map[String, Date]): Int = {
    sqlBatch(STMT_UPDATE_LAST_ACTIVE, entries.map {
      case (canonicalId, lastActive) =>
        params(
          "username" -> canonicalId,
          "lastActive" -> lastActive
        )
    }.toSet).sum
  }

  private val STMT_DELETE_USER_PROFILE =
    s"""| DELETE FROM ${USER_PROFILE.TABLE}
        | WHERE
        |   ${USER_PROFILE.USERNAME} = :username
      """.stripMargin

  def delete(canonicalId: String): Unit = {
    sqlExec(STMT_DELETE_USER_PROFILE, params("username" -> canonicalId), ps => Try(ps.execute()))
  }

  private val STMT_FIND_BY_ID =
    s"""SELECT
       | ${USER_PROFILE.LAST_ACTIVE},
       | ${USER_PROFILE.TENANT_ID},
       | ${USER_PROFILE.CONTENT}
       | FROM ${USER_PROFILE.TABLE}
       | WHERE ${USER_PROFILE.USERNAME} = :username
     """.stripMargin

  @IsReadOnly
  def findById(canonicalId: String): Option[UserProfile] = findOne {
    sqlQuery(STMT_FIND_BY_ID, params("username" -> canonicalId), userProfileRowMapper)
  }

  @IsReadOnly
  def search(email: String, fullName: String, loginAllowed: java.lang.Boolean,
             lastActiveAfter: Date, lastActiveBefore: Date, limit: Option[Long], offset: Option[Long]): List[UserProfile] = {
    val conditions: ArrayBuffer[String] = ArrayBuffer.empty[String]
    if (loginAllowed != null) {
      conditions += s"${USER_PROFILE.ENABLED} = :enabled"
    }
    if (email != null) {
      conditions += s"${USER_PROFILE.EMAIL} = :email"
    }
    if (fullName != null) {
      conditions += s"${USER_PROFILE.FULL_NAME} = :fullName"
    }
    Option(lastActiveAfter).foreach { _ =>
      conditions += s"${USER_PROFILE.LAST_ACTIVE} >= :lastActiveAfter"
    }
    Option(lastActiveBefore).foreach { _ =>
      conditions += s"(${USER_PROFILE.LAST_ACTIVE} <= :lastActiveBefore OR ${USER_PROFILE.LAST_ACTIVE} IS NULL)"
    }

    var sqlStatement = conditions.size match {
      case 0 => STMT_SELECT_ALL_USER_PROFILES_CONTENT
      case _ => s"$STMT_SELECT_ALL_USER_PROFILES_CONTENT WHERE ${conditions.mkString(" AND ")}"
    }

    sqlStatement = s"$sqlStatement ORDER BY ${USER_PROFILE.USERNAME}"
    sqlStatement = addLimitAndOffset(sqlStatement, limit, offset)

    sqlQuery(sqlStatement, params(
      "enabled" -> loginAllowed.asInteger,
      "email" -> email,
      "fullName" -> fullName,
      "lastActiveAfter" -> lastActiveAfter,
      "lastActiveBefore" -> lastActiveBefore
    ), userProfileRowMapper).toList
  }

  private val STMT_SELECT_ALL_USER_PROFILES =
    s"""SELECT ${USER_PROFILE.USERNAME}, ${USER_PROFILE.FULL_NAME}, ${USER_PROFILE.EMAIL}, ${USER_PROFILE.LAST_ACTIVE}, ${USER_PROFILE.TENANT_ID}, ${USER_PROFILE.ENABLED}
       | FROM ${USER_PROFILE.TABLE}
     """.stripMargin

  private val STMT_SELECT_ALL_USER_PROFILES_CONTENT =
    s"""SELECT ${USER_PROFILE.USERNAME}, ${USER_PROFILE.LAST_ACTIVE}, ${USER_PROFILE.TENANT_ID}, ${USER_PROFILE.CONTENT}
       | FROM ${USER_PROFILE.TABLE}
     """.stripMargin

  @IsReadOnly
  def findAll(fullProfile: Boolean): List[UserProfile] = {
    if (fullProfile) {
      sqlQuery(STMT_SELECT_ALL_USER_PROFILES_CONTENT, params(), userProfileRowMapper).toList
    } else {
      sqlQuery(STMT_SELECT_ALL_USER_PROFILES, params(), (rs: ResultSet, _: Int) => {
        val profile = new UserProfile()
        profile.setId(rs.getString(USER_PROFILE.USERNAME))
        profile.setFullName(rs.getString(USER_PROFILE.FULL_NAME))
        profile.setEmail(rs.getString(USER_PROFILE.EMAIL))
        profile.setLastActive(rs.getTimestamp(USER_PROFILE.LAST_ACTIVE))
        profile.setTenantId(rs.getString(USER_PROFILE.TENANT_ID))
        profile.setLoginAllowed(rs.getInt(USER_PROFILE.ENABLED).asBoolean)
        profile
      }).toList
    }
  }

  private val STMT_COUNT_LICENSED_USERS =
    s"""SELECT COUNT(*) FROM ${USER_PROFILE.TABLE} WHERE ${USER_PROFILE.ENABLED} = 1 AND ${USER_PROFILE.LAST_ACTIVE} IS NOT NULL"""

  @IsReadOnly
  def countUserWithLoginAllowed(): Int = {
    sqlQuery(STMT_COUNT_LICENSED_USERS, params(), _.getInt(1)).head
  }

}

