package com.xebialabs.xlrelease.repository.sql.proxy

import com.xebialabs.deployit.booter.local.utils.ReflectionUtils
import com.xebialabs.deployit.booter.local.utils.ReflectionUtils.searchField
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.deployit.plugin.api.udm.base.BaseConfigurationItem
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.status.{FlagStatus, ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.repository.DependencyTargetLoader
import com.xebialabs.xlrelease.repository.sql.persistence.data.{BasicReleaseRow, TaskRow}
import com.xebialabs.xlrelease.repository.sql.persistence.{ReleasePersistence, TaskPersistence}
import org.springframework.core.Ordered
import org.springframework.stereotype.Component

import java.util
import scala.collection.mutable
import scala.util.Try

// the idea of ProxyInstanceInitializer is to pre-populate proxy with properties
// e.g. properties that are needed by dependency node
sealed trait ProxyInstanceInitializer extends Ordered {

  final val DEFAULT_ORDER = 0

  def initialize(proxy: ConfigurationItem): List[String]

  def accept(proxy: ConfigurationItem): Boolean

  override def getOrder: Int = DEFAULT_ORDER
}

@Component
class DefaultProxyInstanceInitializer extends ProxyInstanceInitializer {

  override def initialize(proxy: ConfigurationItem): List[String] = {
    val template: ConfigurationItem = proxy.getType.getDescriptor.newInstance(null)
    val ci = proxy.asInstanceOf[BaseConfigurationItem]
    copySyntheticProperties(template, ci)
    List()
  }

  private def copySyntheticProperties(template: ConfigurationItem, item: ConfigurationItem): Unit = {
    val syntheticPropertiesField = searchField(template.getClass, ConfigurationItem.SYNTHETIC_PROPERTIES_FIELD)
    val syntheticProperties = ReflectionUtils.getField(template, syntheticPropertiesField).asInstanceOf[util.Map[String, AnyRef]]
    ReflectionUtils.setField(item, ConfigurationItem.SYNTHETIC_PROPERTIES_FIELD, syntheticProperties)
  }

  override def accept(proxy: ConfigurationItem): Boolean = true

  override def getOrder: Int = Ordered.HIGHEST_PRECEDENCE
}

@Component
class ReleaseProxyInstanceInitializer(releasePersistence: ReleasePersistence) extends ProxyInstanceInitializer {

  override def initialize(proxy: ConfigurationItem): List[String] = {
    val planItem = proxy.asInstanceOf[Release]
    val optionalReleaseRow: Option[BasicReleaseRow] = releasePersistence.getBasicReleaseRow(planItem.getId)
    optionalReleaseRow.map { row =>
      planItem.setCiUid(row.ciUid)
      planItem.setStatus(ReleaseStatus.valueOf(row.status.toUpperCase))
      val flagStatus = row.realFlagStatus match {
        case 0 => FlagStatus.OK
        case 1 => FlagStatus.ATTENTION_NEEDED
        case 2 => FlagStatus.AT_RISK
        case _ => FlagStatus.OK
      }
      planItem.setFlagStatus(flagStatus)
      planItem.setTitle(row.title)

      planItem.setStartDate(row.startDate)
      planItem.setEndDate(row.endDate)
      val releaseKind = ReleaseKind.fromString(row.kind)
      planItem.setKind(releaseKind)
      planItem.setTenantId(row.tenantId)

      List(
        "getCiUid",
        "getRelease",
        "getReleaseUid",
        "getStatus", "isDone", "isAborted", "isActive", "isPlanned", "isFailing", "isFailed", "isDefunct", "isUpdatable", "hasBeenStarted",
        "getFlagStatus",
        "getTitle", "hasTitle", "isTitleContaining", "getDisplayPath",
        "getStartDate", "hasStartDate",
        "getEndDate", "hasEndDate",
        "getKind",
        "getTenantId"
      )
    }.getOrElse(List())
  }

  override def accept(proxy: ConfigurationItem): Boolean = proxy.isInstanceOf[Release]
}


@Component
class TaskProxyInstanceInitializer(taskPersistence: TaskPersistence) extends ProxyInstanceInitializer {

  override def initialize(proxy: ConfigurationItem): List[String] = {
    val planItem = proxy.asInstanceOf[Task]
    val optionalRow: Option[TaskRow] = taskPersistence.findById(planItem.getId)
    val initializedMethods = mutable.Buffer(
      "getReleaseUid",
      "getStatus", "isDone", "isAborted", "isActive", "isPlanned", "isFailing", "isFailed", "isDefunct", "isUpdatable", "hasBeenStarted",
      "isAutomated",
      "isLocked",
      "getTeam", "hasTeam",
      "getOwner", "hasOwner",
      "getTitle", "hasTitle", "isTitleContaining", "getDisplayPath",
      "getStartDate", "hasStartDate",
      "getEndDate", "hasEndDate"
    )
    optionalRow.map { row =>
      planItem.setReleaseUid(row.releaseUid)
      planItem.setStatus(TaskStatus.valueOf(row.status.toUpperCase))
      planItem match {
        case task: CustomScriptTask =>
          task.setStatusLine(row.statusLine)
          initializedMethods += "getStatusLine"
        case task: ContainerTask =>
          task.setStatusLine(row.statusLine)
          initializedMethods += "getStatusLine"
        case _ => ()
      }
      planItem.setLocked(row.locked)
      planItem.setTeam(row.team)
      planItem.setOwner(row.owner)
      planItem.setTitle(row.title)
      planItem.setStartDate(row.startDate)
      planItem.setEndDate(row.endDate)

      initializedMethods.toList
    }.getOrElse(List())
  }

  override def accept(proxy: ConfigurationItem): Boolean = proxy.isInstanceOf[Task]
}

@Component
class PhaseProxyNodeDataInitializer(dependencyTargetLoader: DependencyTargetLoader) extends ProxyInstanceInitializer {
  override def initialize(proxy: ConfigurationItem): List[String] = {
    val planItem: PlanItem = proxy.asInstanceOf[PlanItem]
    val initializedMethods = List(
      "getDependencyTarget",
      "getReleaseUid",
      "getTitle", "hasTitle", "isTitleContaining", "getDisplayPath",
      "getStatus", "isDone", "isAborted", "isActive", "isPlanned", "isFailing", "isFailed", "isDefunct", "isUpdatable", "hasBeenStarted",
      "getStartDate", "hasStartDate",
      "getEndDate", "hasEndDate",
      "getScheduledStartDate", "hasScheduledStartDate",
      "getDueDate", "hasDueDate",
      "getEndOrDueDate",
      "hasValidStartDates",
      "getPlannedDuration", "hasPlannedDuration",
      "getOrCalculateDueDate", "isOverdue",
      "getComputedPlannedDuration",
      "getActualDuration"
    )
    Try {
      val dependencyTarget = dependencyTargetLoader.getDependencyNodeData(planItem.getId)
      planItem.setReleaseUid(dependencyTarget.targetReleaseUid)
      planItem match {
        case t: Task =>
          t.setTitle(dependencyTarget.taskTitle)
          t.setStatus(dependencyTarget.taskStatus)
        case p: Phase =>
          p.setTitle(dependencyTarget.phaseTitle)
          p.setStatus(dependencyTarget.phaseStatus)
        case r: Release =>
          r.setTitle(dependencyTarget.releaseTitle)
          ()
      }
      planItem.setTitle(dependencyTarget.phaseTitle)

      planItem.setStartDate(dependencyTarget.startDate)
      planItem.setEndDate(dependencyTarget.endDate)
      planItem.setScheduledStartDate(dependencyTarget.scheduledStartDate)
      planItem.setDueDate(dependencyTarget.dueDate)
      initializedMethods
    }.getOrElse(List())
  }

  override def accept(proxy: ConfigurationItem): Boolean = proxy.isInstanceOf[Phase]

  override def getOrder: Int = DEFAULT_ORDER + 1

}
