package com.xebialabs.xlrelease.repository.sql

import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.limits.LimitEnforcer
import com.xebialabs.xlrelease.limits.LimitEnforcer.LimitType
import com.xebialabs.xlrelease.repository.{FolderRepository, FolderVariableRepository, PersistenceInterceptor}
import com.xebialabs.xlrelease.utils.TenantContext
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component
import org.springframework.util.StringUtils.hasText

/**
 * Folder Variable Limit Enforcement Interceptor:
 *
 * Variable Count Limits:
 * - Applied to: Folder variables
 * - Scope: Per-tenant, per-folder
 */
@Component
@IsTransactional
class FolderVariableLimitPersistenceInterceptor(val limitEnforcer: LimitEnforcer,
                                                folderRepository: FolderRepository,
                                                folderVariableRepository: FolderVariableRepository)
  extends PersistenceInterceptor[Variable]
    with VariableSizeLimitValidator
    with Logging {

  folderVariableRepository.registerPersistenceInterceptor(this)

  override def onCreate(variable: Variable): Unit = {
    if (isFolderVariable(variable)) {
      logger.debug(s"[FolderVariableLimitPersistenceInterceptor] Enforcing folder variable limits for folder: ${variable.getFolderId}, key: ${variable.getKey}")
      val tenantId = getTenantIdFromFolder(variable.getFolderId)
      // Enforce folder variable count limit
      limitEnforcer.enforceLimit(tenantId, LimitType.FOLDER_VARIABLES, 1, () => getCurrentFolderVariableCount(variable.getFolderId))

      // Enforce variable size validation
      enforceVariableSizeLimits(variable, tenantId)
    }
  }

  override def onUpdate(variable: Variable): Unit = {
    if (isFolderVariable(variable)) {
      val tenantId = getTenantIdFromFolder(variable.getFolderId)
      // Enforce variable size validation for updates
      enforceVariableSizeLimits(variable, tenantId)
    }
  }

  private def isFolderVariable(variable: Variable): Boolean = {
    variable.getFolderId != null && hasText(variable.getFolderId)
  }

  private def getCurrentFolderVariableCount(folderId: String): Int = {
    try {
      folderVariableRepository.getAllFromParent(folderId).size()
    } catch {
      case ex: Exception =>
        logger.warn(s"Unable to get current folder variable count for folder $folderId", ex)
        0
    }
  }

  private def getTenantIdFromFolder(folderId: String): String = {
    val tenantIdFromContext = TenantContext.getTenant()
    if (hasText(tenantIdFromContext)) {
      tenantIdFromContext
    } else {
      folderRepository.getFolderTenant(folderId)
    }
  }
}
