package com.xebialabs.xlrelease.utils;

import java.util.List;
import java.util.concurrent.TimeoutException;
import java.util.function.Supplier;

import com.xebialabs.xlrelease.domain.events.XLReleaseEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;

import scala.Option;
import scala.PartialFunction;

import static com.xebialabs.xlrelease.utils.EventsAwaiter.VerificationMode.ANY_OF_EVENTS;
import static com.xebialabs.xlrelease.utils.EventsAwaiter.VerificationMode.EVENTS_IN_ANY_ORDER;
import static com.xebialabs.xlrelease.utils.EventsAwaiter.VerificationMode.EVENTS_IN_ORDER;
import static java.util.Arrays.asList;

public class ConditionBuilder<R> {
    private final XLReleaseEventBus eventBus;
    private final Supplier<R> runnable;
    private long timeout = DEFAULT_TIMEOUT;

    public static final long DEFAULT_TIMEOUT = 45000;

    private ConditionBuilder(XLReleaseEventBus eventBus, Supplier<R> runnable) {
        this.eventBus = eventBus;
        this.runnable = runnable;
    }

    public ConditionBuilder<R> withTimeout(long timeout) {
        this.timeout = timeout;
        return this;
    }

    protected long getTimeout() {
        return timeout;
    }

    public static <R> ConditionBuilder<R> execute(XLReleaseEventBus eventBus, Supplier<R> runnable) {
        return new ConditionBuilder<>(eventBus, runnable);
    }

    public <S> R until(String planItemId, List<S> statuses) throws InterruptedException, TimeoutException {
        return until(planItemId, statuses, getTimeout());
    }

    public <S> R until(String planItemId, List<S> statuses, long timeout) throws InterruptedException, TimeoutException {
        return until(() -> new PlanItemStatusAwaiter<>(eventBus, planItemId, statuses, timeout));
    }

    public <S> R until(String planItemId, S... statuses) throws InterruptedException, TimeoutException {
        return until(planItemId, getTimeout(), statuses);
    }

    public R until(long timeout, XLReleaseEvent... eventsInOccurrenceOrder) throws InterruptedException, TimeoutException {
        return until(() -> EventsAwaiter.create(eventBus, asList(eventsInOccurrenceOrder), timeout, EVENTS_IN_ORDER));
    }

    public <S> R until(String planItemId, long timeout, S... statuses) throws InterruptedException, TimeoutException {
        return until(planItemId, asList(statuses), timeout);
    }

    public R until(XLReleaseEvent... eventsInOccurrenceOrder) throws InterruptedException, TimeoutException {
        return until(() -> EventsAwaiter.create(eventBus, asList(eventsInOccurrenceOrder), getTimeout(), EVENTS_IN_ORDER));
    }

    public R untilInAnyOrder(XLReleaseEvent... events) throws InterruptedException, TimeoutException {
        return until(() -> EventsAwaiter.create(eventBus, asList(events), getTimeout(), EVENTS_IN_ANY_ORDER));
    }

    public R untilAnyOf(XLReleaseEvent... events) throws InterruptedException, TimeoutException {
        return until(() -> EventsAwaiter.create(eventBus, asList(events), getTimeout(), ANY_OF_EVENTS));
    }

    public R until(long timeout, PartialFunction<XLReleaseEvent, MatchingAwaiter.MatcherEvent>... matchers) throws TimeoutException, InterruptedException {
        return until(() -> MatchingAwaiter.apply(eventBus, timeout, matchers));
    }

    public R until(PartialFunction<XLReleaseEvent, MatchingAwaiter.MatcherEvent>... matchers) throws TimeoutException, InterruptedException {
        return until(() -> MatchingAwaiter.apply(eventBus, getTimeout(), matchers));
    }

    public CleanupAwaiter listen() {
        CleanupAwaiter awaiter = new CleanupAwaiter(eventBus, getTimeout());
        awaiter.startListening();
        return awaiter;
    }

    public <T> Option<T> pick(PartialFunction<XLReleaseEvent, Option<T>> picker) throws TimeoutException, InterruptedException {
        try (PickingAwaiter condition = new PickingAwaiter<T>(eventBus, getTimeout(), picker)) {
            condition.startListening();
            R result = runnable.get();
            return condition.get();
        }
    }

    private R until(Supplier<BaseConditionAwaiter> conditionSupplier) throws InterruptedException, TimeoutException {
        try (BaseConditionAwaiter condition = conditionSupplier.get()) {
            condition.startListening();
            R result = runnable.get();
            condition.await();
            return result;
        }
    }

}