package com.xebialabs.xlrelease.utils;

import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.xlrelease.domain.events.XLReleaseEvent;
import com.xebialabs.xlrelease.events.XLReleaseEventBus;
import com.xebialabs.xlrelease.events.SynchronizedSubscribe;

import static com.xebialabs.xlrelease.utils.EventsAwaiter.VerificationMode.EVENTS_IN_ORDER;
import static java.lang.String.format;

class EventsAwaiter extends BaseConditionAwaiter {

    private static final Logger logger = LoggerFactory.getLogger(EventsAwaiter.class);
    private static final String LOCK = "";

    enum VerificationMode {
        EVENTS_IN_ORDER, EVENTS_IN_ANY_ORDER, ANY_OF_EVENTS
    }

    private volatile List<XLReleaseEvent> expectedEvents;
    private volatile Queue<XLReleaseEvent> pendingEvents;
    private volatile List<XLReleaseEvent> actualEvents = new ArrayList<>();
    private VerificationMode mode;

    private EventsAwaiter(XLReleaseEventBus eventBus, List<XLReleaseEvent> eventsInOccurrenceOrder, long timeout) {
        this(eventBus, eventsInOccurrenceOrder, timeout, EVENTS_IN_ORDER);
    }

    private EventsAwaiter(XLReleaseEventBus eventBus, List<XLReleaseEvent> events, long timeout, VerificationMode mode) {
        super(eventBus, timeout);
        this.expectedEvents = events;
        this.pendingEvents = new ArrayDeque<>(events);
        this.mode = mode;
    }

    public static EventsAwaiter create(XLReleaseEventBus eventBus, List<XLReleaseEvent> eventsInOccurrenceOrder, long timeout) {
        synchronized (LOCK) {
            return new EventsAwaiter(eventBus, eventsInOccurrenceOrder, timeout);
        }
    }

    public static EventsAwaiter create(XLReleaseEventBus eventBus, List<XLReleaseEvent> events, long timeout, VerificationMode mode) {
        synchronized (LOCK) {
            return new EventsAwaiter(eventBus, events, timeout, mode);
        }
    }

    @SynchronizedSubscribe
    public void onEvent(XLReleaseEvent event) {
        synchronized (LOCK) {
            logger.trace("Received event: {}", event);
            actualEvents.add(event);

            switch (mode) {
                case EVENTS_IN_ORDER:
                    if (Objects.equals(pendingEvents.peek(), event)) {
                        pendingEvents.poll();
                    }
                    break;
                case EVENTS_IN_ANY_ORDER:
                    pendingEvents.remove(event);
                    break;
                case ANY_OF_EVENTS:
                    if (pendingEvents.contains(event)) {
                        pendingEvents.clear();
                    }
                    break;
            }
            if (pendingEvents.isEmpty()) {
                latch.countDown();
            }
        }
    }

    @Override
    protected String getErrorMessage() {
        return format("Did not encounter events %s out of requested ordered list %s within %d milliseconds, events encountered: %s",
            pendingEvents, expectedEvents, timeout, actualEvents);
    }
}
