package com.xebialabs.xlrelease.utils

import com.xebialabs.xlrelease.domain.events.XLReleaseEvent
import com.xebialabs.xlrelease.events.{SynchronizedSubscribe, XLReleaseEventBus}
import com.xebialabs.xlrelease.utils.MatchingAwaiter.{MATCHER, MatchError, MatchFound, MatchIgnore}
import grizzled.slf4j.Logging

import java.util
import scala.annotation.varargs
import scala.collection.mutable
import scala.collection.mutable.ArrayBuffer
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

class MatchingAwaiter(eventBus: XLReleaseEventBus, timeout: Long, matchers: Seq[MATCHER])
  extends BaseConditionAwaiter(eventBus, timeout, matchers.size)
    with Logging {
  protected var events: mutable.Buffer[(String, XLReleaseEvent)] = new ArrayBuffer[(String, XLReleaseEvent)]()
  private lazy val defaultMsg = s"Did not match event within ${this.timeout} milliseconds."
  private var errorMsg = defaultMsg
  private var errorCause: Throwable = _
  private val NL = System.lineSeparator

  override protected def getErrorMessage: String = s"$errorMsg${NL}Received events were: ${formatCapturedEvents()}"

  private def formatCapturedEvents(): String = {
    events.mkString(NL, s",$NL", NL)
  }

  override protected def getErrorCause: Throwable = errorCause

  private var currentMatchers = matchers

  @SynchronizedSubscribe
  def onEvent(event: XLReleaseEvent): Unit = {
    if (currentMatchers == null) throw new IllegalStateException("current matchers are null - check if class was registered to the event bus in a constructor")
    currentMatchers.headOption match {
      case Some(matcher) => Try(matcher.apply(event)) match {
        case Success(MatchFound()) => matched(event)
        case Success(MatchIgnore()) => ignore(event)
        case Success(MatchError(msg)) => fail(event, new IllegalStateException(msg))
        case Failure(ex) => fail(event, ex)
      }
      case None =>
        warn(s"No matcher available to process event.", event)
    }
  }

  private def warn(msg: String, event: XLReleaseEvent): Unit = {
    logger.warn(s"Received event ${event} but there are no matchers anymore, so we will ignore it.")
  }

  protected def fail(event: XLReleaseEvent, ex: Throwable): Unit = {
    events += "failed" -> event
    failed = true
    errorMsg = ex.getMessage
    errorCause = ex
    while (latch.getCount > 0) latch.countDown()
  }

  protected def matched(event: XLReleaseEvent): Unit = {
    logger.info(s"Matched event ${event}")
    events += "matched" -> event
    currentMatchers = currentMatchers.tail
    if (currentMatchers.isEmpty) {
      logger.debug(s"Received events: ${formatCapturedEvents()}")
      close()
    }
    latch.countDown()
  }

  private def ignore(event: XLReleaseEvent): Unit = {
    events += "ignored" -> event
  }

  override protected def getMatchedEvents: util.List[XLReleaseEvent] = {
    events.filter(_._1 == "matched").map(_._2).asJava
  }
}

object MatchingAwaiter {
  type MATCHER = PartialFunction[XLReleaseEvent, MatcherEvent]

  sealed trait MatcherEvent

  case class MatchFound() extends MatcherEvent

  case class MatchIgnore() extends MatcherEvent

  case class MatchError(message: String) extends Throwable(message) with MatcherEvent

  @varargs
  def apply(eventBus: XLReleaseEventBus, timeout: Long, matchers: MATCHER*) = new MatchingAwaiter(eventBus, timeout, matchers)

  @varargs
  def anyOrder(matchers: MATCHER*): MATCHER = {
    var remainingMatchers = Seq(matchers: _*)
    var anyOrderMatcher: MATCHER = {
      case e if remainingMatchers.exists(_.isDefinedAt(e)) =>
        val matches = remainingMatchers.map(_.apply(e))
        val result: MatcherEvent = matches.find(_.isInstanceOf[MatchError]).getOrElse {
          remainingMatchers = remainingMatchers.zip(matches).filterNot(_._2.isInstanceOf[MatchFound]).map(_._1)
          if (remainingMatchers.isEmpty) {
            MatchFound()
          } else {
            MatchIgnore()
          }
        }
        result
    }
    anyOrderMatcher
  }

  @varargs
  def anyOf(matchers: MATCHER*): MATCHER = {
    matchers.reduceLeft(_.orElse(_))
  }

  @varargs
  def allOfInAnyOrder(matchers: MATCHER*): Seq[MATCHER] = {
    matchers.map(_ => anyOf(matchers: _*))
  }
}
