/*
 * Decompiled with CFR 0.152.
 */
package com.xebialabs.deployit.plumbing.scheduler;

import com.google.common.annotations.VisibleForTesting;
import com.xebialabs.deployit.plumbing.scheduler.FutureTaskAdapter;
import java.util.Collections;
import java.util.List;
import java.util.Set;
import java.util.concurrent.CompletableFuture;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.CopyOnWriteArrayList;
import java.util.concurrent.ExecutionException;
import java.util.concurrent.Future;
import java.util.concurrent.FutureTask;
import java.util.concurrent.RejectedExecutionException;
import java.util.concurrent.ScheduledExecutorService;
import java.util.concurrent.ScheduledFuture;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.TimeoutException;
import java.util.concurrent.atomic.AtomicBoolean;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class Scheduler {
    private ScheduledExecutorService executorService;
    private AtomicBoolean shuttingDown = new AtomicBoolean(false);
    private int awaitTerminationSeconds;
    private int taskSchedulerGraceShutdownPeriodSeconds;
    private final Set<FutureTaskAdapter<?>> scheduledFutures = Collections.newSetFromMap(new ConcurrentHashMap());
    private final List<Future<?>> shutdownFutures = new CopyOnWriteArrayList();
    private static final Logger LOG = LoggerFactory.getLogger(Scheduler.class);

    public Scheduler(ScheduledExecutorService executorService, int awaitTerminationSeconds, int taskSchedulerGraceShutdownPeriodSeconds) {
        this.awaitTerminationSeconds = awaitTerminationSeconds;
        this.taskSchedulerGraceShutdownPeriodSeconds = taskSchedulerGraceShutdownPeriodSeconds;
        this.executorService = executorService;
    }

    public synchronized ScheduledFuture<?> scheduleAtFixedRate(Runnable command, long initialDelay, long delay, TimeUnit unit) {
        return this.executorService.scheduleAtFixedRate(Scheduler.withLoggedExceptions(command), initialDelay, delay, unit);
    }

    public synchronized <U> ScheduledFuture<U> scheduleOnce(FutureTaskAdapter<U> command, long delay, TimeUnit unit) {
        this.checkShuttingDown();
        return command.bind(this.executorService.schedule(this.configureFutureTaskAdapter(command), delay, unit));
    }

    private void checkShuttingDown() {
        if (this.shuttingDown.get()) {
            throw new RejectedExecutionException("Scheduler is shutting down");
        }
    }

    private <U> FutureTaskAdapter<U> configureFutureTaskAdapter(FutureTaskAdapter<U> command) {
        command.setScheduler(this);
        this.scheduledFutures.add(command);
        return command;
    }

    public void onTaskCompleted(FutureTaskAdapter<?> command) {
        this.scheduledFutures.remove(command);
    }

    public void handleFailureFuture(Future<?> future, FutureTaskAdapter<?> taskAdapter) {
        if (future != null) {
            if (this.shuttingDown.get()) {
                this.shutdownFutures.add(future);
            } else {
                try {
                    future.get();
                }
                catch (ExecutionException e) {
                    LOG.warn("Got ExecutionException when handling failure future for task {}", taskAdapter, (Object)e);
                }
                catch (InterruptedException e) {
                    LOG.warn("Got InterruptedException when handling failure future for task {}", taskAdapter, (Object)e);
                    Thread.currentThread().interrupt();
                }
            }
        }
    }

    public synchronized Future<?> submit(Runnable command) {
        return this.executorService.submit(Scheduler.withLoggedExceptions(command));
    }

    public synchronized <U> CompletableFuture<U> supplyAsync(FutureTaskAdapter<U> command) {
        this.checkShuttingDown();
        command.bind(this.executorService.schedule(this.configureFutureTaskAdapter(command), 0L, TimeUnit.NANOSECONDS));
        return command.getCompletableFuture();
    }

    public CompletableFuture<Object> supplyAsync(FutureTaskAdapter<Object> command, long timeout, TimeUnit unit) {
        return CompletableFuture.anyOf(this.supplyAsync(command), this.timeoutAfter(timeout, unit));
    }

    private <T> CompletableFuture<T> timeoutAfter(long timeout, TimeUnit unit) {
        CompletableFuture result = new CompletableFuture();
        this.executorService.schedule(() -> result.completeExceptionally(new TimeoutException()), timeout, unit);
        return result;
    }

    public void shutdown() {
        LOG.info("Initiating scheduler shutdown.");
        this.shuttingDown.set(true);
        this.gracefulShutdown();
        LOG.info("Graceful part of scheduler shutdown complete, proceeding with shutdown.");
        this.shutdownAndAwaitTermination(this.executorService);
        LOG.info("ShutdownNow part of scheduler shutdown complete, going to wait for any Futures");
        if (!this.shutdownFutures.isEmpty()) {
            LOG.info("Going to wait for {} shutdown futures", (Object)this.shutdownFutures.size());
            try {
                for (Future<?> future : this.shutdownFutures) {
                    future.get(this.taskSchedulerGraceShutdownPeriodSeconds, TimeUnit.SECONDS);
                }
            }
            catch (InterruptedException e) {
                LOG.warn("Got InterruptedException when waiting for shutdown future, giving up", (Throwable)e);
                Thread.currentThread().interrupt();
                throw new IllegalStateException(e);
            }
            catch (Exception e) {
                LOG.warn("Got exception when waiting for shutdown future, giving up", (Throwable)e);
            }
        }
        LOG.info("Scheduler shutdown finished.");
    }

    private void gracefulShutdown() {
        try {
            this.scheduledFutures.forEach(adapter -> {
                LOG.info("Cancelling {} as part of graceful scheduler shutdown", adapter);
                adapter.cancelExternally();
            });
            this.executorService.shutdown();
            this.executorService.awaitTermination(this.taskSchedulerGraceShutdownPeriodSeconds, TimeUnit.SECONDS);
        }
        catch (InterruptedException e) {
            Thread.currentThread().interrupt();
            throw new IllegalStateException("Graceful shutdown interrupted", e);
        }
    }

    private void shutdownAndAwaitTermination(ScheduledExecutorService executorService) {
        try {
            LOG.debug("Initiating shutdown now.");
            List<Runnable> runnables = executorService.shutdownNow();
            for (Runnable runnable : runnables) {
                if (runnable instanceof FutureTask) {
                    FutureTask futureTask = (FutureTask)runnable;
                    if (futureTask.isDone()) continue;
                    LOG.info("About to cancel task {}", (Object)runnable);
                    futureTask.cancel(true);
                    continue;
                }
                LOG.error("Unable to cancel runnable {}", (Object)runnable);
            }
            boolean terminated = executorService.awaitTermination(this.awaitTerminationSeconds, TimeUnit.SECONDS);
            if (!terminated) {
                LOG.warn("Could not finish termination of executorService. There are still active threads.");
            }
        }
        catch (InterruptedException e) {
            LOG.warn("Could not finish termination of executorService.", (Throwable)e);
            Thread.currentThread().interrupt();
        }
    }

    private static Runnable withLoggedExceptions(Runnable command) {
        return () -> {
            try {
                command.run();
            }
            catch (Exception e) {
                LOG.error("Error while running a scheduled task.", (Throwable)e);
                throw e;
            }
        };
    }

    @VisibleForTesting
    void replaceExecutorService(ScheduledExecutorService executorService) {
        this.executorService = executorService;
        this.shuttingDown.set(false);
        this.scheduledFutures.clear();
        this.shutdownFutures.clear();
    }
}

