package com.xebialabs.xlrelease.cache.caffeine

import com.github.benmanes.caffeine.cache.stats.CacheStats
import com.github.benmanes.caffeine.cache.{Cache, Policy}
import grizzled.slf4j.Logging

import java.util.concurrent.ConcurrentMap
import java.util.function
import java.{lang, util}
import scala.jdk.CollectionConverters._

class CaffeineCacheDelegate[K, V](name: String, cache: Cache[K, V]) extends Cache[K, V] with Logging {
  override def getIfPresent(key: K): V = cache.getIfPresent(key)

  override def get(key: K, mappingFunction: function.Function[_ >: K, _ <: V]): V = cache.get(key, mappingFunction)

  override def getAllPresent(keys: lang.Iterable[_ <: K]): util.Map[K, V] = cache.getAllPresent(keys)

  override def put(key: K, value: V): Unit = cache.put(key, value)

  override def putAll(map: util.Map[_ <: K, _ <: V]): Unit = cache.putAll(map)

  override def invalidate(key: K): Unit = this.invalidateAll(util.Arrays.asList(key))

  override def invalidateAll(keys: lang.Iterable[_ <: K]): Unit = {
    logger.trace(s"Invalidating $keys from cache [$name]")
    val existingKeys = cache.asMap().keySet().asScala.filter(_ != null).map(_.toString)
    val existingKeysToRemove = keys.asScala.filter(_ != null).flatMap(keyToRemove => {
      existingKeys.filter(existingKey => existingKey == keyToRemove || (keyToRemove.toString.startsWith("regex:") && existingKey.matches(keyToRemove.toString.replace("regex:", ""))))
    }).asJava
    cache.invalidateAll(existingKeysToRemove.asInstanceOf[lang.Iterable[_ <: K]])
    logger.trace(s"Invalidated $existingKeysToRemove from cache [$name]")
  }

  override def invalidateAll(): Unit = {
    logger.trace(s"Invalidating all from cache [$name]")
    cache.invalidateAll()
  }

  override def estimatedSize(): Long = cache.estimatedSize()

  override def stats(): CacheStats = cache.stats()

  override def asMap(): ConcurrentMap[K, V] = cache.asMap()

  override def cleanUp(): Unit = cache.cleanUp()

  override def policy(): Policy[K, V] = cache.policy()

  override def getAll(keys: lang.Iterable[_ <: K], mappingFunction: function.Function[_ >: util.Set[_ <: K], _ <: util.Map[_ <: K, _ <: V]]): util.Map[K, V] = cache.getAll(keys, mappingFunction)

}
