package com.xebialabs.xlrelease.scheduler.workers

import com.google.common.base.Preconditions.checkArgument
import com.xebialabs.deployit.booter.local.utils.Strings.isNotBlank
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.security.permission.Permission
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.builder.ReleaseBuilder.newRelease
import com.xebialabs.xlrelease.domain.CreateReleaseTask.CREATED_RELEASE_ID
import com.xebialabs.xlrelease.domain.FailureReasons.CREATE_RELEASE_TASK_FAILED
import com.xebialabs.xlrelease.domain.events.{CreatedFromCreateReleaseTask, ReleaseCreatedEvent, ReleaseStartedFromCreateReleaseTaskEvent}
import com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_RELEASE_TASK
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.domain.{CreateReleaseTask, Release}
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.IdType.DOMAIN
import com.xebialabs.xlrelease.repository.Ids
import com.xebialabs.xlrelease.risk.domain.RiskProfile.RISK_PROFILE
import com.xebialabs.xlrelease.scheduler.CreateReleaseTaskJob
import com.xebialabs.xlrelease.scheduler.workers.Worker.ExecuteJob
import com.xebialabs.xlrelease.script.EncryptionHelper
import com.xebialabs.xlrelease.security.XLReleasePermissions.START_RELEASE
import com.xebialabs.xlrelease.security.authentication.AuthenticationService
import com.xebialabs.xlrelease.security.{PermissionChecker, UsernamePassword}
import com.xebialabs.xlrelease.service.{CiIdService, CommentService, ReleaseService}
import com.xebialabs.xlrelease.user.User.{AUTHENTICATED_USER, SYSTEM}
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.variable.VariableHelper.{containsOnlyVariable, fillVariableValues, withoutVariableSyntax}
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds
import grizzled.slf4j.Logging
import org.springframework.stereotype.Component

import java.lang.String.format
import java.util.{Collections, Optional, ArrayList => JArrayList, List => JList, Map => JMap}

@Component
class CreateReleaseTaskWorker(val eventBus: XLReleaseEventBus,
                              val releaseActorService: ReleaseActorService,
                              val releaseService: ReleaseService,
                              val ciIdService: CiIdService,
                              val authenticationService: AuthenticationService,
                              val permissionChecker: PermissionChecker,
                              val commentService: CommentService
                             ) extends Worker with CreateReleaseTaskLogic with TaskWorkerFailureLogic {

  override def execute: ExecuteJob = {
    case CreateReleaseTaskJob(taskRef) => doRun(taskRef) {
      val task = taskRef.get()
      startTask(task)
    }
  }

}

trait CreateReleaseTaskLogic extends Logging {

  def eventBus: XLReleaseEventBus

  def releaseActorService: ReleaseActorService

  def releaseService: ReleaseService

  def ciIdService: CiIdService

  def authenticationService: AuthenticationService

  def permissionChecker: PermissionChecker

  def commentService: CommentService

  def startTask(task: CreateReleaseTask): Void = {
    try {
      validateRequiredProperties(task)
      val scriptUserPass = checkPermissions(task)
      validateCreateReleaseRecursion(task)
      val subRelease = createRelease(task, scriptUserPass)
      saveCreatedReleaseIdIntoVariable(task, subRelease)

      val shouldCompleteTask = if (task.getStartRelease) {
        checkPermissions(task, subRelease.getId, START_RELEASE)
        startRelease(task, subRelease)
      } else {
        true
      }

      if (shouldCompleteTask) {
        saveCreatedReleaseIdIntoTask(task, subRelease)
        completeTask(task)
      } else {
        failTask(task, format("Release '%s' could not be started", subRelease.getTitle))
      }
    } catch {
      case ex: Exception =>
        logger.error(format("Exception occurred while creating a release. Reason: %s.", ex.getMessage), ex)
        failTask(task, ex.getMessage)
    }
    null
  }

  private def validateCreateReleaseRecursion(task: CreateReleaseTask): Unit = {
    if (isNotBlank(task.getTemplateIdVariable) && isNotBlank(task.getRelease.getOriginTemplateId)) {
      checkArgument(task.getRelease.getOriginTemplateId != task.getTemplateId,
        "The template that will be used to create a new release must be different from the template used to create the current release.".asInstanceOf[Any])
    }
  }

  private def validateRequiredProperties(task: CreateReleaseTask): Unit = {
    checkArgument(isNotBlank(task.getNewReleaseTitle), "Title of the release to be created cannot be empty.".asInstanceOf[Any])
    val errorMessageTemplate: Any = if (isNotBlank(task.getTemplateIdVariable)) {
      "Template that will be used to create a release couldn't be derived from variable " + task.getTemplateIdVariable
    } else {
      "Template that will be used to create a release must be specified."
    }
    checkArgument(isNotBlank(task.getTemplateId), errorMessageTemplate)
    checkArgument(releaseService.exists(task.getTemplateId),
      s"Template provided in variable ${task.getTemplateIdVariable} must be a valid template id.".asInstanceOf[Any])
  }

  private def checkPermissions(task: CreateReleaseTask): UsernamePassword = {
    checkPermissions(task, task.getTemplateId, null)
  }

  private def checkPermissions(task: CreateReleaseTask, releaseId: String, requiredPermission: Permission): UsernamePassword = {
    var usernamePass: UsernamePassword = null
    try {
      usernamePass = authenticationService.loginScriptUser(task)
      // in case template is moved and target folder id is not specified create release task needs to create release in the new folder of the template
      val templateId = releaseService.getFullId(Ids.getFolderlessId(task.getTemplateId))
      val targetFolderId = if (!Ids.isNullId(task.getFolderId)) {
        task.getFolderId
      } else {
        Ids.findFolderId(templateId)
      }
      permissionChecker.checkIsAllowedToCreateReleaseFromTemplate(templateId, targetFolderId)
      if (requiredPermission != null) {
        permissionChecker.check(requiredPermission, releaseId)
      }
    } finally {
      authenticationService.logoutScriptUser()
    }
    usernamePass
  }

  private def createRelease(task: CreateReleaseTask, scriptUsernamePass: UsernamePassword): Release = {
    val subReleaseTemplate = releaseService.findById(task.getTemplateId)
    val releaseMetadata = getSubReleaseParamsFromTask(task, subReleaseTemplate)
    val rootReleaseId = Optional.ofNullable(task.getRelease.getRootReleaseId).orElse(task.getRelease.getId)
    try throwIfMaxConcurrencyReached(releaseService.findByIdIncludingArchived(rootReleaseId))
    catch {
      case ex: NotFoundException =>
        logger.debug(format("Root release not found. Missing release max concurrency check. Reason: %s.", ex.getMessage), ex)
    }
    releaseMetadata.setRootReleaseId(rootReleaseId)
    // inherit owner, scriptUsername, scriptPassword from task's release
    releaseMetadata.setOwner(scriptUsernamePass.username)
    releaseMetadata.setScriptUsername(scriptUsernamePass.username)
    // Password should be already decrypted
    releaseMetadata.setScriptUserPassword(scriptUsernamePass.password)

    //As we check if user are passing encryptedValues or not, it is required to decrypt internally created releaseMetadata for create release task
    EncryptionHelper.decrypt(releaseMetadata)
    val subRelease = releaseService.createFromTemplate(task.getTemplateId, releaseMetadata, task.getFolderId)
    commentService.create(task, format("Created release %s.", getReleaseLink(subRelease)), AUTHENTICATED_USER, true)
    eventBus.publish(ReleaseCreatedEvent(subRelease, CreatedFromCreateReleaseTask(task, subReleaseTemplate.getId)))
    subRelease
  }

  private def startRelease(task: CreateReleaseTask, subRelease: Release): Boolean = {
    val startedRelease = releaseActorService.startRelease(subRelease.getId, SYSTEM)
    val hasBeenStarted = startedRelease.getStatus.hasBeenStarted
    if (hasBeenStarted) {
      commentService.create(task.getId, format("Started release %s.", getReleaseLink(startedRelease)), AUTHENTICATED_USER, true)
      eventBus.publish(ReleaseStartedFromCreateReleaseTaskEvent(task, subRelease))
    }
    hasBeenStarted
  }

  private def throwIfMaxConcurrencyReached(rootRelease: Release): Unit = {
    val maxConcurrency = rootRelease.getMaxConcurrentReleases
    val spawnedReleases = releaseService.findSpawnedReleases(rootRelease.getId, maxConcurrency)
    val rootCount = if (rootRelease.isDefunct) 0 else 1
    val currentlyRunning = spawnedReleases.size + rootCount
    logger.debug("maxConcurrency: {}, currently running releases ({}): {}", maxConcurrency, currentlyRunning, spawnedReleases)
    if (currentlyRunning >= maxConcurrency) {
      throw new IllegalStateException(String.format("Exceeded maximum number of concurrent releases created from %s. "
        + "Current maximum is %s. This value can be configured in deployit-defaults.properties: "
        + "xlrelease.Release.maxConcurrentReleases", rootRelease.getTitle, maxConcurrency))
    }
  }

  private def getSubReleaseParamsFromTask(task: CreateReleaseTask, subReleaseTemplate: Release): Release = {
    val templateVariables: JMap[String, Variable] = VariableHelper.indexByKey(subReleaseTemplate.getVariables)
    import scala.jdk.CollectionConverters._
    val variablesShownOnReleaseStart: JList[Variable] = task.getTemplateVariables.asScala.filter(
      (variable: Variable) => templateVariables.containsKey(variable.getKey) && templateVariables.get(variable.getKey).getShowOnReleaseStart
    ).asJava
    val variables: JList[Variable] = fillVariableValues(subReleaseTemplate.getVariables, variablesShownOnReleaseStart)
    val release: Release = newRelease
      .withVariables(variables)
      .withOriginTemplateId(task.getTemplateId)
      .withTitle(task.getNewReleaseTitle)
      .withOwner(task.getRelease.getScriptUsername)
      .withScriptUsername(task.getRelease.getScriptUsername)
      .withScriptUserPassword(task.getRelease.getScriptUserPassword)
      .withDisableNotifications(task.getRelease.isDisableNotifications)
      .withTags(new JArrayList[String](task.getReleaseTags))
      .build
    release.setProperty(RISK_PROFILE, task.getProperty(RISK_PROFILE))
    releaseService.setDatesFromTemplate(release, subReleaseTemplate)
    release.setStartedFromTaskId(task.getId)
    release.setAbortOnFailure(subReleaseTemplate.isAbortOnFailure)
    release.setAllowPasswordsInAllFields(subReleaseTemplate.isAllowPasswordsInAllFields)
    fixUpVariableIds(release.getId, release.getVariables, ciIdService)
    release
  }

  private def saveCreatedReleaseIdIntoVariable(task: CreateReleaseTask, subRelease: Release): Unit = {
    var variableName: String = ""
    if (containsOnlyVariable(task.getCreatedReleaseId)) {
      variableName = task.getCreatedReleaseId
    }
    if (task.getVariableMapping.containsKey(CREATED_RELEASE_ID)) {
      variableName = task.getVariableMapping.get(CREATED_RELEASE_ID)
    }
    val createdReleaseId: String = subRelease.getId
    val variablesByKeys: JMap[String, Variable] = task.getRelease.getVariablesByKeys
    val variableKey: String = withoutVariableSyntax(variableName)
    if (variablesByKeys.containsKey(variableKey)) {
      val variable: Variable = variablesByKeys.get(variableKey)
      variable.setUntypedValue(createdReleaseId)
      releaseActorService.updateVariable(variable.getId, variable)
    }
  }

  private def saveCreatedReleaseIdIntoTask(task: CreateReleaseTask, subRelease: Release): Unit = {
    if (task.getVariableMapping.containsKey(CREATED_RELEASE_ID)) {
      task.getVariableMapping.remove(CREATED_RELEASE_ID)
    }
    task.setCreatedReleaseId(subRelease.getId)
    releaseActorService.updateTask(task.getId, task, Collections.singleton(UPDATE_RELEASE_TASK), overrideLock = true)
  }

  private def getReleaseLink(release: Release) = format("[%s](#/releases/%s)", release.getTitle, DOMAIN.convertToViewId(release.getId))

  private def failTask(task: CreateReleaseTask, message: String): Unit = {
    releaseActorService.failTaskAsync(task.getId, CREATE_RELEASE_TASK_FAILED.format(message), SYSTEM, Option.empty)
  }

  private def completeTask(task: CreateReleaseTask): Unit = {
    releaseActorService.markTaskAsDoneAsync(COMPLETED, task.getId, null, SYSTEM)
  }
}
