package com.xebialabs.xlrelease.principaldata;

import java.util.Collection;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.dao.DataAccessException;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.AuthorityUtils;
import org.springframework.security.core.userdetails.UsernameNotFoundException;
import com.atlassian.crowd.integration.springsecurity.user.CrowdUserDetails;
import com.atlassian.crowd.integration.springsecurity.user.CrowdUserDetailsServiceImpl;

/**
 * Retrieves email addresses and full names from an crowd directory, reusing the
 * spring security configuration.
 */
public class CrowdUserDataProvider implements UserDataProvider {

    private static final Logger logger = LoggerFactory.getLogger(CrowdUserDataProvider.class);
    private final CrowdUserDetailsServiceImpl crowdUserDetailsServiceImpl;

    public CrowdUserDataProvider(CrowdUserDetailsServiceImpl crowdUserDetailsService) {
        this.crowdUserDetailsServiceImpl = crowdUserDetailsService;
    }

    @Override
    public UserData getUserData(String username) {
        try {
            CrowdUserDetails crowdUserDetails = crowdUserDetailsServiceImpl.loadUserByUsername(username);
            if (crowdUserDetails == null) {
                return UserData.NOT_FOUND;
            }
            String email = crowdUserDetails.getEmail();
            String fullName = crowdUserDetails.getFullName();
            logger.info("Email: {} and displayName: {} have been read from Crowd", email, fullName);
            return new UserData(email, fullName);
        } catch (UsernameNotFoundException e) {
            logger.info("User with username: {} not found", username);
            return UserData.NOT_FOUND;
        } catch (DataAccessException e) {
            logger.warn("Could not get data from Crowd Server for User with username: {}", username);
            return UserData.NOT_FOUND;
        } catch (Exception e) {
            logger.warn("Error accessing Crowd server", e);
            return UserData.NOT_FOUND;
        }
    }

    @Override
    public Collection<GrantedAuthority> getAuthorities(final String username) {
        try {
            CrowdUserDetails crowdUserDetails = crowdUserDetailsServiceImpl.loadUserByUsername(username);
            if (crowdUserDetails == null) {
                return AuthorityUtils.NO_AUTHORITIES;
            }
            return crowdUserDetails.getAuthorities();
        } catch (UsernameNotFoundException e) {
            logger.info("User with username: {} not found", username);
            return AuthorityUtils.NO_AUTHORITIES;
        } catch (DataAccessException e) {
            logger.warn("Could not get data from Crowd Server for User with username: {}", username);
            return AuthorityUtils.NO_AUTHORITIES;
        } catch (Exception e) {
            logger.warn("Error accessing Crowd server", e);
            return AuthorityUtils.NO_AUTHORITIES;
        }
    }
}
