package com.xebialabs.deployit.plumbing;

import java.io.IOException;
import java.time.Instant;
import java.time.ZoneId;
import java.time.ZonedDateTime;
import java.time.format.DateTimeFormatter;
import javax.servlet.*;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;

public class ModifiedResourceFilter implements Filter {

    private static final String IF_MODIFIED_SINCE_HEADER = "If-Modified-Since";
    private static final String LAST_MODIFIED_HEADER = "Last-Modified";
    private ZonedDateTime startDateTime;

    @Override
    public void init(final FilterConfig filterConfig) {
        long startTimestamp = Long.parseLong(filterConfig.getInitParameter("startTimestamp"));
        startDateTime = ZonedDateTime.ofInstant(Instant.ofEpochMilli(startTimestamp), ZoneId.systemDefault());
    }

    @Override
    public void doFilter(final ServletRequest req, final ServletResponse res, final FilterChain chain) throws IOException, ServletException {
        String ifModifiedSinceHeader = ((HttpServletRequest) req).getHeader(IF_MODIFIED_SINCE_HEADER);
        HttpServletResponse response = (HttpServletResponse) res;

        if (ifModifiedSinceHeader != null) {
            ZonedDateTime ifModifiedSinceDateTime = ZonedDateTime.parse(ifModifiedSinceHeader, DateTimeFormatter.RFC_1123_DATE_TIME);
            if (ifModifiedSinceDateTime.isAfter(startDateTime)) {
                response.sendError(304);
            } else {
                addHeaderAndDoFilter(req, res, chain);
            }
        } else {
            addHeaderAndDoFilter(req, res, chain);
        }
    }

    private void addHeaderAndDoFilter(final ServletRequest req, final ServletResponse res, final FilterChain chain) throws IOException, ServletException {
        ((HttpServletResponse) res).setHeader(LAST_MODIFIED_HEADER, ZonedDateTime.now().format(DateTimeFormatter.RFC_1123_DATE_TIME));
        chain.doFilter(req, res);
    }

    @Override
    public void destroy() {}

}
