package com.xebialabs.deployit.plumbing.authentication;

import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.access.AccessDecisionVoter;
import org.springframework.security.access.ConfigAttribute;
import org.springframework.security.authentication.UsernamePasswordAuthenticationToken;
import org.springframework.security.core.Authentication;
import org.springframework.security.core.GrantedAuthority;
import org.springframework.security.core.authority.SimpleGrantedAuthority;
import org.springframework.security.core.context.SecurityContextHolder;
import org.springframework.security.oauth2.client.authentication.OAuth2AuthenticationToken;

import com.xebialabs.xlrelease.domain.UserProfile;
import com.xebialabs.xlrelease.security.XLReleasePermissions;
import com.xebialabs.xlrelease.security.authentication.RunnerAuthenticationToken;
import com.xebialabs.xlrelease.service.UserLastActiveActorService;
import com.xebialabs.xlrelease.service.UserProfileService;

import static com.xebialabs.xlrelease.security.XLReleasePermissions.ADMIN_USERNAME;
import static com.xebialabs.xlrelease.security.XLReleasePermissions.isAdmin;

// TODO: this class has a very bad case of side effects that it should not have

// TODO: move profile creation to the authentication providers all knowledge is there (if you properly configure them (LDAP))
// TODO: this class should not read from the DB. This not needed if properly using granted authorities in authentication. And will lead to speed up in the while application since a subset of permission checks can be done without touching the DB.
// TODO: in stead of isAdmin and checking on the admin user name using an ROLE_ADMIN in granted authorities would be much nicer
public class LoginPermissionVoter implements AccessDecisionVoter<Object> {
    private static final Logger logger = LoggerFactory.getLogger(LoginPermissionVoter.class);

    private final UserProfileService userProfileService;
    private final UserLastActiveActorService userLastActiveActorService;

    public LoginPermissionVoter(UserProfileService userProfileService, UserLastActiveActorService userLastActiveActorService) {
        this.userProfileService = userProfileService;
        this.userLastActiveActorService = userLastActiveActorService;
    }

    public boolean supports(ConfigAttribute attribute) {
        return true;
    }

    @Override
    public boolean supports(Class<?> clazz) {
        return true;
    }

    @Override
    public int vote(Authentication authentication, Object object, Collection<ConfigAttribute> attributes) {
        String username = authentication.getName();
        logger.debug("Checking login permission for [{}]", username);

        if (authentication instanceof RunnerAuthenticationToken) {
            return ACCESS_GRANTED;
        }

        if (isAdmin(username)) {
            userLastActiveActorService.updateLastActive(ADMIN_USERNAME);
            return ACCESS_GRANTED;
        }

        // TODO: if we would put the permissions in the Authentication object then we would not need this call here.
        // TODO: this triggers for every web request hitting the filter chain
        UserProfile userProfile = userProfileService.findByUsername(username);

        if (userProfile != null && userProfile.isLoginAllowed()) {
            logger.debug("User [{}] is authorized for login", username);

            onAccessGranted(authentication, userProfile);

            return ACCESS_GRANTED;
        }

        logger.error("User [{}] is not authorized for login", username);

        return ACCESS_DENIED;
    }

    private void onAccessGranted(Authentication authentication, UserProfile userProfile) {
        userLastActiveActorService.updateLastActive(userProfile.getCanonicalId());

        // TODO: this does not belong here
        // TODO: https://digitalai.atlassian.net/browse/REL-1768
        Object details = authentication.getDetails();
        String realUserName = userProfile.getName();

        var oldAuthorities = authentication.getAuthorities();
        var authenticatedUserAuthority = new SimpleGrantedAuthority(XLReleasePermissions.AUTHENTICATED_USER);
        Collection<? extends GrantedAuthority> newAuthorities = oldAuthorities;
        if (!oldAuthorities.contains(authenticatedUserAuthority)) {
            Collection<GrantedAuthority> modifiableAuthorities = new HashSet<>();
            modifiableAuthorities.add(authenticatedUserAuthority);
            modifiableAuthorities.addAll(oldAuthorities);
            newAuthorities = Collections.unmodifiableCollection(modifiableAuthorities) ;
        }

        if (authentication instanceof OAuth2AuthenticationToken) {
            final OAuth2AuthenticationToken oldToken = (OAuth2AuthenticationToken) authentication;
            var newAuthentication = new OAuth2AuthenticationToken(oldToken.getPrincipal(), newAuthorities, oldToken.getAuthorizedClientRegistrationId());
            SecurityContextHolder.getContext().setAuthentication(newAuthentication);
            return;
        }

        UsernamePasswordAuthenticationToken token = new UsernamePasswordAuthenticationToken(
                realUserName,
                authentication.getCredentials(),
                newAuthorities);
        token.setDetails(details);

        // TODO: this is broken and breaks the spring security API's
        SecurityContextHolder.getContext().setAuthentication(token);
    }

}
