package com.xebialabs.xlrelease.principaldata;

import java.util.Collection;
import java.util.Collections;
import java.util.concurrent.ExecutionException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.security.core.GrantedAuthority;
import com.google.common.cache.CacheBuilder;
import com.google.common.cache.CacheLoader;
import com.google.common.cache.LoadingCache;

import com.xebialabs.xlrelease.config.XlrConfig;
import com.xebialabs.xlrelease.principaldata.exceptions.GroupEmailNotFoundException;

import static java.util.concurrent.TimeUnit.MINUTES;

/**
 * Cache decorator for PrincipalDataProvider
 */
public class PrincipalDataProviderCache implements PrincipalDataProvider {
    private static final Logger logger = LoggerFactory.getLogger(PrincipalDataProviderCache.class);

    private PrincipalDataProvider principalDataProvider;
    private XlrConfig xlrConfig;

    private LoadingCache<String, UserData> userDataCache;
    private LoadingCache<String, Collection<? extends GrantedAuthority>> authoritiesCache;
    private LoadingCache<String, String> groupEmailCache;

    PrincipalDataProviderCache(PrincipalDataProvider principalDataProvider, XlrConfig xlrConfig) {
        this.principalDataProvider = principalDataProvider;
        this.xlrConfig = xlrConfig;

        initCaches();
    }

    public void clearCache() {
        userDataCache.invalidateAll();
        authoritiesCache.invalidateAll();
        groupEmailCache.invalidateAll();
    }

    public void initCaches() {
        this.userDataCache = CacheBuilder.newBuilder()
                .expireAfterWrite(xlrConfig.cacheTtl("external-users").toMinutes(), MINUTES)
                .maximumSize(xlrConfig.cacheMaxSize("external-users"))
                .build(
                        new CacheLoader<>() {
                            @Override
                            public UserData load(String username) throws Exception {
                                return principalDataProvider.getUserData(username);
                            }
                        });

        this.authoritiesCache = CacheBuilder.newBuilder()
                .expireAfterWrite(xlrConfig.cacheTtl("external-authorities").toMinutes(), MINUTES)
                .maximumSize(xlrConfig.cacheMaxSize("external-authorities"))
                .build(
                        new CacheLoader<>() {
                            @Override
                            public Collection<? extends GrantedAuthority> load(String username) {
                                return principalDataProvider.getAuthorities(username);
                            }
                        }
                );

        this.groupEmailCache = CacheBuilder.newBuilder()
                .expireAfterWrite(xlrConfig.cacheTtl("external-group-emails").toMinutes(), MINUTES)
                .maximumSize(xlrConfig.cacheMaxSize("external-group-emails"))
                .build(
                        new CacheLoader<>() {
                            @Override
                            public String load(String groupName) throws Exception {
                                String email = principalDataProvider.getGroupEmail(groupName);
                                if (email != null) {
                                    return email;
                                }
                                throw new GroupEmailNotFoundException(groupName);
                            }
                        }
                );
    }


    @Override
    public UserData getUserData(String username) {
        try {
            return userDataCache.get(username);
        } catch (ExecutionException e) {
            throw new RuntimeException(e);
        }
    }

    @Override
    public String getGroupEmail(String groupName) {
        try {
            return groupEmailCache.get(groupName);
        } catch (ExecutionException e) {
            logger.warn("Cannot fetch group email: {}", e.getMessage());
            return null;
        }
    }

    @Override
    public Collection<? extends GrantedAuthority> getAuthorities(final String username) {
        try {
            return authoritiesCache.get(username);
        } catch (ExecutionException e) {
            logger.warn("Cannot fetch authorities: {}", e.getMessage());
            return Collections.emptySet();
        }
    }

    @Override
    public void invalidate(String username) {
        this.userDataCache.invalidate(username);
        this.authoritiesCache.invalidate(username);
    }
}
