package com.xebialabs.xlrelease.repository

import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.xlrelease.domain.DependencyTarget
import com.xebialabs.xlrelease.domain.status.{FlagStatus, PhaseStatus, TaskStatus}
import com.xebialabs.xlrelease.domain.variables.{StringVariable => XlrStringVariable}
import com.xebialabs.xlrelease.repository.IdMatchers.{PhaseId, PlanItemId, ReleaseId, TaskId}
import com.xebialabs.xlrelease.repository.ReleaseJsonParser._
import com.xebialabs.xlrelease.service.VariableService
import com.xebialabs.xlrelease.support.akka.spring.ScalaSpringSupport
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.variable.VariableHelper.{containsVariables, replaceAll, withVariableSyntax}
import org.springframework.context.{ApplicationContext, ApplicationContextAware}
import org.springframework.stereotype.Component

import java.util.Date
import javax.xml.bind.DatatypeConverter
import scala.beans.BeanProperty
import scala.collection.mutable
import scala.jdk.CollectionConverters._

@Component
class DependencyTargetLoader extends ApplicationContextAware with ScalaSpringSupport {

  @BeanProperty
  protected var applicationContext: ApplicationContext = _

  lazy val releaseRepository: ReleaseRepository = springBean[ReleaseRepository]
  lazy val folderVariableRepository: FolderVariableRepository = springBean[FolderVariableRepository]
  lazy val variableService: VariableService = springBean[VariableService]

  def getDependencyNodeData(id: String): DependencyTarget = {
    id match {
      case PlanItemId(_) => getPlanItemDependencyNodeData(id)
      case _ => throw new IllegalArgumentException(s"Id [$id] is not a plan item id")
    }
  }

  private def getPlanItemDependencyNodeData(planItemId: String): DependencyTarget = {
    val releaseId = Ids.releaseIdFrom(planItemId)
    val targetReleaseUid = releaseRepository.getUid(releaseId).orNull
    val json = releaseRepository.getReleaseJson(releaseId)
    val level = planItemId match {
      case ReleaseId(_) => ReleaseLevel
      case TaskId(_) => TaskLevel
      case PhaseId(_) => PhaseLevel
    }
    implicit val releaseData: ReleaseData = ReleaseJsonParser.parse(json, level)
    val allStringVariableValues = getAllStringVariableValues(planItemId, releaseData)
    val releaseTitle = releaseData.title
    val releaseFlagStatus = FlagStatus.valueOf(releaseData.realFlagStatus.toUpperCase)
    val releaseFlagComment = releaseData.flagComment

    val phaseData: Option[PhaseData] = getPhaseData(planItemId, releaseData)
    val taskData: Option[BaseTaskData] = getTaskData(planItemId, phaseData)
    val planItemData: PlanItemData = taskData.getOrElse(phaseData.getOrElse(releaseData))

    val phaseId = phaseData.map(_ => Ids.phaseIdFrom(planItemId)).orNull
    val phaseTitle = phaseData.map(_.title).orNull
    val resolvedPhaseTitle = resolve(phaseTitle, allStringVariableValues)
    val phaseStatus = phaseData.map(pd => toPhaseStatus(pd.status)).orNull

    val taskId = taskData.map(_ => Ids.taskIdFrom(planItemId)).orNull
    val taskTitle = taskData.map(_.title).orNull
    val resolvedTaskTitle = resolve(taskTitle, allStringVariableValues)
    val taskStatus = taskData.map(td => toTaskStatus(td.status)).orNull

    val dependencyTarget = DependencyTarget(
      targetReleaseUid = targetReleaseUid,
      targetId = planItemId,
      targetType = Type.valueOf(planItemData.itemType),
      releaseTitle = releaseTitle,
      releaseFlagStatus = releaseFlagStatus,
      releaseFlagComment = releaseFlagComment,
      phaseId = phaseId,
      phaseTitle = phaseTitle,
      resolvedPhaseTitle = resolvedPhaseTitle,
      phaseStatus = phaseStatus,
      taskId = taskId,
      taskTitle = taskTitle,
      resolvedTaskTitle = resolvedTaskTitle,
      taskStatus = taskStatus,
      startDate = toDate(planItemData.startDate),
      endDate = toDate(planItemData.endDate),
      scheduledStartDate = toDate(planItemData.scheduledStartDate),
      dueDate = toDate(planItemData.dueDate),
      plannedDuration = planItemData.plannedDuration
    )

    dependencyTarget
  }


  private def toPhaseStatus(statusValue: String): PhaseStatus = {
    Option(statusValue).map(value => PhaseStatus.valueOf(value.toUpperCase)).orNull
  }

  private def toTaskStatus(statusValue: String): TaskStatus = {
    Option(statusValue).map(value => TaskStatus.valueOf(value.toUpperCase)).orNull
  }

  private def getPhaseData(planItemId: String, releaseData: ReleaseData): Option[PhaseData] = {
    if (Ids.isPhaseId(planItemId) || Ids.isTaskId(planItemId)) {
      val phaseId = Ids.phaseIdFrom(Ids.getFolderlessId(planItemId))
      val phaseData = releaseData.getPhase(phaseId)
      Some(phaseData)
    } else {
      None
    }
  }

  private def getTaskData(planItemId: String, phaseData: Option[PhaseData]): Option[BaseTaskData] = {
    planItemId match {
      case TaskId(_) =>
        phaseData.map { pd =>
          val taskId = Ids.getFolderlessId(planItemId)
          pd.getTask(taskId)
        }
      case _ => None
    }
  }

  private def toDate(stringValue: String): Date = {
    if (null != stringValue) {
      DatatypeConverter.parseDateTime(stringValue).getTime()
    } else {
      null
    }
  }

  private def getAllStringVariableValues(planItemId: String, releaseData: ReleaseData): java.util.Map[String, String] = {
    val releaseVars = releaseData.allStringVariables().map(v => VariableHelper.withVariableSyntax(v.key) -> v.value).toMap
    val releaseId = Ids.releaseIdFrom(planItemId)
    val allVars = releaseVars ++ releasePropertyVariables(releaseId, releaseData) ++ folderVariables(planItemId) ++ globalVariables()
    allVars.asJava
  }

  private def folderVariables(planItemId: String): Map[String, String] = {
    val folderId = Ids.findFolderId(planItemId)
    if (!Ids.isRoot(folderId)) {
      folderVariableRepository.getAllFromAncestry(folderId).asScala.collect {
        case v: XlrStringVariable => VariableHelper.withVariableSyntax(v.getKey) -> v.getValue
      }.toMap
    } else {
      Map()
    }
  }

  private def globalVariables(): mutable.Map[String, String] = {
    val globalVariables = variableService.findGlobalVariablesOrEmpty()
    globalVariables.getStringVariableValues.asScala
  }

  private def releasePropertyVariables(releaseId: String, releaseData: ReleaseData): Map[String, String] = {
    val releaseTags = if (releaseData.tags != null) releaseData.tags.mkString(", ") else null
    // see ReleasePropertyVariableKey for possible values, not all of them make much sense
    Map[String, String](
      withVariableSyntax("release.id") -> releaseId,
      withVariableSyntax("release.title") -> releaseData.title,
      withVariableSyntax("release.tags") -> releaseTags
    )
  }

  private def resolve(title: String, allStringVariableValues: java.util.Map[String, String]): String = {
    // ENG-8033: re-think if we want to resolve variables at all on a dependency that is not started yet
    Option(title) match {
      case None => null
      case Some(title) if containsVariables(title) => replaceAll(title, allStringVariableValues)
      case Some(title) => title
    }
  }
}
