package com.xebialabs.xlrelease.actors

import com.google.common.base.Preconditions._
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.deployit.plugin.api.reflect.Type
import com.xebialabs.deployit.repository.ItemConflictException
import com.xebialabs.xlrelease.actors.ReleaseExecutionActor.METRIC_PREFIX
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.AttachmentMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.DependenciesMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.ExecutionServiceMessages.{BackpressureResponse, BackpressureWait, FailScriptTaskAsync, FailTaskAsync}
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.GateMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.PhasesMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.ReleaseMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.TaskMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.TeamMessages.{AddTeam, DeleteTeam, UpdateTeam, UpdateTeams}
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.TemplateMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.VariableMessages._
import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages._
import com.xebialabs.xlrelease.actors.ReleaseSupervisorActor.{GcRequest, ReleaseDeleted, ReleaseIsActive, ReleasePoisonPill}
import com.xebialabs.xlrelease.actors.TaskPredicate._
import com.xebialabs.xlrelease.actors.sharding.ReleaseShardingMessages.ReleaseAction
import com.xebialabs.xlrelease.api.v1.forms.VariableOrValue
import com.xebialabs.xlrelease.domain.CreateReleaseTask.CREATED_RELEASE_ID
import com.xebialabs.xlrelease.domain.FailureReasons.CREATE_RELEASE_TASK_FAILED
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.events.ReleaseRiskScoreUpdated
import com.xebialabs.xlrelease.domain.status.TaskStatus.COMPLETED
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective
import com.xebialabs.xlrelease.domain.tasks.TaskUpdateDirective.UPDATE_RELEASE_TASK
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.repository.IdMatchers.{PhaseId, ReleaseId, TaskId}
import com.xebialabs.xlrelease.repository.{CiCloneHelper, PhaseVersion}
import com.xebialabs.xlrelease.scheduler.workers.Worker._
import com.xebialabs.xlrelease.script.ContainerTaskResult
import com.xebialabs.xlrelease.script.DefaultScriptService.{BaseScriptTaskResults, CustomScriptTaskResults, ScriptTaskResults}
import com.xebialabs.xlrelease.serialization.json.repository.ResolveOptions
import com.xebialabs.xlrelease.user.User
import com.xebialabs.xlrelease.user.User.SYSTEM
import com.xebialabs.xlrelease.utils.PartialFunctionBuilder
import com.xebialabs.xlrelease.variable.VariableHelper.{containsOnlyVariable, containsVariables, withoutVariableSyntax}
import com.xebialabs.xlrelease.views.MovementIndexes
import grizzled.slf4j.Logging
import org.apache.pekko.actor._
import org.joda.time.DateTime
import org.springframework.security.core.Authentication
import org.springframework.security.core.context.SecurityContextHolder

import java.util.{Collections, Date, Map => JMap}
import scala.jdk.CollectionConverters._
import scala.util.{Failure, Success, Try}

// scalastyle:off file.size.limit

/**
 * All messages of release actor need to be serializable as they travel between cluster nodes.
 */
object ReleaseExecutionActorMessages {

  case class CallerContext(authentication: Option[Authentication] = Option(SecurityContextHolder.getContext.getAuthentication))

  sealed class Command(val invalidatesCache: Boolean = true) extends Serializable {
    val callerContext: CallerContext = CallerContext()
  }

  object ExecutionServiceMessages {

    case class StartRelease(user: User, releaseStartedImmediatelyAfterBeingCreated: Boolean = false, isPartOfBulkOperation: Boolean = false, needsResponse: Boolean = true) extends Command

    case class AbortTask(taskId: String, comment: String) extends Command

    case class ReopenTask(taskId: String, comment: String) extends Command

    case class RetryTask(taskId: String, comment: String) extends Command

    case class RetryTasks(taskIds: List[String], comment: String) extends Command

    case class StartTask(taskId: String, comment: String) extends Command

    case class StartTaskWithInput(taskId: String, inputs: List[Variable]) extends Command

    case class StartTaskAsync(taskId: String, comment: String, user: User = User.AUTHENTICATED_USER) extends Command

    case class ResumeRelease() extends Command

    case class AbortRelease(abortComment: String, isPartOfBulkOperation: Boolean = false) extends Command

    case class AbortTasks(taskIds: List[String], comment: String) extends Command

    case class ReopenTasks(taskIds: List[String], comment: String) extends Command

    case class MarkTaskAsDone(taskId: String, targetStatus: TaskStatus, comment: String, user: User, logArtifactId: Option[String]) extends Command

    case class MarkTaskAsDoneWithScriptResults(taskId: String, targetStatus: TaskStatus, comment: String, logArtifactId: Option[String], scriptResults: Option[BaseScriptTaskResults], user: User, executionId: String) extends Command

    case class SkipTaskDueToPreconditionCheck(taskId: String, executionId: String, executionLog: String) extends Command

    case class FinishContainerTask(taskResult: ContainerTaskResult) extends Command

    case class FinishCustomScriptTask(taskId: String, comment: String, executionId: String, logArtifactId: Option[String], scriptResults: Option[BaseScriptTaskResults]) extends Command

    case class FinishScriptTask(taskId: String, comment: String, executionId: String, logArtifactId: Option[String], scriptResults: Option[BaseScriptTaskResults]) extends Command

    case class FailTask(taskId: String, comment: String, scriptResults: Option[BaseScriptTaskResults], user: User = User.AUTHENTICATED_USER, executionId: Option[String] = None) extends Command

    case class FailTaskAsync(taskId: String, comment: String, scriptResults: Option[BaseScriptTaskResults], user: User = User.AUTHENTICATED_USER, executionId: Option[String] = None) extends Command

    case class FailTasks(taskIds: Seq[String], comment: String, user: User = User.AUTHENTICATED_USER) extends Command

    case class FailScriptTask(taskId: String, scriptFailMessage: String, scriptExecutionId: String, logArtifactId: Option[String], scriptResults: Option[BaseScriptTaskResults], user: User = User.LOG_OUTPUT) extends Command

    case class FailScriptTaskAsync(taskId: String, scriptFailMessage: String, scriptExecutionId: String, logArtifactId: Option[String], scriptResults: Option[BaseScriptTaskResults], user: User = User.LOG_OUTPUT) extends Command

    case class ResumeTask(taskId: String) extends Command

    case class TaskPreconditionValidated(taskId: String, executionId: String) extends Command

    case class PostponeTaskUntilEnvironmentsAreReserved(taskId: String, postponeUntil: Date, comment: String, executionId: String) extends Command

    case class BackpressureWait(taskId: String) extends Command(false)

    case class BackpressureResponse(releaseId: String) extends Serializable
  }

  object ReleaseMessages {

    case class UpdateRelease(release: Release) extends Command

    case class DeleteRelease() extends Command

    case class MovePhase(movementIndexes: MovementIndexes) extends Command

    case class UpdateVariables(vars: List[Variable]) extends Command

    case class StatusUpdated(releaseStatus: ReleaseStatus) extends Command

    case class ReleaseOverdue() extends Command(invalidatesCache = false) with NotInfluenceReceiveTimeout

    case class TasksOverdue(taskIds: Seq[String]) extends Command(invalidatesCache = false) with NotInfluenceReceiveTimeout

    case class TasksDueSoon(taskIds: Seq[String]) extends Command(invalidatesCache = false) with NotInfluenceReceiveTimeout

    case class UpdateRiskScores(releaseId:String, score: Integer, totalScore: Integer) extends Command(invalidatesCache = false) with NotInfluenceReceiveTimeout

  }

  object TeamMessages {

    case class AddTeam(team: Team) extends Command

    case class UpdateTeam(teamId: String, team: Team) extends Command

    case class UpdateTeams(teams: List[Team]) extends Command

    case class DeleteTeam(teamId: String) extends Command

  }

  object TaskMessages {

    case class AddTask(containerId: String, task: Task, position: Option[Integer]) extends Command

    case class ReassignTaskToOwner(taskId: String, owner: String) extends Command

    case class ChangeTaskType(taskId: String, taskType: Type) extends Command

    case class CompleteTask(taskId: String, comment: String) extends Command

    case class CompleteTasks(taskIds: List[String], comment: String) extends Command

    case class FailTaskManually(taskId: String, comment: String) extends Command

    case class FailTasksManually(taskIds: List[String], comment: String) extends Command

    case class SkipTask(taskId: String, comment: String, user: User = User.AUTHENTICATED_USER) extends Command

    case class SkipTasks(taskIds: List[String], comment: String, user: User = User.AUTHENTICATED_USER) extends Command

    case class DeleteTask(taskId: String) extends Command

    case class DeleteTasks(taskIds: List[String]) extends Command

    case class ReassignTaskToTeam(taskId: String, team: String) extends Command

    case class ReassignTasks(taskIds: List[String], team: String, owner: String) extends Command

    case class UpdateTask(taskId: String, task: Task, updateDirectives: java.util.Set[TaskUpdateDirective], overrideLock: Boolean = false) extends Command

    case class UpdateTaskVariables(taskId: String, variables: List[Variable], modifiedAt: DateTime, updateDirectives: Set[TaskUpdateDirective]) extends Command

    case class UpdateTaskStatusLine(taskId: String, statusLine: String) extends Command

    case class MoveTask(movementIndexes: MovementIndexes) extends Command

    case class DuplicateTask(taskId: String) extends Command

    case class CopyTask(taskId: String, targetContainerId: String, targetPosition: Int) extends Command

    case class LockTask(taskId: String) extends Command

    case class UnlockTask(taskId: String) extends Command

    case class ProcessCreateReleaseTaskExecutionResult(result: CreateReleaseTaskExecutionResult) extends Command

  }


  object PhasesMessages {

    case class DuplicatePhase(phaseId: String) extends Command

    case class AddPhase(phase: Option[Phase], position: Option[Integer]) extends Command

    case class CopyPhase(originPhaseId: String, targetPosition: Int) extends Command

    case class UpdatePhase(phaseId: String, phase: Phase) extends Command

    case class DeletePhase(phaseId: String) extends Command

  }

  case class CreateLink(entityId: String, sourceId: String, targetId: String) extends Command

  case class DeleteLink(entityId: String) extends Command

  case class UpdatePlanItemDates(entityId: String, scheduledStartDate: Date, dueDate: Date, plannedDuration: Integer) extends Command


  object TemplateMessages {

    case class DeleteTemplate() extends Command

    case class UpdateTemplate(template: Release) extends Command

    case class RestartPhase(phaseId: String, taskId: String, phaseVersion: PhaseVersion, resumeRelease: Boolean) extends Command

    case class CreateReleaseFromTrigger(
                                         folderId: String,
                                         releaseTitle: String,
                                         releaseVariables: Seq[Variable],
                                         releaseTags: Seq[String],
                                         autoStart: Boolean,
                                         scheduledStartDate: Option[Date],
                                         triggerId: String) extends Command {
    }

  }

  object AttachmentMessages {

    case class DeleteAttachment(attachmentId: String) extends Command

    case class DeleteAttachmentFromTask(taskId: String, attachmentId: String) extends Command

    case class CreateAttachmentOnRelease(attachment: Attachment) extends Command

    case class CreateAttachmentOnTask(taskId: String, attachment: Attachment) extends Command

  }

  object GateMessages {

    case class CreateGateCondition(conditionId: String) extends Command

    case class UpdateGateCondition(conditionId: String, gateCondition: GateCondition) extends Command

    case class UpdateGateConditionVersioned(conditionId: String, gateCondition: GateCondition, modifiedAt: DateTime) extends Command

    case class DeleteGateCondition(conditionId: String) extends Command

  }

  case class LogMessage(taskId: String, text: String, author: User, sendNotifications: Boolean) extends Command

  object DependenciesMessages {

    case class AddDependency(entityId: String, targetIdOrVariable: String) extends Command

    case class UpdateDependency(entityId: String, targetIdOrVariable: String) extends Command

    case class UpdateDependencyVersioned(entityId: String, targetIdOrVariable: String, modifiedAt: DateTime) extends Command

    case class DeleteDependency(entityId: String) extends Command

    case class ArchiveDependencies(dependencyIds: Seq[String]) extends Command

  }

  object ScriptServiceMessages {

    case class SaveScriptResults(taskId: String, scriptTaskResults: ScriptTaskResults) extends Command

    case class SaveCustomScriptResults(taskId: String, results: CustomScriptTaskResults, executionId: String) extends Command

  }

  object VariableMessages {

    case class CreateVariable(variable: Variable) extends Command

    case class UpdateVariable(variableId: String, variable: Variable) extends Command

    case class ReplaceVariable(variable: Variable, replacement: VariableOrValue) extends Command

    case class DeleteVariable(variableId: String) extends Command

    case class RenameVariable(variableId: String, newVariable: Variable) extends Command

  }

  abstract class ExtensionCommand extends Command

}

object ReleaseExecutionActor {
  val name = "release-execution-actor"

  val METRIC_PREFIX: String = s"${classOf[ReleaseExecutionActor].getSimpleName}:"

  def props(actorServiceHolder: ActorServiceHolder)(releaseSupervisorRef: ActorRef, releaseId: String) = Props(
    new ReleaseExecutionActor(releaseSupervisorRef, actorServiceHolder, releaseId)
  ).withDispatcher("xl.dispatchers.release-dispatcher")
}

/**
 * This actor is responsible for operations on release.
 */
class ReleaseExecutionActor(releaseSupervisor: ActorRef,
                            val actorServiceHolder: ActorServiceHolder,
                            releaseId: String)
  extends Actor with ActorLogging with ExceptionTranslateActor with Logging {

  protected lazy val handleExtensionMessagesBuilder = new PartialFunctionBuilder[Any, Unit]

  private def currentStatus = Option(actorServiceHolder.releaseService.getStatus(releaseId))

  private var cachedReleaseStatus: Option[ReleaseStatus] = currentStatus

  private def needsPolling(predicate: (ReleaseStatus) => Boolean) = cachedReleaseStatus.orElse(currentStatus).exists(predicate)

  private def release: Release = {
    // TODO why do we need decorated release here - optimize it somehow
    ReleaseActorCache.get(releaseId)(() => Try(Option(actorServiceHolder.releaseService.findById(releaseId))) match {
      case Failure(e) =>
        log.warning("Error while finding release/template {}: {}", releaseId, e)
        throw new NotFoundException(s"Error while finding release $releaseId")
      case Success(None) =>
        log.warning("Release/Template {} not found in repository", releaseId)
        throw new NotFoundException(s"Release/Template $releaseId not found in repository")
      case Success(Some(release)) =>
        release
    })
  }

  // scalastyle:off cyclomatic.complexity method.length
  def handleDefaultMessages: Receive = {
    case ExecutionServiceMessages.StartRelease(user, releaseStartedImmediatelyAfterBeingCreated, isPartOfBulkOperation, needsResponse) => replyOrFail {
      log.debug("Starting release {}", releaseId)
      val startedRelease = actorServiceHolder.executionService.start(release, user, releaseStartedImmediatelyAfterBeingCreated, isPartOfBulkOperation)
      if (needsResponse) {
        startedRelease
      } else {
        true
      }
    }

    case ExecutionServiceMessages.AbortTask(taskId, comment) => replyOrFail {
      log.debug("AbortTask {} with {}", taskId, comment)
      actorServiceHolder.taskService.abortTask(taskId, comment)
    }

    case ExecutionServiceMessages.AbortTasks(taskIds, comment) => replyOrFail {
      log.debug("AbortTasks {} with {}", taskIds, comment)
      actorServiceHolder.taskService.abortTasks(taskIds.asJava, comment)
    }

    case ExecutionServiceMessages.ReopenTask(taskId, comment) => replyOrFail {
      log.debug("ReopenTask {} with {}", taskId, comment)
      actorServiceHolder.taskService.reopenTask(taskId, comment)
    }

    case ExecutionServiceMessages.ReopenTasks(taskIds, comment) => replyOrFail {
      log.debug("ReopenTasks {} with {}", taskIds, comment)
      actorServiceHolder.taskService.reopenTasks(taskIds.asJava, comment)
    }

    case ExecutionServiceMessages.RetryTask(taskId, comment) => replyOrFail {
      log.debug("RetryTask {} with {}", taskId, comment)
      actorServiceHolder.taskService.retryTask(taskId, comment)
    }

    case ExecutionServiceMessages.RetryTasks(taskIds, comment) => replyOrFail {
      log.debug("RetryTasks {} with {}", taskIds, comment)
      actorServiceHolder.taskService.retryTasks(taskIds.asJava, comment)
    }

    case ExecutionServiceMessages.StartTask(taskId, comment) => replyOrFail {
      log.debug("StartTask {} with {}", taskId, comment)
      actorServiceHolder.taskService.startPendingTask(taskId, comment)
    }

    case ExecutionServiceMessages.StartTaskWithInput(taskId, inputs) => replyOrFail {
      log.debug("StartTaskWithInput {} with {}", taskId, inputs)
      actorServiceHolder.taskService.startWithInput(taskId, inputs.asJava)
    }

    case ExecutionServiceMessages.StartTaskAsync(taskId, comment, user) => logException {
      log.debug("StartTaskAsync {} with {}", taskId, comment)
      actorServiceHolder.executionService.startPendingTask(taskId, release, comment, user)
    }

    case ExecutionServiceMessages.FailTask(taskId, comment, taskResults, user, executionId) => replyOrFail {
      log.debug("FailTask {} with {}", taskId, comment)
      actorServiceHolder.executionService.fail(release.getTask(taskId), comment, taskResults, user, executionId)
      true
    }

    case ExecutionServiceMessages.FailTaskAsync(taskId, comment, taskResults, user, executionId) => logException {
      log.debug("FailTaskAsync {} with {}", taskId, comment)
      actorServiceHolder.executionService.fail(release.getTask(taskId), comment, taskResults, user, executionId)
    }

    case ExecutionServiceMessages.FailTasks(taskIds, comment, user) => replyOrFail {
      log.debug("FailTasks {} with {}", taskIds, comment)
      val failedTaskIds: Seq[String] = for {
        taskId <- taskIds
        _ = Try(actorServiceHolder.executionService.fail(release.getTask(taskId), comment, user)).recoverWith {
          case ex: Exception =>
            logger.warn(s"Unable to fail task $taskId", ex)
            Failure(ex)
        }
      } yield taskId
      failedTaskIds
    }

    case ExecutionServiceMessages.FailScriptTask(taskId, scriptFailMessage, scriptExecutionId, logArtifactId, scriptResults, user) => replyOrFail {
      log.debug("FailScriptTask {} with {}", taskId, scriptFailMessage)
      actorServiceHolder.executionService.scriptFailed(release.getTask(taskId), scriptFailMessage, scriptExecutionId, logArtifactId, scriptResults, user)
      true
    }

    case ExecutionServiceMessages.FailScriptTaskAsync(taskId, scriptFailMessage, scriptExecutionId, logArtifactId, scriptResults, user) => {
      // this can be invoked from within ReleaseExecutionActor and it will not try to return anything
      log.debug("FailScriptTask {} with {}", taskId, scriptFailMessage)
      actorServiceHolder.executionService.scriptFailed(release.getTask(taskId), scriptFailMessage, scriptExecutionId, logArtifactId, scriptResults, user)
    }

    case ExecutionServiceMessages.ResumeTask(taskId) => logException {
      log.debug("ResumeTask {}", taskId)
      actorServiceHolder.executionService.resumeTask(taskId)
    }

    case ExecutionServiceMessages.MarkTaskAsDone(taskId, targetStatus, comment, user, logArtifactId) => replyOrFail {
      log.debug("MarkTaskAsDone {} with {}", taskId, comment)
      actorServiceHolder.executionService.markTaskAsDone(release, targetStatus, taskId, comment, logArtifactId, None, user)
    }

    case ExecutionServiceMessages.SkipTaskDueToPreconditionCheck(taskId, executionId, executionLog) => replyOrFail {
      actorServiceHolder.executionService.skipTaskDueToPreconditionCheck(release, taskId, executionId, executionLog)
      true
    }

    case ExecutionServiceMessages.MarkTaskAsDoneWithScriptResults(taskId, targetStatus, comment, logArtifactId, scriptResults, user, executionId) =>
      replyOrFail {
        log.debug("MarkTaskAsDone {} with {}", taskId, comment)
        actorServiceHolder.executionService.markTaskAsDone(release, targetStatus, taskId, comment, logArtifactId, scriptResults, user, executionId)
      }

    case ExecutionServiceMessages.FinishContainerTask(taskResult) => replyOrFail {
      val taskId = taskResult.taskId
      log.debug("FinishContainerTask {} with {}", taskId, taskResult)
      Try(actorServiceHolder.executionService.finishContainerTask(release, taskId, taskResult))
        .recover {
          // storing the script results throw might exception, we need to fail the task with a fresh release
          case e: Exception =>
            self ! FailTaskAsync(taskId, e.getMessage, None, User.SYSTEM, None)
        }
      true
    }

    case ExecutionServiceMessages.FinishCustomScriptTask(taskId, comment, executionId, logArtifactId, scriptResults) => replyOrFail {
      log.debug("FinishCustomScriptTask {} with {}", taskId, comment)
      Try(actorServiceHolder.executionService.finishCustomScriptTask(release, taskId, comment, executionId, logArtifactId, scriptResults))
        .recover {
          // storing the script results throw might exception, we need to fail the task with a fresh release and
          case e: Exception =>
            self ! FailScriptTaskAsync(taskId, e.getMessage, executionId, logArtifactId, None, User.SYSTEM)
        }
      true
    }

    case ExecutionServiceMessages.FinishScriptTask(taskId, comment, executionId, logArtifactId, scriptResults) => replyOrFail {
      log.debug("FinishScriptTask {} with {}", taskId, comment)
      Try(actorServiceHolder.executionService.finishScriptTask(release, taskId, comment, executionId, logArtifactId, scriptResults))
        .recover {
          // storing the script results might throw exception, we need to fail the task with a fresh release
          case e: Exception =>
            self ! FailScriptTaskAsync(taskId, e.getMessage, executionId, logArtifactId, None, User.SYSTEM)
        }
      true
    }

    case ExecutionServiceMessages.ResumeRelease() => replyOrFail {
      log.debug("Resuming release {}", releaseId)
      actorServiceHolder.executionService.resume(releaseId)
    }

    case ExecutionServiceMessages.AbortRelease(abortComment, isPartOfBulkOperation) => replyOrFail {
      log.debug("Aborting release {}", releaseId)
      actorServiceHolder.executionService.abort(releaseId, abortComment, isPartOfBulkOperation)
    }

    case ExecutionServiceMessages.TaskPreconditionValidated(taskId, executionId) => replyOrFail {
      log.debug("TaskPreconditionValidated {}", taskId)
      actorServiceHolder.executionService.taskPreconditionValidated(taskId, release, executionId)
      true
    }

    case ExecutionServiceMessages.PostponeTaskUntilEnvironmentsAreReserved(taskId, postponeUntil, comment, executionId) => replyOrFail {
      log.debug("PostponeTaskUntilEnvironmentsAreReserved {} {} {}", taskId, postponeUntil, comment)
      actorServiceHolder.executionService.postponeUntilEnvironmentsAreReserved(taskId, postponeUntil, comment, executionId)
      true
    }

    case UpdateRelease(updatedRelease: Release) => replyOrFail {
      log.debug("Updating release {}", releaseId)
      actorServiceHolder.releaseService.updateRelease(releaseId, updatedRelease)
    }

    case DeleteRelease() => replyOrFail {
      log.debug("Delete release {}", releaseId)
      actorServiceHolder.releaseService.delete(releaseId)
      releaseSupervisor ! ReleaseDeleted
    }

    case AddTeam(team) => replyOrFail {
      log.debug("AddTeam {}", team)
      actorServiceHolder.teamService.addTeam(release, team)
    }

    case DeleteTeam(teamId) => replyOrFail {
      log.debug("DeleteTeam {}", teamId)
      actorServiceHolder.teamService.deleteTeam(release, teamId)
    }

    case UpdateTeam(teamId, team) => replyOrFail {
      log.debug("UpdateTeam {}", teamId)
      actorServiceHolder.teamService.updateTeam(teamId, team)
    }

    case MovePhase(indexes: MovementIndexes) => replyOrFail {
      log.debug("Moving phase: {}", indexes)
      actorServiceHolder.phaseService.movePhase(release, indexes)
    }

    case AddTask(containerId, task, position) => replyOrFail {
      log.debug("Adding task to phase {}", containerId)
      actorServiceHolder.taskService.create(containerId, task, position.orNull)
    }

    case ReassignTaskToOwner(taskId, owner) => replyOrFail {
      log.debug("ReassignTaskToOwner {} to {}", taskId, owner)
      actorServiceHolder.taskService.reassignToOwner(taskId, owner)
    }

    case ChangeTaskType(taskId, taskType) => replyOrFail {
      log.debug("ChangeTaskType {} to {}", taskId, taskType)
      actorServiceHolder.taskService.changeTaskType(taskId, taskType)
    }

    case CompleteTask(taskId, comment) => replyOrFail {
      log.debug("CompleteTask {} with {}", taskId, comment)
      actorServiceHolder.taskService.completeTask(taskId, comment)
    }

    case CompleteTasks(taskIds, comment) => replyOrFail {
      log.debug("CompleteTasks {} with {}", taskIds, comment)
      actorServiceHolder.taskService.completeTasks(taskIds.asJava, comment)
    }

    case FailTaskManually(taskId, comment) => replyOrFail {
      log.debug("FailTaskManually {} with {}", taskId, comment)
      actorServiceHolder.taskService.failTask(taskId, comment)
    }

    case FailTasksManually(taskIds, comment) => replyOrFail {
      log.debug("FailTasksManually {} with {}", taskIds, comment)
      actorServiceHolder.taskService.failTasks(taskIds.asJava, comment)
    }

    case SkipTask(taskId, comment, user) => replyOrFail {
      log.debug("SkipTask {} with {}", taskId, comment)
      actorServiceHolder.taskService.skipTask(taskId, comment, user)
    }

    case SkipTasks(taskIds, comment, user) => replyOrFail {
      log.debug("SkipTasks {} with {}", taskIds, comment)
      actorServiceHolder.taskService.skipTasks(taskIds.asJava, comment, user)
    }

    case UpdateTask(taskId, task, updateDirectives, overrideLock) => replyOrFail {
      log.debug("UpdateTask {}", taskId)
      actorServiceHolder.taskService.updateTaskWith(taskId, task, updateDirectives, overrideLock)
    }

    case UpdateTaskVariables(taskId, variables, modifiedAt, updateDirectives) => replyOrFail {
      log.debug("UpdateTaskVariables {}", taskId)
      actorServiceHolder.taskService.updateTaskVariables(taskId, variables.asJava, modifiedAt, updateDirectives.asJava)
    }

    case UpdateTaskStatusLine(taskId, statusLine) => logException {
      log.debug("UpdateTaskStatusLine {}", taskId)
      actorServiceHolder.taskService.setStatusLine(taskId, statusLine)
    }

    case DeleteTask(taskId) => replyOrFail {
      log.debug("DeleteTask {}", taskId)
      actorServiceHolder.taskService.delete(taskId)
    }

    case DeleteTasks(taskIds) => replyOrFail {
      log.debug("DeleteTasks {}", taskIds)
      actorServiceHolder.taskService.deleteTasks(taskIds.asJava)
    }

    case ReassignTaskToTeam(taskId, team) => replyOrFail {
      log.debug("ReassignTask {} to {}", taskId, team)
      actorServiceHolder.taskService.applyNewTeam(team, actorServiceHolder.taskService.findById(taskId))
    }

    case ReassignTasks(taskIds, team, owner) => replyOrFail {
      log.debug("ReassignTasks {} to team={} and owner={}", taskIds, team, owner)
      actorServiceHolder.taskService.reassignTasks(taskIds.asJava, team, owner)
    }

    case MoveTask(indexes: MovementIndexes) => replyOrFail {
      log.debug("Moving task: {}", indexes)
      actorServiceHolder.taskService.moveTask(indexes)
    }

    case LockTask(taskId) => replyOrFail {
      log.debug("Locking task: {}", taskId)
      actorServiceHolder.taskService.lockTask(taskId)
    }

    case UnlockTask(taskId) => replyOrFail {
      log.debug("Unlocking task: {}", taskId)
      actorServiceHolder.taskService.unlockTask(taskId)
    }

    case ProcessCreateReleaseTaskExecutionResult(result) =>
      log.debug("Processing CreateReleaseTask execution result: {}", result)
      result match {
        case CreateReleaseTaskExecutionResultSuccess(taskId, executionId, createdReleaseId) =>
          log.debug("CreateReleaseTaskExecutionResultSuccess {} {} {}", taskId, executionId, createdReleaseId)
          val task = release.getTask(taskId).asInstanceOf[CreateReleaseTask]
          saveCreatedReleaseIdIntoVariable(task, createdReleaseId)
          saveCreatedReleaseIdIntoTask(task, createdReleaseId)
          completeTask(taskId)
        case CreateReleaseTaskExecutionResulCreateReleaseFailure(taskId, executionId, message) =>
          log.debug("CreateReleaseTaskExecutionResulCreateReleaseFailure {} {} {}", taskId, executionId, message)
          val task = release.getTask(taskId).asInstanceOf[CreateReleaseTask]
          failTask(task, executionId, message)
        case CreateReleaseTaskExecutionResultStartReleaseFailure(taskId, executionId, createdReleaseId, message) =>
          log.debug("CreateReleaseTaskExecutionResultStartReleaseFailure {} {} {} {}", taskId, executionId, createdReleaseId, message)
          val task = release.getTask(taskId).asInstanceOf[CreateReleaseTask]
          saveCreatedReleaseIdIntoVariable(task, createdReleaseId)
          failTask(task, executionId, message)
      }

      def saveCreatedReleaseIdIntoVariable(task: CreateReleaseTask, createdReleaseId: String): Unit = {
        val variableName: String = getVariableNameForCreatedReleaseId(task)
        val variableKey: String = withoutVariableSyntax(variableName)
        val variablesByKeys: JMap[String, Variable] = task.getRelease.getVariablesByKeys
        if (variablesByKeys.containsKey(variableKey)) {
          val variable: Variable = CiCloneHelper.cloneCi(variablesByKeys.get(variableKey))
          variable.setUntypedValue(createdReleaseId)
          variable.setId(variable.getId)
          actorServiceHolder.variableService.updateVariable(release, variable)
        }
      }

      def getVariableNameForCreatedReleaseId(task: CreateReleaseTask): String = {
        var variableName: String = ""
        if (containsOnlyVariable(task.getCreatedReleaseId)) {
          variableName = task.getCreatedReleaseId
        }
        if (task.getVariableMapping.containsKey(CREATED_RELEASE_ID)) {
          variableName = task.getVariableMapping.get(CREATED_RELEASE_ID)
        }
        variableName
      }

      def saveCreatedReleaseIdIntoTask(task: CreateReleaseTask, createdReleaseId: String) = {
        if (task.getVariableMapping.containsKey(CREATED_RELEASE_ID)) {
          task.getVariableMapping.remove(CREATED_RELEASE_ID)
        }
        task.setCreatedReleaseId(createdReleaseId)

        actorServiceHolder.taskService.updateTaskWith(task.getId, task, Collections.singleton(UPDATE_RELEASE_TASK), true)
      }

      def completeTask(taskId: String): Unit = {
        actorServiceHolder.executionService.markTaskAsDone(release, COMPLETED, taskId, null, None, None, SYSTEM)
      }

      def failTask(task: CreateReleaseTask, executionId: String, message: String): Unit = {
        actorServiceHolder.executionService.fail(task, CREATE_RELEASE_TASK_FAILED.format(message), Option.empty, SYSTEM, Option(executionId))
      }

    case AddPhase(phase, position) => replyOrFail {
      log.debug("Adding phase to {}", releaseId)
      actorServiceHolder.phaseService.create(releaseId, phase.orNull, position.orNull)
    }

    case CopyPhase(originPhaseId, targetPosition) => replyOrFail {
      log.debug("Copying phase {} to position {}", originPhaseId, targetPosition)
      actorServiceHolder.phaseService.copyPhase(release, originPhaseId, targetPosition)
    }

    case UpdatePhase(phaseId, phase) => replyOrFail {
      log.debug("UpdatePhase {} to {}", phase, releaseId)
      actorServiceHolder.phaseService.update(phaseId, phase)
    }

    case DeletePhase(phaseId) => replyOrFail {
      log.debug("DeletePhase {} from {}", phaseId, releaseId)
      actorServiceHolder.phaseService.delete(phaseId)
    }

    case CopyTask(taskId, targetContainerId, targetPosition) => replyOrFail {
      log.debug("Copy task {} to position {} inside a container {}", taskId, targetPosition, targetContainerId)
      actorServiceHolder.taskService.copyTask(taskId, targetContainerId, targetPosition)
    }

    case DuplicateTask(taskId) => replyOrFail {
      log.debug("Duplicating task {}", taskId)
      actorServiceHolder.taskService.duplicateTask(taskId)
    }

    case DuplicatePhase(phaseId) => replyOrFail {
      log.debug("Duplicating phase {}", phaseId)
      actorServiceHolder.phaseService.duplicatePhase(phaseId)
    }

    case UpdateVariables(vars) => replyOrFail {
      log.debug("Updating these variables: {}", vars)
      actorServiceHolder.releaseService.updateVariables(releaseId, vars.asJava)
    }

    case UpdateTeams(teams) => replyOrFail {
      log.debug("Updating teams: {}", teams)
      actorServiceHolder.teamService.updateTeams(releaseId, teams.asJava)
    }

    case RestartPhase(phaseId, taskId, phaseVersion, resumeRelease) => replyOrFail {
      log.debug("Restarting phase {} from {}", phaseId, taskId)
      val restartedRelease = actorServiceHolder.phaseService.restartPhases(releaseId, phaseId, taskId, phaseVersion, resumeRelease, release)
      if (resumeRelease) {
        actorServiceHolder.executionService.resume(restartedRelease.getId)
      } else {
        restartedRelease
      }
    }

    case DeleteAttachment(attachmentId) => replyOrFail {
      log.debug("Deleting attachment {}", attachmentId)
      actorServiceHolder.attachmentService.deleteAttachment(releaseId, attachmentId)
    }

    case DeleteAttachmentFromTask(taskId, attachmentId) => replyOrFail {
      log.debug("Deleting attachment {}", attachmentId)
      actorServiceHolder.attachmentService.deleteAttachmentFromTask(releaseId, taskId, attachmentId)
    }

    case CreateAttachmentOnRelease(attachment) => replyOrFail {
      log.debug("Creating attachment {}", attachment)
      actorServiceHolder.attachmentService.createAttachmentOnReleaseFromActor(release, attachment)
      attachment
    }

    case CreateAttachmentOnTask(taskId, attachment) => replyOrFail {
      log.debug("Creating attachment {} on task {}", attachment, taskId)
      actorServiceHolder.attachmentService.createAttachmentOnTaskFromActor(release, taskId, attachment)
      attachment
    }

    case AddDependency(parentId, targetIdOrVariable) => replyOrFail {
      log.debug("AddDependency {} to {}", parentId, targetIdOrVariable)
      val gate: GateTask = release.getTask(parentId).asInstanceOf[GateTask]

      if (containsVariables(targetIdOrVariable)) {
        checkArgument(!gate.isInProgress, "Cannot add a dependency with a variable target to a running task.", "")
      }

      actorServiceHolder.dependencyService.create(gate, targetIdOrVariable)
    }

    case UpdateDependency(dependencyId, targetIdOrVariable) => replyOrFail {
      log.debug("Updating dependency {} to {}", dependencyId, targetIdOrVariable)
      actorServiceHolder.dependencyService.updateTarget(dependencyId, targetIdOrVariable)
    }
    case UpdateDependencyVersioned(dependencyId, targetIdOrVariable, modifiedAt) => replyOrFail {
      log.debug("Updating dependency versioned {} to {}", dependencyId, targetIdOrVariable)
      actorServiceHolder.dependencyService.updateTarget(dependencyId, targetIdOrVariable, modifiedAt)
    }

    case DeleteDependency(dependencyId) => replyOrFail {
      log.debug("Deleting dependency {}", dependencyId)
      actorServiceHolder.dependencyService.delete(release, dependencyId)
    }

    case ArchiveDependencies(dependencyIds) => replyOrFail {
      log.debug("Archiving dependencies: {}", dependencyIds)
      actorServiceHolder.dependencyService.archiveDependencies(releaseId, dependencyIds)
    }

    case msg@LogMessage(taskId, text, author, notify) => replyOrFail {
      log.debug("Logging message {}", msg)
      val task = release.getTask(taskId)
      actorServiceHolder.commentService.create(task, text, author, notify)
    }

    case ScriptServiceMessages.SaveCustomScriptResults(taskId, results, executionId) => replyOrFail {
      log.debug("Saving script results for task {}", taskId)
      actorServiceHolder.executionService.saveCustomScriptResults(release, taskId, results, executionId)
      true
    }

    case ScriptServiceMessages.SaveScriptResults(taskID, scriptTaskResults) => replyOrFail {
      actorServiceHolder.scriptResultsService.resolveScriptTaskResults(release.getTask(taskID), scriptTaskResults)
      // return true here, because Changes, returned by resolveScriptTaskResults are not
      // serializable, but we still need to send some response
      true
    }

    case CreateGateCondition(gateId) => replyOrFail {
      actorServiceHolder.gateConditionService.create(gateId)
    }

    case UpdateGateCondition(conditionId, gateCondition: GateCondition) => replyOrFail {
      actorServiceHolder.gateConditionService.update(conditionId, gateCondition)
    }

    case UpdateGateConditionVersioned(conditionId, gateCondition, modifiedAt) => replyOrFail {
      actorServiceHolder.gateConditionService.update(conditionId, gateCondition, modifiedAt)
    }

    case DeleteGateCondition(conditionId) => replyOrFail {
      actorServiceHolder.gateConditionService.delete(conditionId)
    }

    case UpdatePlanItemDates(planItemId, scheduledStartDate, dueDate, plannedDuration) => replyOrFail {

      def read(id: String): PlanItem = id match {
        case ReleaseId(_) => actorServiceHolder.releaseService.findById(id)
        case PhaseId(_) => actorServiceHolder.phaseService.findById(id)
        case TaskId(_) => actorServiceHolder.taskService.findById(id)
      }

      def update(item: PlanItem): PlanItem = {
        item match {
          case release: Release if release.isTemplate => actorServiceHolder.releaseService.updateTemplate(item.getId, release)
          case release: Release if !release.isTemplate => actorServiceHolder.releaseService.updateRelease(item.getId, release)
          case phase: Phase => actorServiceHolder.phaseService.update(item.getId, phase)
          case task: Task => actorServiceHolder.taskService.updateTaskWith(task.getId, task)
        }
      }

      val item: PlanItem = read(planItemId)
      checkArgument(item.isUpdatable)
      item.updateDates(scheduledStartDate, dueDate, plannedDuration)
      update(item)
    }

    case DeleteTemplate() => replyOrFail {
      actorServiceHolder.releaseService.deleteTemplate(releaseId)
      releaseSupervisor ! ReleaseDeleted
    }

    case UpdateTemplate(template) => replyOrFail {
      actorServiceHolder.releaseService.updateTemplate(releaseId, template)
    }

    case CreateReleaseFromTrigger(folderId, releaseTitle, releaseVariables, releaseTags, autoStart, scheduledStartDate, triggerId) =>
      replyOrFail[CreateReleaseResult] {
        // Check if release can be triggered from inside actor
        actorServiceHolder.releaseService.getTriggerReleaseBlocker(release) match {
          case Some(reason) => ReleaseNotCreated(reason)
          case None =>
            val variableValues = releaseVariables.map(v => v.getKey -> v.getValue).toMap.asJava

            ReleaseCreated(actorServiceHolder.releaseService.createFromTemplate(
              release,
              folderId,
              releaseTitle,
              null,
              variableValues,
              releaseTags.asJava,
              autoStart,
              scheduledStartDate.orNull,
              triggerId,
              null
            ))
        }
      }

    case CreateLink(containerId, sourceId, targetId) => replyOrFail {
      actorServiceHolder.linkService.create(containerId, sourceId, targetId)
    }

    case DeleteLink(linkId) => replyOrFail {
      actorServiceHolder.linkService.delete(linkId)
    }

    case CreateVariable(variable) => replyOrFail {
      if (release.getVariablesByKeys.containsKey(variable.getKey)) {
        throw new ItemConflictException("A variable already exists by key [%s] in [%s]", variable.getKey, release.getId)
      }
      actorServiceHolder.variableService.addVariable(release, variable)
    }

    case UpdateVariable(variableId, variable) => replyOrFail {
      log.debug("UpdateVariable {}", variableId)
      variable.setId(variableId)
      actorServiceHolder.variableService.updateVariable(release, variable)
    }

    case RenameVariable(variableId, newVariable) => replyOrFail {
      log.debug("RenameVariable {} to {}", variableId, newVariable.getKey)
      if (release.getVariablesByKeys.containsKey(newVariable.getKey)) {
        throw new ItemConflictException("A variable already exists by key [%s] in [%s]", newVariable.getKey, release.getId)
      }
      actorServiceHolder.variableService.updateVariable(release, newVariable)
    }

    case DeleteVariable(variableId) => replyOrFail {
      log.debug("DeleteVariable {}", variableId)
      actorServiceHolder.variableService.deleteVariable(release, variableId)
    }

    case ReplaceVariable(variable, replacement) => replyOrFail {
      log.debug("ReplaceVariable {} with {}", variable.getKey, replacement)
      actorServiceHolder.variableService.replaceVariable(release, variable, replacement)
    }

    case StatusUpdated(status) => logException {
      if (cachedReleaseStatus.orNull != status && status != null) {
        log.debug("StatusUpdated to {}", status)
        cachedReleaseStatus = Option(status)
      }
    }

    case ReleaseOverdue() => logException {
      log.debug(s"Notify overdue for release ${release.getId}")
      actorServiceHolder.releaseService.notifyOverdueRelease(release)
    }

    case TasksOverdue(taskIds) => logException {
      actorServiceHolder.taskService.notifyOverdueTasks(release, taskIds.asJava)
    }

    case TasksDueSoon(taskIds) => logException {
      actorServiceHolder.taskService.notifyTasksDueSoon(release, taskIds.asJava)
    }

    case UpdateRiskScores(releaseId, score, totalScore) => logException {
      log.debug(s"Updating cached risk scores for release ${release.getId}")
      actorServiceHolder.eventBus.publish(ReleaseRiskScoreUpdated(releaseId, score, totalScore))
    }
  }

  // scalastyle:on cyclomatic.complexity method.length

  def handleUnknownMessage: Receive = {
    case a@_ =>
      throw new IllegalStateException(s"Don't know how to process $a")
  }

  actorServiceHolder.actorExtensionRegistry.getHandlerFactoriesForActorType(classOf[ReleaseExecutionActor]).foreach { factory =>
    handleExtensionMessagesBuilder += factory.getHandler(self, () => sender(), release)
  }

  override def receive: Receive = {
    doCount(handleBackpressure.orElse(garbageCollect.orElse(withCallerContext(handleDefaultMessages.orElse(handleExtensionMessagesBuilder.result()).orElse(handleUnknownMessage)))))
  }

  private def doCount(delegate: Receive): Receive = {
    case msg: Any if delegate.isDefinedAt(msg) =>
      countMessage(msg)
      delegate(msg)
    case x => delegate(x)
  }

  private def countMessage(msg: Any): Unit = {
    try {
      val fullClassName = msg.getClass.getName
      val shortClassName = fullClassName.substring(fullClassName.lastIndexOf(".") + 1)
      val internalClassName = shortClassName.substring(shortClassName.lastIndexOf("$") + 1)
      val metricName = if (internalClassName.isBlank) {
        shortClassName
      } else {
        internalClassName
      }
      actorServiceHolder.meterRegistry.counter(s"$METRIC_PREFIX$metricName").increment()
    } catch {
      case _: Throwable =>
        actorServiceHolder.meterRegistry.counter(s"${METRIC_PREFIX}UNKNOWN").increment()
    }
  }

  private def handleBackpressure: Receive = {
    case BackpressureWait(taskId) =>
      log.debug(s"Got BackpressureWait for $taskId, replying")
      sender() ! BackpressureResponse(releaseId)
  }

  def garbageCollect: Receive = {
    case _@GcRequest =>
      val gcKeepAlive = shouldKeepAlive()
      log.debug(s"GC: keepAlive = $gcKeepAlive for $releaseId")
      if (gcKeepAlive) {
        sender() ! ReleaseIsActive
      } else {
        sender() ! ReleasePoisonPill
        context.become(bouncing)
      }
  }

  private def shouldKeepAlive(): Boolean = {
    Try {
      ReleaseActorCache.get(releaseId)(() => {
        val resolveOptions = ResolveOptions.WITHOUT_DECORATORS.withReferences
        actorServiceHolder.releaseService.findByIdIncludingArchived(releaseId, resolveOptions)
      })
    } match {
      case Success(release) =>
        lazy val releaseHasNonCollectibleActiveTasks = release.getAllTasks.asScala.exists(
          inProgressAutomatedTask or
            preconditionInProgressTask or
            facetCheckInProgressTask or
            failureHandlerInProgressTask
        )
        release.isPending || (release.isPlannedOrActive && releaseHasNonCollectibleActiveTasks)
      case Failure(exception: NotFoundException) =>
        log.debug(s"GC: ${exception.getMessage}, forcing actor shutdown")
        false
      case Failure(exception) =>
        if (log.isDebugEnabled) {
          log.warning(exception, s"GC: ${exception.getMessage}")
        }
        log.warning(s"Unable to find release [$releaseId] for GC, forcing actor shutdown")
        false
    }
  }

  def bouncing: Receive = {
    case GcRequest => ()
    case msg: AnyRef =>
      log.debug("Bouncing message after GC happened: {}, {}", releaseId, msg)
      releaseSupervisor.tell(ReleaseAction(releaseId, msg), sender())
    case msg =>
      log.error("Unable to bounce message after GC happened: {}, {}", releaseId, msg)
  }

  private def replyOrFail[T](call: => T): Unit = sender() ! (Try(call) match {
    case Success(t) if t != null =>
      logger.debug(s"About to reply with $t")
      t
    case Success(_) =>
      Status.Failure(new NullPointerException("Method returned null and this cannot be processed"))
    case Failure(ex) =>
      logger.warn("Failed to process release actor message", ex)
      val translatedException = translate(ex)
      Status.Failure(translatedException)
  })


  private def logException[T](call: => T): Unit = Try(call) match {
    case Success(_) => ()
    case Failure(ex) => ()
      logger.warn(s"Failed to process release actor message for release $releaseId with exception ${ex.getClass.getCanonicalName}", ex)
  }

  private def withCallerContext(delegate: Receive): Receive = {
    case cmd: Command =>
      val maybeAuthentication = cmd.callerContext.authentication
      log.debug("Setting caller as {} for command {}", maybeAuthentication.map(_.getName), cmd)
      maybeAuthentication.foreach(SecurityContextHolder.getContext.setAuthentication)
      try {
        if (cmd.invalidatesCache) {
          log.debug("Invalidating cache entry '{}' before command: {}", releaseId, cmd)
          ReleaseActorCache.remove(releaseId)
        }
        delegate(cmd)
      } finally {
        if (cmd.invalidatesCache) {
          log.debug("Invalidating cache entry '{}' after command: {}", releaseId, cmd)
          ReleaseActorCache.remove(releaseId)
        }
        SecurityContextHolder.clearContext()
      }
    case msg@_ =>
      delegate(msg)
  }
}


sealed trait CreateReleaseResult

case class ReleaseCreated(release: Release) extends CreateReleaseResult

case class ReleaseNotCreated(msg: String) extends CreateReleaseResult
