package com.xebialabs.xlrelease.service

import com.google.common.annotations.VisibleForTesting
import com.google.common.base.Preconditions.{checkArgument, checkState}
import com.xebialabs.deployit.io.StreamWrappingOverthereFile
import com.xebialabs.deployit.plugin.api.reflect.{PropertyDescriptor, PropertyKind}
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem
import com.xebialabs.xlrelease.actors.ReleaseActorService
import com.xebialabs.xlrelease.api.internal.InternalMetadataDecoratorService
import com.xebialabs.xlrelease.api.internal.ReleaseGlobalAndFolderVariablesDecorator.GLOBAL_AND_FOLDER_VARIABLES
import com.xebialabs.xlrelease.api.internal.ReleaseServerUrlDecorator.SERVER_URL
import com.xebialabs.xlrelease.config.XlrConfig
import com.xebialabs.xlrelease.domain.FailureReasons._
import com.xebialabs.xlrelease.domain._
import com.xebialabs.xlrelease.domain.blackout.BlackoutMetadata.BLACKOUT
import com.xebialabs.xlrelease.domain.status.TaskStatus.{COMPLETED, COMPLETED_IN_ADVANCE, SKIPPED, SKIPPED_IN_ADVANCE}
import com.xebialabs.xlrelease.domain.status.{ReleaseStatus, TaskStatus}
import com.xebialabs.xlrelease.domain.tasks.TaskExecutor
import com.xebialabs.xlrelease.domain.utils.ReleaseCloneHelper
import com.xebialabs.xlrelease.domain.variables.Variable
import com.xebialabs.xlrelease.events.{ReleaseVariablesUpdateOperation, XLReleaseEventBus, XLReleaseOperations}
import com.xebialabs.xlrelease.repository.{CiCloneHelper, Ids, ReleaseRepository, TaskRepository}
import com.xebialabs.xlrelease.scheduler._
import com.xebialabs.xlrelease.script._
import com.xebialabs.xlrelease.user.User
import com.xebialabs.xlrelease.user.User.{AUTHENTICATED_USER, LOG_OUTPUT, SYSTEM}
import com.xebialabs.xlrelease.utils.{PythonScriptCiHelper, SensitiveValueScrubber}
import com.xebialabs.xlrelease.variable.VariableHelper
import com.xebialabs.xlrelease.variable.VariableHelper.containsOnlyVariable
import com.xebialabs.xlrelease.variable.VariablePersistenceHelper.fixUpVariableIds
import grizzled.slf4j.{Logger, Logging}
import io.micrometer.core.annotation.Timed
import org.springframework.stereotype.Service

import java.io.ByteArrayInputStream
import java.nio.charset.StandardCharsets.UTF_8
import java.text.SimpleDateFormat
import java.util
import java.util.Arrays.asList
import java.util.Date
import java.util.concurrent.TimeUnit
import scala.collection.Seq
import scala.concurrent.duration.Duration.apply
import scala.jdk.CollectionConverters._


// scalastyle:off number.of.methods
@Service
class ScalaExecutionService(
                             releaseRepository: ReleaseRepository,
                             val taskRepository: TaskRepository,
                             dependencyService: DependencyService,
                             scriptLifeCycle: ScriptLifeCycle,
                             taskBackup: TaskBackup,
                             releaseActorService: ReleaseActorService,
                             eventBus: XLReleaseEventBus,
                             taskExecutors: util.List[_ <: TaskExecutor[_ <: Task]],
                             decoratorService: InternalMetadataDecoratorService,
                             teamService: TeamService,
                             changeExecutionService: ChangeExecutionService,
                             scriptResultsService: ScriptResultsService,
                             workManager: WorkManager,
                             releaseService: ReleaseService,
                             commentService: CommentService,
                             val ciIdService: CiIdService
                           ) extends ExecutionService with Logging with ContainerTaskResultChanges {

  private val taskExecutorsPerType: Map[Class[_], TaskExecutor[_]] = taskExecutors.asScala.map(executor => executor.getTaskClass -> executor).toMap

  @Timed
  override def start(release: Release, user: User, releaseStartedImmediatelyAfterBeingCreated: Boolean, isPartOfBulkOperation: Boolean): Release = {
    if (release.hasBeenStarted) {
      logger.info(s"Will not start release, because it is already started, probably due to duplicate message delivery ${release.getId}")
      release
    } else {
      decorateWithMetadata(release)
      val changes = if (isPartOfBulkOperation) {
        release.startAsPartOfBulkOperation()
      } else {
        release.start(releaseStartedImmediatelyAfterBeingCreated)
      }
      processChangesAndPublishOperations(changes, user)
      release
    }
  }

  @Timed
  override def resume(releaseId: String): Release = {
    val release = releaseRepository.findById(releaseId)
    checkState(release.isPaused, "Only paused releases may be restarted. Release '%s' is %s.", release.getTitle, release.getStatus)
    decorateWithMetadata(release)
    val changes = release.resume
    processChangesAndPublishOperations(changes, AUTHENTICATED_USER)
    release
  }

  @Timed
  override def abort(releaseId: String, abortComment: String, isPartOfBulkOperation: Boolean): Release = {
    val release = releaseRepository.findById(releaseId)
    checkState(release.hasNoAutomatedTaskRunning, "The release %s could not be aborted because it has automated tasks that are in progress.", release.getTitle)
    checkState(release.getStatus ne ReleaseStatus.COMPLETED, "You can not abort completed release.", release.getTitle)
    teamService.decorateWithStoredTeams(release)
    decorateWithMetadata(release)
    val changes = if (isPartOfBulkOperation) {
      release.abortAsPartOfBulkOperation(abortComment)
    } else {
      release.abort(abortComment)
    }
    publishBeforeEvents(changes)
    applyChanges(changes, AUTHENTICATED_USER)
    publishAfterEvents(changes)
    failDependentGates(changes, release.getTitle)
    release
  }


  @Timed
  override def startPendingTask(taskId: String, release: Release, comment: String, user: User): Unit = {
    val shouldConsiderBlackoutPeriods = User.SYSTEM == user
    if (shouldConsiderBlackoutPeriods) {
      decorateWithMetadata(release)
    }
    val changes = release.startPendingTask(taskId)
    changes.addComment(release.getTask(taskId), comment)
    processChangesAndPublishOperations(changes, user)
  }

  @Timed
  override def postponeUntilEnvironmentsAreReserved(taskId: String, postponeUntil: Date, comment: String, executionId: String): Unit = {
    val task = taskRepository.findById[Task](taskId)
    if (task.isStillExecutingScript(executionId)) {
      task.setExecutionId(null)
      val changes = task.postponeUntilEnvironmentsAreReserved(postponeUntil)
      changes.update(task)
      processChangesAndPublishOperations(changes, LOG_OUTPUT)
    } else {
      logger.info(s"Will not postpone task until environments are reserved, because execution id does not match: $taskId")
    }
  }

  @Timed
  override def fail(task: Task,
                    addedComment: String,
                    baseScriptTaskResults: Option[DefaultScriptService.BaseScriptTaskResults],
                    user: User,
                    executionId: Option[String]
                   ): Unit = {
    if (executionId.isDefined && // this message is sent using "at least once" delivery, and can be duplicate
      !task.isStillExecutingScript(executionId.get)) {
      logger.info(s"Skipping duplicate FailTask message for task ${task.getId}")
    } else {
      fail(new Changes, task, addedComment, None, baseScriptTaskResults, user, fromAbort = false)
    }
  }

  @Timed
  override def fail(task: Task, addedComment: String, user: User): Unit = {
    // this one is not used to fail script tasks as those can only be aborted
    fail(new Changes(), task, addedComment, None, None, user, fromAbort = false)
  }

  // this is called when scriptTask/customScriptTask fail with exception, we need User SYSTEM as user so error will be attached in comment
  @Timed
  override def scriptFailed(task: Task,
                            scriptFailMessage: String,
                            scriptExecutionId: String,
                            logArtifactId: Option[String],
                            scriptResults: Option[DefaultScriptService.BaseScriptTaskResults],
                            user: User): Unit = {
    val changes = new Changes
    if (task.isStillExecutingScript(scriptExecutionId)) { // we only want to capture results if this message contains correct executionId
      // Otherwise, since we retry Actor message delivery, this message may appear to be an
      // obsolete one, and we should completely ignore it
      scriptResults.foreach(s => changes.addAll(scriptResultsService.resolveScriptTaskResults(task, s)))
      fail(changes, task, SCRIPT_TASK_FAILED.format(scriptFailMessage), logArtifactId, None, user, fromAbort = false)
    }
    else {
      logger.debug(s"Will not fail task: '${task.getId}', it has been aborted. " +
        s"Task status is ${task.getStatus} and task executionId is ${task.getExecutionId}, expected executionId is $scriptExecutionId"
      )
      // we want to add as an attachment whatever was in the execution log
      attachScriptOutput(changes, task, scriptFailMessage, logArtifactId)
      processChangesAndPublishOperations(changes, User.LOG_OUTPUT)
    }
  }

  @Timed
  override def saveCustomScriptResults(release: Release,
                                       taskId: String,
                                       results: DefaultScriptService.CustomScriptTaskResults,
                                       executionId: String): Unit = {
    val task = release.getTask(taskId)
    task match {
      case customScriptTask: CustomScriptTask =>
        // since we have "at least once message delivery" for this message this message
        // is likely a duplicate
        val taskExecutionId = customScriptTask.getExecutionId
        val taskStatus = customScriptTask.getStatus
        if (!customScriptTask.isStillExecutingScript(executionId)) {
          // this message is considered duplicate
          logger.info(s"Discarding duplicate SaveCustomScriptResults message because executionId ($executionId) " +
            s"does not match task $taskId executionId ($taskExecutionId) or task status ($taskStatus) is wrong"
          )
        } else {
          val changes = scriptResultsService.resolveScriptTaskResults(customScriptTask, results)
          processChangesAndPublishOperations(changes, AUTHENTICATED_USER)
        }
      case _ =>
        logger.info(s"Discarding SaveCustomScriptResults message because task appeared to be not CustomScriptTask any more ${task.getId}")
    }
  }


  @Timed
  override def resumeTask(taskId: String): Unit = {
    executeTask(taskRepository.findById[Task](taskId))
  }

  @Timed
  override def abortTask(task: Task, addedComment: String): Unit = {
    workManager.abortJobByTaskId(Ids.getFolderlessId(task.getId))
    scriptLifeCycle.tryAborting(task.getExecutionId)
    val comment = if (task.isAbortScriptInProgress) {
      ABORT_SCRIPT_TASK_ABORTED.format(addedComment)
    } else {
      SCRIPT_TASK_ABORTED.format(addedComment)
    }
    fail(new Changes, task, comment, None, None, AUTHENTICATED_USER, fromAbort = true)
  }

  @Timed
  override def skipTaskDueToPreconditionCheck(release: Release, taskId: String, executionId: String, executionLog: String): Unit = {
    val task = release.getTask(taskId)
    if (task.isStillExecutingScript(executionId)) {
      markTaskAsDone(release, TaskStatus.SKIPPED, taskId, executionLog, User.LOG_OUTPUT)
    } else {
      logger.info(s"Will not skip task due to precondition check because executionId does not match: ${taskId}")
    }
  }

  @Timed
  override def finishCustomScriptTask(release: Release,
                                      taskId: String,
                                      comment: String,
                                      executionId: String,
                                      logArtifactId: Option[String],
                                      scriptResults: Option[DefaultScriptService.BaseScriptTaskResults]): Unit = {
    val task = release.getTask(taskId).asInstanceOf[CustomScriptTask]
    val changes = new Changes()
    if (task.isStillExecutingScript(executionId)) {
      scriptResults.foreach(s => changes.addAll(scriptResultsService.resolveScriptTaskResults(task, s)))
      if (task.isWaitingForSignal) {
        logger.info(s"Will not complete task: '$taskId', staying in progress until task condition is met.")
      } else {
        if (task.isAbortScriptInProgress) {
          fail(changes, task, "Abort script finished.", logArtifactId, None, SYSTEM, fromAbort = false)
        } else {
          markTaskAsDone(changes, release, COMPLETED, taskId, comment, logArtifactId, None, LOG_OUTPUT)
        }
      }
    } else {
      logger.debug(s"Will not complete task: '$taskId', it has been aborted.")
    }
  }

  @Timed
  override def finishScriptTask(release: Release,
                                taskId: String,
                                comment: String,
                                executionId: String,
                                logArtifactId: Option[String],
                                scriptResults: Option[DefaultScriptService.BaseScriptTaskResults]): Unit = {
    val task = release.getTask(taskId)
    val changes = new Changes()
    if (task.isStillExecutingScript(executionId)) {
      scriptResults.foreach(sr => changes.addAll(scriptResultsService.resolveScriptTaskResults(task, sr)))
      markTaskAsDone(changes, release, COMPLETED, taskId, comment, logArtifactId, None, User.LOG_OUTPUT)
    } else {
      logger.debug(s"Will not complete task: '$taskId', executionId does not match, it has been aborted.")
    }
  }

  @Timed
  override def finishContainerTask(release: Release,
                                   taskId: String,
                                   taskResult: ContainerTaskResult): Unit = {
    val task = release.getTask(taskId).asInstanceOf[ContainerTask]
    val fullTaskId = task.getId
    val changes = new Changes
    changes.addAll(processContainerTaskResult(task, taskResult))
    if (taskResult.targetStatus.isOneOf(COMPLETED, SKIPPED, COMPLETED_IN_ADVANCE, SKIPPED_IN_ADVANCE)) {
      changes.addComment(task, taskResult.comment)
      changes.addAll(release.markTaskAsDone(fullTaskId, taskResult.targetStatus))
    } else {
      changes.addAll(release.failTask(fullTaskId, taskResult.comment))
    }
    // TODO can this task be completed in advance?
    // TODO can this task be scheduled?
    // TODO can this task wait for an environment?
    processChangesAndPublishOperations(changes, SYSTEM)
  }

  @Timed
  override def taskPreconditionValidated(taskId: String, release: Release, executionId: String): Unit = {
    // removed decorate

    val task = release.getTask(taskId)
    val changes = new Changes()
    if (task.isStillExecutingScript(executionId)) {
      task.setExecutionId(null)
      changes.update(task)
      changes.addAll(release.taskPreconditionValidated(taskId))
      processChangesAndPublishOperations(changes, LOG_OUTPUT)
    } else {
      logger.debug("Will not accept precondition validation results because executionId does not match.")
    }
  }

  @Timed
  override def markTaskAsDone(release: Release, targetStatus: TaskStatus, taskId: String, addedComment: String, user: User): Unit = {
    markTaskAsDone(release, targetStatus, taskId, addedComment, None, None, user)
  }

  @Timed
  override def markTaskAsDone(release: Release,
                              targetStatus: TaskStatus,
                              taskId: String,
                              addedComment: String,
                              logArtifactId: Option[String],
                              scriptResults: Option[DefaultScriptService.BaseScriptTaskResults],
                              user: User): Unit = {
    markTaskAsDone(new Changes, release, targetStatus, taskId, addedComment, logArtifactId, scriptResults, user)
  }

  @Timed
  override def markTaskAsDone(release: Release,
                              targetStatus: TaskStatus,
                              taskId: String,
                              addedComment: String,
                              logArtifactId: Option[String],
                              scriptResults: Option[DefaultScriptService.BaseScriptTaskResults],
                              user: User,
                              executionId: String): Boolean = {
    val task = release.getTask(taskId)
    if (task.isStillExecutingScript(executionId)) {
      markTaskAsDone(release, targetStatus, taskId, addedComment, logArtifactId, scriptResults, user)
    } else {
      logger.info(s"Skipping duplicate message MarkTaskAsDone for $taskId")
    }
    true
  }

  private def markTaskAsDone(changes: Changes,
                             release: Release,
                             targetStatus: TaskStatus,
                             taskId: String,
                             addedComment: String,
                             logArtifactId: Option[String],
                             scriptResults: Option[DefaultScriptService.BaseScriptTaskResults],
                             user: User): Unit = {
    decorateWithMetadata(release)
    val task = release.getTask(taskId)
    scriptResults.foreach(baseScriptTaskResults => changes.addAll(scriptResultsService.resolveScriptTaskResults(task, baseScriptTaskResults)))
    if (LOG_OUTPUT == user) {
      attachScriptOutput(changes, task, addedComment, logArtifactId)
    } else {
      changes.addComment(task, addedComment)
    }
    if (targetStatus.isDoneInAdvance) {
      changes.addAll(task.markAsDone(taskId, targetStatus))
    } else {
      changes.addAll(release.markTaskAsDone(taskId, targetStatus))
    }
    processChangesAndPublishOperations(changes, user)
  }

  private def executePostAction(action: PostAction): Unit = action match {
    case ExecuteFacetAction(task) => executeFacetCheck(ReleaseCloneHelper.clone(task))
    case ExecutePreconditionAction(task) => executePrecondition(ReleaseCloneHelper.clone(task))
    case ExecuteTaskAction(task) => executeTask(ReleaseCloneHelper.clone(task))
    case ExecuteFailureHandlerAction(task) => executeFailureHandler(ReleaseCloneHelper.clone(task))
    case ExecuteAbortScriptAction(task) => executeAbortScript(ReleaseCloneHelper.clone(task))
    case ExecuteNextCustomScriptPath(task) => executeNextCustomScriptPath(ReleaseCloneHelper.clone(task))
  }

  private def executeFacetCheck(task: Task): Unit = {
    try {
      workManager.submit(FacetCheckJob(taskRef(task)))
    } catch {
      case exception: Exception =>
        val msg = s"Unable to schedule facet check. Reason: ${exception.getMessage}"
        releaseActorService.failTaskAsync(task.getId, msg, SYSTEM, Option.empty)
    }
  }

  private def executePrecondition(task: Task): Unit = {
    try {
      workManager.submit(PreconditionJob(taskRef(task)))
    } catch {
      case exception: Exception =>
        val msg = s"Unable to schedule precondition check. Reason: ${exception.getMessage}"
        releaseActorService.failTaskAsync(task.getId, msg, SYSTEM, Option.empty)
    }
  }

  private def executeTask(task: Task): Unit = {
    try {
      val taskExecutor = taskExecutorsPerType.get(task.getClass).orNull.asInstanceOf[TaskExecutor[Task]]
      checkArgument(taskExecutor != null, "Cannot execute task because there is no executor defined for task type '%s'", task.getType.toString)
      taskExecutor.execute(taskRef(task))
    } catch {
      case exception: Exception =>
        logger.error(s"Unable to schedule task execution for task ${task.getId}", exception)
        val msg = s"Unable to schedule task execution. Reason: ${exception.getMessage}"
        releaseActorService.failTaskAsync(task.getId, msg, SYSTEM, Option.empty)
    }
  }

  private def executeFailureHandler(task: Task): Unit = {
    try {
      workManager.submit(FailureHandlerJob(taskRef(task)))
    } catch {
      case exception: Exception =>
        val msg = s"Unable to schedule failure handler. Reason: ${exception.getMessage}"
        releaseActorService.failTaskAsync(task.getId, msg, SYSTEM, Option.empty)
    }
  }

  private def executeAbortScript(task: Task): Unit = {
    checkArgument(task.isInstanceOf[CustomScriptTask] || task.isInstanceOf[ContainerTask], "Cannot execute abort script because task is not an instance of CustomScriptTask or ContainerTask", task.getType.toString)
    executeTask(task)
  }

  private def executeNextCustomScriptPath(task: CustomScriptTask): Unit = {
    logger.trace(s"Scheduling next script path: ${task.getNextScriptPath}")
    scheduleNextCustomScriptTaskExecution(task)
  }

  private def scheduleNextCustomScriptTaskExecution(customScriptTask: CustomScriptTask): Unit = {
    val delay = if (customScriptTask.getInterval != null) {
      customScriptTask.getInterval * 1000
    } else {
      XlrConfig.getInstance.durations_customScriptTaskScheduleInterval.toMillis
    }
    workManager.replace(NextCustomScriptTaskJob(taskRef(customScriptTask), apply(delay, TimeUnit.MILLISECONDS)))
  }

  private def taskRef[T <: Task](task: T): TaskSoftReference[T] = {
    val ref = TaskSoftReferenceFactory.taskRef(releaseService, task)

    task match {
      case ct: ContainerTask => ref.setCapabilities(ct.getCapabilities())
      case _ =>
    }

    ref
  }

  protected def decorateWithMetadata(release: Release): Unit = {
    // in com.xebialabs.xlrelease.actors.ReleaseExecutionActor.release we additionally decorate with teams and effective security
    decoratorService.decorate(release, asList(BLACKOUT, GLOBAL_AND_FOLDER_VARIABLES, SERVER_URL))
  }

  protected def processChangesAndPublishOperations(changes: Changes, user: User): Unit = {
    publishBeforeEvents(changes)
    processChanges(changes, user)
    publishAfterEvents(changes)
    completeDependentGates(changes)
  }

  protected def publishBeforeEvents(changes: Changes): Unit = {
    XLReleaseOperations.runActionInterceptors(changes.getOperations)(eventBus)
  }

  protected def processChanges(changes: Changes, user: User): Unit = {
    taskBackup.backupTasks(changes.getTasksToBackup)
    applyChanges(changes, user)
    executePostActions(changes)
  }

  protected def publishAfterEvents(changes: Changes): Unit = {
    XLReleaseOperations.publishEvents(changes.getOperations)(eventBus)
  }

  protected def applyChanges(changes: Changes, user: User): Unit = {
    changeExecutionService.applyChanges(changes, user)
  }

  @VisibleForTesting
  private[service] def executePostActions(changes: Changes): Unit = {
    changes.getPostActions.forEach(this.executePostAction)
  }

  private def completeDependentGates(changes: Changes): Unit = {
    val updatedItems = changes.getUpdatedItems.asScala.toSet
    val completableGateIds: Seq[String] = dependencyService.getCompletableGateIds(toPlanItems(updatedItems))
    logger.trace(s"Completable gate ids: ${completableGateIds.mkString(", ")}")
    completableGateIds.foreach(releaseActorService.markTaskAsDoneAsync(TaskStatus.COMPLETED, _, null, SYSTEM))
  }

  private def toPlanItems(updatedItems: Set[ConfigurationItem]): Set[PlanItem] = updatedItems.collect {
    case item: PlanItem => item
  }

  private def fail(changes: Changes,
                   task: Task,
                   addedComment: String,
                   logArtifactId: Option[String],
                   baseScriptTaskResults: Option[DefaultScriptService.BaseScriptTaskResults],
                   user: User,
                   fromAbort: Boolean): Unit = {
    val release = task.getRelease
    baseScriptTaskResults.foreach(s => changes.addAll(scriptResultsService.resolveScriptTaskResults(task, s)))
    if (LOG_OUTPUT == user) {
      changes.addAll(release.failTask(task.getId, "", fromAbort))
      attachScriptOutput(changes, task, addedComment, logArtifactId)
    } else {
      changes.addAll(release.failTask(task.getId, addedComment, user, fromAbort))
    }
    publishBeforeEvents(changes)
    processChanges(changes, user)
    publishAfterEvents(changes)
    if (release.isAbortOnFailure && release.isAborted) {
      failDependentGates(changes, release.getTitle)
    }
  }


  private def attachScriptOutput(changes: Changes, task: Task, addedComment: String, logArtifactId: Option[String]): Unit = {
    logArtifactId match {
      case Some(id) =>
        changes.linkScriptOutputLog(task, id)
      case None =>
        val content = addedComment.getBytes(UTF_8)
        if (content.nonEmpty) {
          val filename = new SimpleDateFormat("'script_output_'yyyyMMddHHmmss'.log'").format(new Date)
          val attachment = new Attachment(new StreamWrappingOverthereFile(filename, new ByteArrayInputStream(content)), "text/plain")
          changes.addAttachment(task, attachment)
        }
    }
  }

  private def failDependentGates(changes: Changes, releaseTitle: String): Unit = {
    val updatedItems = changes.getUpdatedItems.asScala.toSet
    val failableGateIds = dependencyService.getFailableGateIds(toPlanItems(updatedItems))
    for (gateId <- failableGateIds) {
      releaseActorService.failTaskAsync(gateId, GATE_TASK_DEPENDS_ON_AN_ABORTED_RELEASE.format(releaseTitle), SYSTEM, Option.empty)
    }
  }

  @Timed
  override def reopenTask(task: Task, addedComment: String): Unit = {
    val changes = task.reopen
    changes.addComment(task, addedComment)
    publishBeforeEvents(changes)
    applyChanges(changes, AUTHENTICATED_USER)
    publishAfterEvents(changes)
  }

  @Timed
  override def retry(task: Task, addedComment: String): Unit = {
    val release = task.getRelease
    decorateWithMetadata(release) // ???
    val changes = release.retryTask(task.getId)
    changes.addComment(task, addedComment)
    processChangesAndPublishOperations(changes, AUTHENTICATED_USER)
  }

  @Timed
  override def startWithInput(taskId: String, variables: util.List[Variable]): Task = {
    val task = taskRepository.findById[Task](taskId)
    val updated = updateVariables(variables, task)
    val release = updated.getRelease
    decorateWithMetadata(release)
    processChangesAndPublishOperations(release.startWithInput(taskId), AUTHENTICATED_USER)
    task
  }

  private def updateVariables(variables: util.List[Variable], task: Task): Task = {
    val inputVariablesByKey = VariableHelper.indexByKey(variables)
    task.getInputVariables.stream
      .filter(variable => inputVariablesByKey.containsKey(variable.getKey))
      .forEach(variable => variable.setUntypedValue(inputVariablesByKey.get(variable.getKey).getValue))
    // why do we update task here - it should be part of the changes executed - was REL-5387
    taskRepository.update(task)
  }

}

//TODO: Refactor this
trait ContainerTaskResultChanges {

  protected def logger: Logger

  protected def ciIdService: CiIdService

  def processContainerTaskResult(task: ContainerTask, taskResult: ContainerTaskResult): Changes = {
    // for ideas what to set take a look at ScriptResultsService.customScriptTask: next script path, status line, interval
    val changes = new Changes()
    // set properties on a task, update variables, etc
    // add post action if needed (execute next custom script path)
    // if we need transitional properties (as used by custom script tasks) handle it here
    setOutputPropertiesAndVariables(task, taskResult.outputProperties)
    changes.update(task)
    changes
  }

  private def setOutputPropertiesAndVariables(task: ContainerTask, outputAttributes: util.Map[String, AnyRef]): Changes = {
    // logic from ScriptResultsService.setOutputPropertiesAndVariables
    val changes = new Changes
    val outputProperties = task.getOutputProperties()
    val outputValues = ScriptServiceHelper.extractPropertyValues(task.getId, outputProperties.asJava, outputAttributes, SensitiveValueScrubber.disabled())

    val newVariableValues = new util.HashMap[String, AnyRef]
    val newPasswordVariableValues = new util.HashMap[String, AnyRef]
    for (propertyDescriptor <- outputProperties) {
      val propertyName = propertyDescriptor.getName
      val fqPropertyName = propertyName
      val propertyValue = task.getProperty[AnyRef](propertyName)
      // In pre-5.0 a string output property value contains a variable name to put the resulting value to
      val oldVariableName: Option[String] = propertyValue match {
        case strValue: String if containsOnlyVariable(strValue) => Some(strValue)
        case _ => None
      }
      // In 5.0+ mapping from output properties to target variables is stored in a separate property: variableMapping
      val newVariableName = if (task.getVariableMapping.containsKey(fqPropertyName)) {
        val variableName = task.getVariableMapping.get(fqPropertyName)
        task.getVariableMapping.remove(fqPropertyName) // if (!task.hasNextScriptToExecute)
        Some(variableName)
      } else {
        None
      }
      var outputValue = outputValues.get(propertyName)

      // see: com.xebialabs.xlrelease.service.ScriptResultsService.setOutputPropertiesAndVariables
      if (task.isKeepPreviousOutputPropertiesOnRetry() && !PythonScriptCiHelper.isEmpty(task.getProperty(propertyName))) {
        outputValue = task.getProperty(propertyName)
        if (propertyDescriptor.getKind() == PropertyKind.STRING) {
          outputValue = VariableHelper.withoutVariableSyntax(outputValue.asInstanceOf[String])
        }
      }

      // TODO decide if outputValue could be truncated like it is truncated for CustomScriptTask
      if (outputValue != null) {
        task.setProperty(propertyName, outputValue)
        changes.update(task)
        if (oldVariableName.isDefined || newVariableName.isDefined) {
          val variableName = oldVariableName.orElse(newVariableName).get
          logger.debug(s"Output property $fqPropertyName of ${task.getId} is mapped to $variableName variable")
          if (propertyDescriptor.isPassword) {
            newPasswordVariableValues.put(variableName, outputValue)
          } else {
            newVariableValues.put(variableName, outputValue)
          }
        }
      }
    }
    // TODO check if we can get rid of this somehow - users have releases with quite a lot of tasks and a lot of variables we don't want to copy
    val release = task.getRelease
    val oldVariables = CiCloneHelper.cloneCis(release.getVariables)
    release.setVariableValues(newVariableValues)
    release.setPasswordVariableValues(newPasswordVariableValues)
    fixUpVariableIds(release.getId, release.getVariables, ciIdService) // TODO this should not happen for existing variables
    // let's add the operation, maybe nothing is changed but that will be calculated by VariablesDiff
    changes.addOperation(ReleaseVariablesUpdateOperation(oldVariables, release.getVariables, logDelete = true))

    changes
  }

  private implicit class TaskOps(task: ContainerTask) {
    implicit def getOutputProperties(): Seq[PropertyDescriptor] = {
      val desc = task.getType.getDescriptor
      desc.getPropertyDescriptors.asScala.filter(_.getCategory == Task.CATEGORY_OUTPUT).toSeq
    }
  }

}
