package org.springframework.data.domain;

import java.io.Serial;
import java.util.List;
import java.util.function.Function;
import org.springframework.lang.Nullable;

/**
 * This is a copy of {@link org.springframework.data.domain.PageImpl} with a change in constructor.
 * This keeps total value as it is instead of modifying it.
 */
public class ReleasePageImpl<T> extends Chunk<T> implements Page<T> {
    @Serial
    private static final long serialVersionUID = -6969139863724462053L;
    private final long total;

    public ReleasePageImpl(List<T> content, Pageable pageable, long total) {
        super(content, pageable);
        this.total = total;
    }

    public ReleasePageImpl(List<T> content) {
        this(content, Pageable.unpaged(), content.size());
    }

    public int getTotalPages() {
        return this.getSize() == 0 ? 1 : (int) Math.ceil((double) this.total / (double) this.getSize());
    }

    public long getTotalElements() {
        return this.total;
    }

    public boolean hasNext() {
        return this.getNumber() + 1 < this.getTotalPages();
    }

    public boolean isLast() {
        return !this.hasNext();
    }

    public <U> Page<U> map(Function<? super T, ? extends U> converter) {
        return new ReleasePageImpl<>(this.getConvertedContent(converter), this.getPageable(), this.total);
    }

    public String toString() {
        String contentType = "UNKNOWN";
        List<T> content = this.getContent();
        if (!content.isEmpty() && content.get(0) != null) {
            contentType = content.get(0).getClass().getName();
        }

        return String.format("Page %s of %d containing %s instances", this.getNumber() + 1, this.getTotalPages(), contentType);
    }

    public boolean equals(@Nullable Object obj) {
        if (this == obj) {
            return true;
        } else if (!(obj instanceof ReleasePageImpl)) {
            return false;
        } else {
            ReleasePageImpl<?> that = (ReleasePageImpl) obj;
            return this.total == that.total && super.equals(obj);
        }
    }

    public int hashCode() {
        int result = 17;
        result += 31 * (int) (this.total ^ this.total >>> 32);
        result += 31 * super.hashCode();
        return result;
    }
}
