package com.xebialabs.xlrelease.service

import com.xebialabs.xlrelease.api.v1.views.TeamMemberView.MemberType
import com.xebialabs.xlrelease.api.v1.views.{TeamMemberView, TeamView => PublicTeamView}
import com.xebialabs.xlrelease.db.sql.transaction.IsTransactional
import com.xebialabs.xlrelease.domain.Team
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.events.XLReleaseEventBus
import com.xebialabs.xlrelease.repository.Ids.isFolderId
import com.xebialabs.xlrelease.repository.TeamRepository.TeamRow
import com.xebialabs.xlrelease.repository.{SecuredCis, SecurityRepository, TeamArrayPropertyChange, TeamRepository}
import com.xebialabs.xlrelease.security.SecuredCi
import com.xebialabs.xlrelease.views.TeamView
import org.springframework.data.domain.{Page, PageImpl}

import java.util
import scala.annotation.varargs
import scala.jdk.CollectionConverters._

@IsTransactional
trait TeamOperationsService {
  self: TeamService =>
  def securityRepository: SecurityRepository

  def teamRepository: TeamRepository

  def ciIdService: CiIdService

  def securedCis: SecuredCis

  def eventBus: XLReleaseEventBus

  @varargs
  def executeTeamUpdateOperations(operations: TeamUpdateOperation*): Unit = {
    val events: Seq[_ <: TeamEvent] = operations.collect {
      case bi: BreakInheritanceOperation => breakInheritance(bi)
      case create: CreateTeamOperation => createTeamIfNeeded(create)
      case update: UpdateTeamOperation => updateTeam(update)
      case delete: DeleteTeamOperation => deleteTeam(delete)
    }.flatten
    events.foreach(e => eventBus.publish(e))
    operations.headOption.foreach { op =>
      eventBus.publish(TeamsUpdatedEvent(op.containerId))
    }
  }

  def breakInheritance(op: BreakInheritanceOperation): Seq[_ <: TeamEvent] = {
    val containerId = op.containerId
    // find old id
    val oldSecuredCi = securedCis.getEffectiveSecuredCi(containerId) // this is PREVIOUS value
    val oldContainerId = oldSecuredCi.getId
    if (isFolderId(oldContainerId)) {
      // take all data for old container
      val teams = teamRepository.getTeams(oldSecuredCi).asScala
      // replace all team ids with new container id
      teams.foreach(t => t.setId(ciIdService().getUniqueId("Team", containerId)))
      teamRepository.saveTeamsToPlatform(containerId, teams.asJava)
    }
    // make sure effective securedCi is correct
    // securedCis.setAsEffectiveSecuredCi(containerId, isEffective = true) // this is already done by saveTeamsToPlatform
    Seq() // events are emitted in saveTeamsToPlatform if at all
  }

  private def createTeamIfNeeded(op: CreateTeamOperation): Seq[_ <: TeamEvent] = {
    val teamId = securityRepository.createTeam(op.containerId, op.team, op.membership.roles, op.membership.principals, op.membership.permissions)
    val createdTeam = new Team()
    createdTeam.setId(teamId)
    createdTeam.setTeamName(op.team)
    createdTeam.setMembers(op.membership.principals.toList.asJava)
    createdTeam.setRoles(op.membership.roles.toList.asJava)
    createdTeam.setPermissions(op.membership.permissions.toList.asJava)
    Seq(TeamCreatedEvent(op.containerId, createdTeam))
  }

  private def updateTeam(op: UpdateTeamOperation): Seq[_ <: TeamEvent] = {
    val roles = TeamArrayPropertyChange(op.addedMembership.roles, op.removedMembership.roles)
    val principals = TeamArrayPropertyChange(op.addedMembership.principals, op.removedMembership.principals)
    val permissions = TeamArrayPropertyChange(op.addedMembership.permissions, op.removedMembership.permissions)
    val event = op.teamId.orElse(securityRepository.findTeamId(op.containerId, op.getTeamName)).flatMap { teamId =>
      val original = securityRepository().getTeam(teamId)
      securityRepository.updateTeam(teamId, roles, principals, permissions)
      val updated = securityRepository().getTeam(teamId)
      if (original != updated) {
        Some(TeamUpdatedEvent(op.containerId, original, updated))
      } else {
        None
      }
    }

    Seq.empty ++ event
  }

  private def deleteTeam(op: DeleteTeamOperation): Seq[_ <: TeamEvent] = {
    val event = op.teamId.orElse(securityRepository.findTeamId(op.containerId, op.getTeamName)).map { teamId =>
      val team = securityRepository().getTeam(teamId)
      securityRepository.deleteTeam(teamId)
      TeamDeletedEvent(op.containerId, team)
    }
    Seq.empty ++ event
  }

  def getTeamViews(containerId: String): java.util.List[TeamView] = {
    val securedCi = securedCis.getEffectiveSecuredCi(containerId)
    getTeamViews(securedCi).getContent
  }

  def getPublicTeamViews(containerId: String): java.util.List[PublicTeamView] = {
    // the difference between public and internal TeamView is in serialization
    // we could remove internal one if we go and fix our UI client
    getTeamViews(containerId).asScala.map { tv =>
      val pv = new PublicTeamView()
      pv.setId(tv.getId)
      pv.setTeamName(tv.getTeamName)
      pv.setMembers(tv.getMembers)
      pv.setPermissions(tv.getPermissions)
      pv.setSystemTeam(tv.isSystemTeam)
      pv
    }.asJava
  }

  // team views
  private def getTeamViews(container: SecuredCi): Page[TeamView] = {
    val membersPerTeam = getMembersPerTeam(container)
    val permissionsPerTeam = getPermissionsPerTeam(container)

    val teamViews = for {
      (teamInfo, members) <- membersPerTeam
      permissions = new util.ArrayList(permissionsPerTeam.getOrElse(teamInfo, Seq.empty[String]).asJava)
      fullTeamId = fromPlatformId(container.getId, teamInfo.teamId)
      systemTeam = Team.isSystemTeam(teamInfo.teamName)
    } yield new TeamView(fullTeamId, teamInfo.teamName, new util.ArrayList(members.asJava), permissions, systemTeam)

    new PageImpl[TeamView](teamViews.toSeq.asJava)
  }

  private def getMembersPerTeam(container: SecuredCi): Map[TeamRow, Seq[TeamMemberView]] = {
    val rows = teamRepository.getMembersPerTeam(container)
    val res = rows.groupMap(r => r.team)(r => r.member).collect {
      case (tf, v) => (tf, v.flatten.map { mr =>
        val memberView = new TeamMemberView()
        memberView.setName(mr.name)
        memberView.setFullName(mr.fullName)
        memberView.setRoleId(mr.roleId)
        memberView.setType(if (mr.roleId != null) MemberType.ROLE else MemberType.PRINCIPAL)
        memberView
      })
    }
    res
  }

  private def getPermissionsPerTeam(container: SecuredCi): Map[TeamRow, Seq[String]] = {
    val rows = teamRepository.getTeamPermissionRows(container)
    rows.groupMap(r => r.teamInfo)(r => r.permissionName)
  }

  private def fromPlatformId(containerId: String, id: String): String = containerId + "/" + id
}
