package com.xebialabs.xlrelease.actors.initializer

import com.xebialabs.xlplatform.cluster.ClusterMode._
import com.xebialabs.xlrelease.actors.{ActorServiceHolder, ActorSystemHolder, ArchivingActor}
import com.xebialabs.xlrelease.config.XlrConfig
import grizzled.slf4j.Logging
import org.apache.pekko.actor.{ActorRef, PoisonPill, Props}
import org.apache.pekko.cluster.singleton.{ClusterSingletonManager, ClusterSingletonManagerSettings}
import org.springframework.beans.factory.annotation.Autowired
import org.springframework.context.annotation.Profile
import org.springframework.core.annotation.Order
import org.springframework.stereotype.Component

trait ArchivingInitializer extends ActorInitializer with Logging {
  def archivingProps(xlrConfig: XlrConfig, actorServiceHolder: ActorServiceHolder): Props = {
    ArchivingActor.props(actorServiceHolder.archivingScheduleService, xlrConfig)
  }
}

@Component
@Order(2)
@Profile(Array(STANDALONE, HOT_STANDBY))
class NonClusteredArchivingInitializer @Autowired()(xlrConfig: XlrConfig, systemHolder: ActorSystemHolder, actorServiceHolder: ActorServiceHolder)
  extends ArchivingInitializer {

  lazy val archivingActor: ActorRef = systemHolder.actorSystem.actorOf(archivingProps(xlrConfig, actorServiceHolder))

  override def initialize(): Unit = {
    if (xlrConfig.maintenanceModeEnabled) {
      logger.debug("[MAINTENANCE MODE] Archiving scheduler actor - DISABLED")
    } else {
      logger.debug("Initializing non-clustered archiving actor...")
      archivingActor
    }
  }
}

@Component
@Order(2)
@Profile(Array(FULL))
class ClusteredArchivingInitializer @Autowired()(xlrConfig: XlrConfig, systemHolder: ActorSystemHolder, actorServiceHolder: ActorServiceHolder)
  extends ArchivingInitializer with Logging {

  lazy val archivingActor: ActorRef = clusteredArchivingActor(archivingProps(xlrConfig, actorServiceHolder))

  override def initialize(): Unit = {
    if (xlrConfig.maintenanceModeEnabled) {
      logger.debug("[MAINTENANCE MODE] Archiving scheduler actor - DISABLED")
    } else {
      logger.debug("Initializing clustered archiving actor...")
      archivingActor
    }
  }

  private[initializer] def clusteredArchivingActor(archivingProps: Props): ActorRef = {
    systemHolder.actorSystem.actorOf(ClusterSingletonManager.props(
      singletonProps = archivingProps,
      terminationMessage = PoisonPill,
      settings = ClusterSingletonManagerSettings(systemHolder.actorSystem)),
      name = ArchivingActor.name)
  }
}

