package com.xebialabs.xlrelease.service.blackout

import com.xebialabs.xlrelease.actors.ReleaseExecutionActorMessages.ExtensionCommand
import com.xebialabs.xlrelease.actors.extension.ActorExtensionHandlerFactory
import com.xebialabs.xlrelease.actors.{ReleaseActorService, ReleaseExecutionActor}
import com.xebialabs.xlrelease.domain.blackout.{BlackoutMetadata, BlackoutPeriod}
import com.xebialabs.xlrelease.domain.events.{BlackoutDeletedEvent, BlackoutUpdatedEvent, TaskUpdatedEvent}
import com.xebialabs.xlrelease.domain.{Release, Task}
import com.xebialabs.xlrelease.events.{EventBus, Subscribe}
import com.xebialabs.xlrelease.repository.CiCloneHelper.cloneCi
import com.xebialabs.xlrelease.repository._
import com.xebialabs.xlrelease.service.blackout.BlackoutReleaseExecutionActorMessages.{ResetScheduledStartDate, UpdateScheduledStartDate}
import grizzled.slf4j.Logging
import io.micrometer.core.annotation.Timed
import org.apache.pekko.actor.Actor._
import org.apache.pekko.actor.{Actor, ActorRef, ActorSystem}
import org.springframework.stereotype.Service

import java.util.Date
import jakarta.annotation.{PostConstruct, PreDestroy}
import scala.jdk.CollectionConverters._

object BlackoutReleaseExecutionActorMessages {

  case class UpdateScheduledStartDate(taskId: String, newScheduledStartDate: Date) extends ExtensionCommand

  case class ResetScheduledStartDate(taskId: String) extends ExtensionCommand

}

@Service
class BlackoutTaskEventHandler(val actorSystem: ActorSystem,
                               val eventBus: EventBus,
                               val releaseActorService: ReleaseActorService,
                               val taskRepository: TaskRepository)
  extends ActorExtensionHandlerFactory with Logging {

  @PostConstruct
  def register(): Unit = {
    eventBus.register(this)
    logger.debug("Blackout task event handler registered in event bus.")
  }

  @PreDestroy
  def unregister(): Unit = {
    eventBus.deregister(this)
    logger.debug("Blackout task event handler de-registered from event bus.")
  }

  @Subscribe
  @Timed
  def onBlackoutUpdated(event: BlackoutUpdatedEvent): Unit = {
    val now = new Date()

    if (isAffectingTasks(event, now)) {
      val taskIds = taskRepository.findAffectedByBlackout(event.original).asScala
      logger.debug(s"${taskIds.length} taskIds to be updated after blackout ${event.original.getLabel} update.")

      val metadata = BlackoutMetadata(Seq(BlackoutPeriod(event.updated.getStartDate, event.updated.getEndDate)))
      val isInBlackout = metadata.isInBlackout(now)
      val newScheduledStartDate = metadata.getEndOfBlackout(now)

      taskIds.foreach { taskId =>
        scheduleTaskUpdate(taskId, if (isInBlackout) UpdateScheduledStartDate(taskId, newScheduledStartDate) else ResetScheduledStartDate(taskId))
      }
    }
  }

  @Subscribe
  @Timed
  def onBlackoutDeleted(event: BlackoutDeletedEvent): Unit = {
    val taskIds = taskRepository.findAffectedByBlackout(event.blackout).asScala
    logger.debug(s"${taskIds.length} taskIds to be started after blackout ${event.blackout.getLabel} removal.")

    taskIds.foreach { taskId => scheduleTaskUpdate(taskId, ResetScheduledStartDate(taskId)) }
  }

  private def isAffectingTasks(event: BlackoutUpdatedEvent, now: Date): Boolean = {
    val endDateChanged = event.original.getEndDate != event.updated.getEndDate
    val startDateMovedIntoFuture = now.after(event.original.getStartDate) && now.before(event.updated.getStartDate)
    endDateChanged || startDateMovedIntoFuture
  }


  private def scheduleTaskUpdate(taskId: String, command: ExtensionCommand): Unit = {
    releaseActorService.executeCommandAsync(taskId, command)
  }

  override def getHandler(self: ActorRef, sender: () => ActorRef, release: => Release): Receive = {
    case UpdateScheduledStartDate(taskId, newScheduledStartDate) => updateTaskWith(taskId) {
      updateScheduledStartDate(_, newScheduledStartDate)
    }
    case ResetScheduledStartDate(taskId) => updateTaskWith(taskId) {
      resetScheduledStartDate
    }
  }

  private def updateTaskWith(taskId: String)(updater: Task => Unit): Unit = {
    val task: Task = taskRepository.findById(taskId)
    val original = cloneCi(task)

    updater(task)

    taskRepository.update(task)
    eventBus.publish(TaskUpdatedEvent(original, task))
  }

  private def updateScheduledStartDate(task: Task, newScheduledStartDate: Date): Unit = {
    logger.debug(s"Updating task '${task.getId}' scheduled start date due to new blackout end date '$newScheduledStartDate'")

    val originalScheduledStartDate = task.getOriginalScheduledStartDate
    task.setScheduledStartDate(newScheduledStartDate)
    task.setPostponedDueToBlackout(true)
    task.setOriginalScheduledStartDate(originalScheduledStartDate)
  }

  private def resetScheduledStartDate(task: Task): Unit = {
    logger.debug(s"Resetting task '${task.getId}' scheduled start date due to blackout not being applicable anymore.")

    task.setScheduledStartDate(task.getOriginalScheduledStartDate)
    task.setPostponedDueToBlackout(false)
  }

  override def supports(clazz: Class[_ <: Actor]): Boolean = classOf[ReleaseExecutionActor].isAssignableFrom(clazz)


}
