package com.xebialabs.xlrelease.service

import com.xebialabs.deployit.checks.Checks.checkArgument
import com.xebialabs.deployit.exception.NotFoundException
import com.xebialabs.xlrelease.api.v1.forms.{TenantLinkRequest, TenantResourceLinkOperation}
import com.xebialabs.xlrelease.domain.Tenant
import com.xebialabs.xlrelease.domain.events._
import com.xebialabs.xlrelease.domain.status.TenantStatus
import com.xebialabs.xlrelease.events.EventBus
import com.xebialabs.xlrelease.repository.{Ids, TenantRepository}
import org.springframework.stereotype.Service
import org.springframework.util.StringUtils.hasText

import java.util.{List => JList}
import scala.jdk.CollectionConverters._

@Service
class TenantService(tenantRepository: TenantRepository, eventBus: EventBus) {

  def getAllTenants(): List[Tenant] = {
    tenantRepository.findAll().toList
  }

  def findTenantById(tenantId: String): Tenant = {
    checkArgument(hasText(tenantId), "Tenant ID cannot be empty")
    tenantRepository.findById(tenantId).getOrElse(throw new NotFoundException(s"No tenant found with ID '$tenantId'"))
  }

  def findTenantByName(name: String): Tenant = {
    checkArgument(hasText(name), "Tenant name cannot be empty")
    tenantRepository.findByName(name.toLowerCase.trim).getOrElse(throw new NotFoundException(s"No tenant found with name '$name'"))
  }

  def createTenant(tenantName: String, tenantStatus: TenantStatus): Tenant = {
    checkArgument(hasText(tenantName), "Tenant name cannot be empty")
    checkArgument(tenantStatus != null, "Tenant status cannot be null")

    val normalizedTenantName = tenantName.toLowerCase.trim
    val createdTenant = tenantRepository.create(normalizedTenantName, tenantStatus)
    eventBus.publish(TenantCreatedEvent(createdTenant))
    createdTenant
  }

  def updateTenant(tenant: Tenant): Unit = {
    checkArgument(tenant != null, "Tenant cannot be null")
    checkArgument(hasText(tenant.tenantName), "Tenant name cannot be empty")
    checkArgument(tenant.tenantStatus != null, "Tenant status cannot be null")

    val original = findTenantById(tenant.tenantId)
    tenantRepository.update(tenant)
    eventBus.publish(TenantUpdatedEvent(original, tenant))
  }

  def deleteTenant(tenantId: String): Unit = {
    checkArgument(hasText(tenantId), "Tenant ID cannot be empty")
    val tenant = findTenantById(tenantId)

    tenantRepository.delete(tenantId)
    eventBus.publish(TenantDeletedEvent(tenant))
  }

  def linkResource(tenantId: String, linkRequests: JList[TenantLinkRequest]): Unit = {
    checkArgument(hasText(tenantId), "Tenant ID cannot be empty")
    checkArgument(linkRequests != null, "Link requests cannot be null")

    // Verify tenant exists
    findTenantById(tenantId)

    linkRequests.asScala.toList.groupBy(_.getLinkOperation).foreach {
      case (TenantResourceLinkOperation.FOLDER, folderRequests) =>
        val folderIds = folderRequests.map(_.getTargetId).filter(hasText).filter(Ids.isFolderId).map(Ids.getName).map(Ids.normalizeId)
        if (folderIds.nonEmpty) {
          tenantRepository.linkFoldersToTenant(tenantId, folderIds)
          eventBus.publish(FolderTenantLinkEvent(tenantId, folderIds))
        }

      case (TenantResourceLinkOperation.USER_PROFILE, userProfileRequests) =>
        val userProfileIds = userProfileRequests.map(_.getTargetId).filter(hasText).map(_.toLowerCase)
        if (userProfileIds.nonEmpty) {
          tenantRepository.linkUserProfilesToTenant(tenantId, userProfileIds)
          eventBus.publish(UserProfileTenantLinkEvent(tenantId, userProfileIds))
        }
    }
  }
}
