package com.xebialabs.xltest.fitnesse;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

import com.xebialabs.xltest.domain.*;

import fitnesse.FitNesseContext;
import fitnesse.reporting.history.*;
import fitnesse.wiki.*;
import fitnesse.wiki.fs.FileSystemPage;
import fitnesse.wiki.fs.FileSystemPageFactory;
import util.FileUtil;

import static com.xebialabs.xltest.importers.ImporterUtil.sendEvent;
import static java.lang.String.format;

public class PageHistoryExtractor {
    private static final Logger LOG = LoggerFactory.getLogger(PageHistoryExtractor.class);

    private final File resultsDirectory;
    private final FileSystemPage root;
    private final TestHistory history = new TestHistory();

    public PageHistoryExtractor(File fitNesseRoot) {
        resultsDirectory = new File(fitNesseRoot, "files/" + FitNesseContext.testResultsDirectoryName);
        root = new FileSystemPageFactory().makePage(fitNesseRoot, "FitNesseRoot", null, new SystemVariableSource());
        history.readHistoryDirectory(resultsDirectory);

    }

    public List<Importable> createImportables() throws ImportFailedException {
        List<Importable> toImport = new ArrayList<>();

        File[] resultFiles = resultsDirectory.listFiles();
        if (resultFiles == null) {
            return Collections.emptyList();
        }

        try {
            for (File pageNamedFile : resultFiles) {
                if (!pageNamedFile.isDirectory()) {
                    continue;
                }

                // check if it contains test results or suite results
                String pageName = pageNamedFile.getName();
                LOG.trace("Will try to make report for " + pageName);

                PageHistory pageHistory = getPageHistory(resultsDirectory, pageName);

                if (pageHistoryIsSuite(pageHistory)) {
                    for (Date date : pageHistory.datesInChronologicalOrder()) {
                        ExecutionReport executionReport = getExecutionReport(pageHistory, date);
                        if (executionReport instanceof SuiteExecutionReport) {
                            toImport.add(new FitNesseSuiteImporter((SuiteExecutionReport) executionReport, pageName));
                        }
                    }
                }
            }
        } catch (Exception e) {
            throw new ImportFailedException(e);
        }

        Collections.sort(toImport, new Comparator<Importable>() {
            @Override
            public int compare(Importable o1, Importable o2) {
                return o1.toString().compareTo(o2.toString());
            }
        });

        return toImport;
    }

    public void produceEventsForReport(UUID testRunId, SuiteExecutionReport suiteExecutionReport, EventHandler eventHandler) throws ImportException {
        for (SuiteExecutionReport.PageHistoryReference reference : suiteExecutionReport.getPageHistoryReferences()) {
            Set<String> tags = findTags(reference);
            String firstErrorMessage = findFirstErrorMessage(reference);

            sendEvent(eventHandler, testRunId, Event.FUNCTIONAL_RESULT, new TestSummary(
                    Event.normalizeName(reference.getPageName(), '.'),
                    tags,
                    reference.getTime(),
                    (int) reference.getRunTimeInMillis(),
                    reference.getTestSummary().getRight(),
                    reference.getTestSummary().getWrong(),
                    reference.getTestSummary().getWrong() + reference.getTestSummary().getExceptions() > 0 ? "FAILED" : "PASSED",
                    reference.getTestSummary().getExceptions(),
                    firstErrorMessage).toMap());
        }
    }

    private Set<String> findTags(SuiteExecutionReport.PageHistoryReference reference) {
        WikiPagePath wikiPagePath = PathParser.parse(reference.getPageName());
        PageCrawler pageCrawler = root.getPageCrawler();
        Set<String> tags = set();
        if (pageCrawler.pageExists(wikiPagePath)) {
            WikiPage thePage = pageCrawler.getPage(wikiPagePath);
            tags = findTags(thePage);
        }
        return tags;
    }

    /**
     * Find tags on wiki page, so we can add tags on parent pages as well.
     *
     * @param page
     * @return a set of tags
     */
    private Set<String> findTags(WikiPage page) {
        if (page == null || page.isRoot()) {
            return set();
        }

        String tags = page.getData().getAttribute(PageData.PropertySUITES);
        Set<String> tagSet = isNotBlank(tags) ? set(tags.split(",")) : set();
        tagSet.addAll(findTags(page.getParent()));
        return tagSet;
    }

    private String findFirstErrorMessage(SuiteExecutionReport.PageHistoryReference reference) {
        String firstErrorMessage = null;
        try {
            TestExecutionReport testExecutionReport = getExecutionReport(getPageHistory(resultsDirectory, reference.getPageName()), new Date(reference.getTime()));
            firstErrorMessage = findFirstErrorMessage(testExecutionReport);
        } catch (Exception e) {
            LOG.warn("Unable to find test result file for test page {}", reference.getPageName());
            LOG.debug("Could not determine first error message", e);
        }
        return firstErrorMessage;
    }

    private String findFirstErrorMessage(TestExecutionReport testExecutionReport) {
        TestExecutionReport.TestResult testResult = testExecutionReport.getResults().get(0);

        for (TestExecutionReport.InstructionResult instructionResult : testResult.getInstructions()) {
            for (TestExecutionReport.Expectation expectation : instructionResult.getExpectations()) {
                if ("fail".equals(expectation.getStatus())) {
                    return format("Actual: '%s'; Expected: '%s'", expectation.getActual(), expectation.getExpected());
                }
                if ("error".equals(expectation.getStatus())) {
                    return expectation.getEvaluationMessage();
                }
            }
        }
        return null;
    }

    private <T extends ExecutionReport> T getExecutionReport(PageHistory pageHistory, Date date) throws Exception {
        TestResultRecord testResultRecord = pageHistory.get(date);
        return (T) ExecutionReport.makeReport(FileUtil.getFileContent(testResultRecord.getFile()));
    }

    private PageHistory getPageHistory(File resultsDirectory, String pageName) throws FileNotFoundException {
        if (!resultsDirectory.canRead()) {
            throw new FileNotFoundException("Could not find " + resultsDirectory);
        }
        return history.getPageHistory(pageName);
    }

    private boolean pageHistoryIsSuite(PageHistory pageHistory) throws Exception {
        if (pageHistory == null) {
            return false;
        }
        ExecutionReport report = getExecutionReport(pageHistory, pageHistory.getLatestDate());
        return report instanceof SuiteExecutionReport;
    }

    public static Set<String> set(String... tags) {
        Set<String> tagSet = new TreeSet<>();
        for (String tag : tags) {
            String trimmed = tag.trim();
            if (!"".equals(trimmed)) {
                tagSet.add(trimmed);
            }
        }
        return tagSet;
    }

    private boolean isNotBlank(String tags) {
        return tags != null && !"".equals(tags);
    }

    public class FitNesseSuiteImporter implements Importable {
        private final SuiteExecutionReport executionReport;
        private final String pageName;

        public FitNesseSuiteImporter(SuiteExecutionReport executionReport, String pageName) {
            this.executionReport = executionReport;
            this.pageName = pageName;
        }

        public long getTimestamp() {
            return executionReport.getDate().getTime();
        }

        @Override
        public void doImport(UUID testRunId, EventHandler eventHandler) throws ImportException {
            LOG.debug("doImport({},{})", testRunId, eventHandler);
            sendEvent(eventHandler, testRunId, Event.IMPORT_STARTED, Event.props(
                    "fileName", pageName,
                    "lastModified", executionReport.getDate().getTime(),
                    "suiteName", executionReport.getRootPath() // the suite name
            ));

            try {
                produceEventsForReport(testRunId, executionReport, eventHandler);
            } finally {
                int duration = (int) executionReport.getTotalRunTimeInMillis();
                sendEvent(eventHandler, testRunId, Event.IMPORT_FINISHED, Event.props(
                        "duration", duration
                ));
            }
        }
    }
}
