package com.xebialabs.xltest;

import com.xebialabs.deployit.ServerConfiguration;
import com.xebialabs.deployit.core.rest.api.MetadataServiceImpl;
import com.xebialabs.deployit.engine.api.MetadataService;
import com.xebialabs.deployit.jcr.JackrabbitRepositoryFactoryBean;
import com.xebialabs.deployit.jcr.JcrTemplate;
import com.xebialabs.deployit.plugin.api.services.Repository;
import com.xebialabs.deployit.repository.*;
import com.xebialabs.deployit.service.validation.Validator;
import com.xebialabs.deployit.util.DeployitKeys;
import com.xebialabs.deployit.util.PasswordEncrypter;
import com.xebialabs.xltest.service.EventRepository;
import com.xebialabs.xltest.service.EventRepositoryImpl;
import com.xebialabs.xltest.service.InMemoryRepositoryService;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.DisposableBean;
import org.springframework.context.annotation.Bean;
import org.springframework.context.annotation.ComponentScan;
import org.springframework.context.annotation.Configuration;
import org.springframework.context.annotation.ImportResource;
import org.springframework.core.io.ClassPathResource;
import org.springframework.core.io.FileSystemResource;
import org.springframework.core.io.Resource;

import javax.crypto.SecretKey;
import javax.jcr.RepositoryException;
import java.io.IOException;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.ForkJoinPool;

/**
 * This class contains the "configurable" part of the XL Test application.
 */
@Configuration
@ComponentScan({"com.xebialabs.deployit.service.validation", "com.xebialabs.xltest"})
@ImportResource("classpath:spring/xltest-context.xml")
public class ContextConfig implements DisposableBean {
    private static final Logger LOG = LoggerFactory.getLogger(ContextConfig.class.getName());

    private JackrabbitRepositoryFactoryBean factory;

    /**
     * Bean for ConfigurationItem support.
     *
     * @return
     */
    @Bean
    public MetadataService metadataService() {
        return new MetadataServiceImpl();
    }

    @Bean
    public Repository repository(RepositoryService repositoryService, Validator validator) {
        WorkDir workDir = null;
        return new RepositoryAdapter(repositoryService, validator, workDir);
    }

    @Bean
    public RepositoryServiceHolder repositoryServiceHolder(RepositoryService repositoryService) {
        return new RepositoryServiceHolder(repositoryService);
    }

    @Bean
    public RepositoryService repositoryService() throws IOException, RepositoryException {
        if ("true".equals(System.getProperty("testMode"))) {
            LOG.info("Using in memory repository service");
            return new InMemoryRepositoryService();
        } else {
            JackrabbitRepositoryFactoryBean factory = getFactory();

            ServerConfiguration deployitConfig = new ServerConfiguration();
            deployitConfig.setDefaults();
            factory.setDeployitConfig(deployitConfig);

            factory.afterPropertiesSet();
            javax.jcr.Repository repo = factory.getObject();

            SecretKey secretKey = DeployitKeys.getPasswordEncryptionKey("keyStorePassword");
            PasswordEncrypter.init(secretKey);
            PasswordEncrypter passwordEncrypter = PasswordEncrypter.getInstance();
            JcrRepositoryService jcrRepositoryService = new JcrRepositoryService(new JcrTemplate(repo, deployitConfig), passwordEncrypter);

            // Fix the demo data...
            new DemoData().provision(Main.booter, jcrRepositoryService);

            return jcrRepositoryService;
        }
    }

    public JackrabbitRepositoryFactoryBean getFactory() {
        if (factory == null) {
            factory = makeFactory();
        }
        return factory;
    }

    private JackrabbitRepositoryFactoryBean makeFactory() {
        JackrabbitRepositoryFactoryBean factory = new JackrabbitRepositoryFactoryBean();
        factory.setHomeDir(new FileSystemResource("repository"));
        factory.setRepositoryName("xltest");
        Resource configuration = new ClassPathResource("jackrabbit-repository.xml");
        factory.setConfiguration(configuration);
        factory.setAutoInitialize(true);
        return factory;
    }

    @Bean
    public EventRepository eventRepository() throws NoSuchMethodException {
        return new EventRepositoryImpl(Main.booter);
    }

    @Bean
    public ExecutorService executorService() {
        return new ForkJoinPool(128);
    }

    @Override
    public void destroy() throws Exception {
        factory.destroy();
    }
}
