package com.xebialabs.xltest.resources;

import java.io.IOException;
import java.util.*;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.*;
import javax.ws.rs.core.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;
import org.springframework.stereotype.Controller;
import com.jayway.jsonpath.spi.JsonProviderFactory;

import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.deployit.repository.SearchParameters;
import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.domain.Report;
import com.xebialabs.xltest.domain.TestRun;
import com.xebialabs.xltest.repository.NotFoundException;
import com.xebialabs.xltest.repository.TestRunsRepository;

import static com.xebialabs.deployit.plugin.api.reflect.Type.valueOf;

@Controller
@Path("/reports")
@Consumes(MediaType.APPLICATION_JSON)
@Produces(MediaType.APPLICATION_JSON)
public class ReportController {
    private static final Logger LOG = LoggerFactory.getLogger(ReportController.class);

    private final RepositoryService repositoryService;
    private final TestRunsRepository testRunsRepository;
    private final AutowireCapableBeanFactory beanFactory;

    @Autowired
    public ReportController(RepositoryService repositoryService, TestRunsRepository testRunsRepository, AutowireCapableBeanFactory beanFactory) {
        this.repositoryService = repositoryService;
        this.testRunsRepository = testRunsRepository;
        this.beanFactory = beanFactory;
    }

    @GET
    @Path("/{testRunId:[\\w-]+}")
    public Collection<Report> getReports(@PathParam("testRunId") String testRunId, @Context final HttpServletResponse response) throws IOException {
        try {
            TestRun testRun = testRunsRepository.getTestRun(UUID.fromString(testRunId));
            List<Report> reports = findReportForTestSetDefinitionType();
            if (reports != null && !reports.isEmpty()) {
                return reports;
            }
            response.sendError(404, "Can't find any Report associated to test runs of: " + testRun.getId());
            return null;
        } catch (NotFoundException e) {
            response.sendError(404, "Can't find test run");
            return null;
        }
    }

    private List<Report> findReportForTestSetDefinitionType() {
        SearchParameters query = new SearchParameters().setType(valueOf(Report.class));
        List<Report> reportCis = repositoryService.listEntities(query);
        return reportCis;
    }

    @GET
    @Path("/{testRunId:[\\w-]+}/Configuration/Reports/{reportName:.*}")
    public String generateReport(@PathParam("testRunId") String testRunId, @PathParam("reportName") String reportName, @Context UriInfo uriInfo) throws IOException {
        // Note that we use 'Configuration/Reports' in the pattern to prevent a clash with the method below
        Report report = repositoryService.read("Configuration/Reports/" + reportName);
        LOG.info("Found report with value {}", report);
        Map<String, Object> resultObject = generateReport(UUID.fromString(testRunId), report, convertMultivaluedMapToMap(uriInfo.getQueryParameters()));
        return JsonProviderFactory.createProvider().toJson(resultObject);
    }

    @GET
    @Path("/{reportName}/testspecification/{testSpecificationName:.*}")
    public Response generateReportFromTestSpecification(@PathParam("testSpecificationName") String testSpecificationName, @PathParam("reportName") String reportName, @Context UriInfo uriInfo) throws IOException {
        UUID testRunId = findLastRunOf(testSpecificationName);
        if (testRunId == null) {
            return Response.noContent().build();
        }
        Report report = repositoryService.read("Configuration/Reports/" + reportName);
        Map<String, Object> resultObject = generateReport(testRunId, report, convertMultivaluedMapToMap(uriInfo.getQueryParameters()));
        return Response.ok(JsonProviderFactory.createProvider().toJson(resultObject)).build();
    }


    private UUID findLastRunOf(String testSpecificationName) {
        TestRun lastTestRun = null;
        for (TestRun testRun : testRunsRepository.getAllTestRuns()) {
            if (testRun.getTestSpecificationName().equals(testSpecificationName)) {
                if (lastTestRun == null) {
                    lastTestRun = testRun;
                } else {
                    if (testRun.getStartTime().after(lastTestRun.getStartTime())) {
                        lastTestRun = testRun;
                    }
                }
            }
        }
        return lastTestRun == null ? null : lastTestRun.getTestRunId();
    }


    @POST
    @Path("/{testRunId:.+}")
    public String generateReport(@PathParam("testRunId") String testRunId, Report report, @Context UriInfo uriInfo) {
        Map<String, Object> resultObject = generateReport(UUID.fromString(testRunId), report, convertMultivaluedMapToMap(uriInfo.getQueryParameters()));
        return JsonProviderFactory.createProvider().toJson(resultObject);
    }

    /**
     * Yes, we drop all but the first parameter value.
     *
     * @param multivaluedMap
     * @return
     */
    private Map<String, String> convertMultivaluedMapToMap(MultivaluedMap<String, String> multivaluedMap) {
        Map<String, String> map = new TreeMap();
        for (String key : multivaluedMap.keySet()) {
            map.put(key, multivaluedMap.getFirst(key));
        }
        return map;
    }

    private Map<String, Object> generateReport(UUID testRunId, Report report, Map<String, String> queryParameters) {
        TestRun run = testRunsRepository.getTestRun(testRunId);
        return generateReport(run, report, queryParameters);
    }

    private Map<String, Object> generateReport(TestRun testRun, Report report, Map<String, String> queryParameters) {
        if (report != null) {
            beanFactory.autowireBean(report);
            report.addAttribute("queryParameters", queryParameters);
            report.addAttribute("startDate", testRunsRepository.makeStartDateIfProvided(queryParameters.get("startDate")));
            report.addAttribute("endDate", testRunsRepository.makeEndDateIfProvided(queryParameters.get("endDate")));

            Map<String, Object> resultObject = new TreeMap();
            resultObject.put(Event.TYPE, report.getReportType().toLowerCase());
            resultObject.put("report", report.compute(testRun));
            return resultObject;
        }
        return Collections.emptyMap();
    }
}

