package com.xebialabs.xltest.service;

import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.jayway.jsonpath.spi.JsonProvider;
import com.jayway.jsonpath.spi.JsonProviderFactory;

import com.xebialabs.xltest.domain.Event;
import com.xebialabs.xltest.repository.XltRepositoryException;

/**
 * This repository is an introspection based "wrapper" around
 * com.xebialabs.xltest.elasticsearch.ElasticSearchBooter.
 * We use reflection since the booter is launched in it's own, isolated scope.
 */
public class EventRepositoryImpl implements EventRepository {
    private final static Logger LOG = LoggerFactory.getLogger(EventRepositoryImpl.class);

    private final Object client;
    private final Method insertMethod;
    private final Method queryMethod;

    public EventRepositoryImpl(Object client) throws NoSuchMethodException {
        super();
        this.client = client;
        insertMethod = client.getClass().getDeclaredMethod("insert", String.class, String.class);
        queryMethod = client.getClass().getDeclaredMethod("query", String.class);
    }

    public void insert(Event event) {
        JsonProvider jsonProvider = JsonProviderFactory.createProvider();
        String content = jsonProvider.toJson(event.getProperties());
        try {
            insertMethod.invoke(client, event.getType(), content);
        } catch (Exception e) {
            e.printStackTrace();
            throw new XltRepositoryException("Problem encountered in inserting event " + event, e);
        }
    }

    public List<Event> query(String query) {
        try {
            LOG.debug("About to query event repository with query {}", query);
            return parseElasticSearchJson((List<String>) queryMethod.invoke(client, query));
        } catch (ReflectiveOperationException e) {
            throw new RuntimeException("Unable to read from internal store", e);
        }
    }

    private List<Event> parseElasticSearchJson(List<String> hits) {
        if (hits == null) {
            return Collections.emptyList();
        }
        List<Event> retrievedEvents = new ArrayList<Event>(hits.size());
        for (String hit : hits) {
            Map<String, Object> object = (Map<String, Object>) JsonProviderFactory.createProvider().parse(hit);
            retrievedEvents.add(new Event(object));
        }
        return retrievedEvents;
    }

}
