/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.script;

import java.io.File;
import java.io.FileInputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.util.Locale;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.ConcurrentMap;
import java.util.concurrent.TimeUnit;
import org.elasticsearch.ElasticsearchIllegalArgumentException;
import org.elasticsearch.ElasticsearchIllegalStateException;
import org.elasticsearch.action.ActionListener;
import org.elasticsearch.action.delete.DeleteRequest;
import org.elasticsearch.action.delete.DeleteResponse;
import org.elasticsearch.action.get.GetResponse;
import org.elasticsearch.action.index.IndexRequest;
import org.elasticsearch.action.index.IndexResponse;
import org.elasticsearch.client.Client;
import org.elasticsearch.common.Nullable;
import org.elasticsearch.common.ParseField;
import org.elasticsearch.common.Strings;
import org.elasticsearch.common.base.Charsets;
import org.elasticsearch.common.bytes.BytesReference;
import org.elasticsearch.common.cache.Cache;
import org.elasticsearch.common.cache.CacheBuilder;
import org.elasticsearch.common.collect.ImmutableMap;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.Streams;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.TimeValue;
import org.elasticsearch.common.util.concurrent.ConcurrentCollections;
import org.elasticsearch.common.xcontent.XContentBuilder;
import org.elasticsearch.common.xcontent.XContentFactory;
import org.elasticsearch.common.xcontent.XContentParser;
import org.elasticsearch.common.xcontent.XContentType;
import org.elasticsearch.env.Environment;
import org.elasticsearch.index.VersionType;
import org.elasticsearch.index.fielddata.IndexFieldDataService;
import org.elasticsearch.index.mapper.MapperService;
import org.elasticsearch.index.query.TemplateQueryParser;
import org.elasticsearch.script.AbstractFloatSearchScript;
import org.elasticsearch.script.CompiledScript;
import org.elasticsearch.script.ExecutableScript;
import org.elasticsearch.script.NativeScriptFactory;
import org.elasticsearch.script.ScriptEngineService;
import org.elasticsearch.script.ScriptException;
import org.elasticsearch.script.SearchScript;
import org.elasticsearch.search.lookup.SearchLookup;
import org.elasticsearch.watcher.FileChangesListener;
import org.elasticsearch.watcher.FileWatcher;
import org.elasticsearch.watcher.ResourceWatcherService;

public class ScriptService
extends AbstractComponent {
    public static final String DEFAULT_SCRIPTING_LANGUAGE_SETTING = "script.default_lang";
    public static final String DISABLE_DYNAMIC_SCRIPTING_SETTING = "script.disable_dynamic";
    public static final String DISABLE_DYNAMIC_SCRIPTING_DEFAULT = "sandbox";
    public static final String SCRIPT_INDEX = ".scripts";
    private static String defaultLang = "groovy";
    private final ImmutableMap<String, ScriptEngineService> scriptEngines;
    private final ConcurrentMap<String, CompiledScript> staticCache = ConcurrentCollections.newConcurrentMap();
    private final Cache<CacheKey, CompiledScript> cache;
    private final File scriptsDirectory;
    private final DynamicScriptDisabling dynamicScriptingDisabled;
    private Client client = null;
    public static final ParseField SCRIPT_LANG = new ParseField("lang", "script_lang");
    public static final ParseField VALUE_SCRIPT_FILE = new ParseField("value_script_file", new String[0]);
    public static final ParseField VALUE_SCRIPT_ID = new ParseField("value_script_id", new String[0]);
    public static final ParseField VALUE_SCRIPT_INLINE = new ParseField("value_script", new String[0]);
    public static final ParseField KEY_SCRIPT_FILE = new ParseField("key_script_file", new String[0]);
    public static final ParseField KEY_SCRIPT_ID = new ParseField("key_script_id", new String[0]);
    public static final ParseField KEY_SCRIPT_INLINE = new ParseField("key_script", new String[0]);
    public static final ParseField SCRIPT_FILE = new ParseField("script_file", "file");
    public static final ParseField SCRIPT_ID = new ParseField("script_id", "id");
    public static final ParseField SCRIPT_INLINE = new ParseField("script", "scriptField");

    @Inject
    public ScriptService(Settings settings, Environment env, Set<ScriptEngineService> scriptEngines, ResourceWatcherService resourceWatcherService) {
        super(settings);
        int cacheMaxSize = this.componentSettings.getAsInt("cache.max_size", (Integer)500);
        TimeValue cacheExpire = this.componentSettings.getAsTime("cache.expire", null);
        this.logger.debug("using script cache with max_size [{}], expire [{}]", cacheMaxSize, cacheExpire);
        defaultLang = settings.get(DEFAULT_SCRIPTING_LANGUAGE_SETTING, "mvel");
        this.dynamicScriptingDisabled = DynamicScriptDisabling.parse(settings.get(DISABLE_DYNAMIC_SCRIPTING_SETTING, DISABLE_DYNAMIC_SCRIPTING_DEFAULT));
        CacheBuilder<Object, Object> cacheBuilder = CacheBuilder.newBuilder();
        if (cacheMaxSize >= 0) {
            cacheBuilder.maximumSize(cacheMaxSize);
        }
        if (cacheExpire != null) {
            cacheBuilder.expireAfterAccess(cacheExpire.nanos(), TimeUnit.NANOSECONDS);
        }
        this.cache = cacheBuilder.build();
        ImmutableMap.Builder<String, ScriptEngineService> builder = ImmutableMap.builder();
        for (ScriptEngineService scriptEngine : scriptEngines) {
            for (String type : scriptEngine.types()) {
                builder.put(type, scriptEngine);
            }
        }
        this.scriptEngines = builder.build();
        this.staticCache.put("doc.score", new CompiledScript("native", new DocScoreNativeScriptFactory()));
        this.scriptsDirectory = new File(env.configFile(), "scripts");
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Using scripts directory [{}] ", this.scriptsDirectory);
        }
        FileWatcher fileWatcher = new FileWatcher(this.scriptsDirectory);
        fileWatcher.addListener(new ScriptChangesListener());
        if (this.componentSettings.getAsBoolean("auto_reload_enabled", (Boolean)true).booleanValue()) {
            resourceWatcherService.add(fileWatcher);
        } else {
            fileWatcher.init();
        }
    }

    @Inject(optional=true)
    public void setClient(Client client) {
        this.client = client;
    }

    public void close() {
        for (ScriptEngineService engineService : this.scriptEngines.values()) {
            engineService.close();
        }
    }

    public CompiledScript compile(String script) {
        return this.compile(defaultLang, script);
    }

    public CompiledScript compile(String lang, String script) {
        return this.compile(lang, script, ScriptType.INLINE);
    }

    public CompiledScript compile(String lang, String script, ScriptType scriptType) {
        CompiledScript compiled;
        if (this.logger.isTraceEnabled()) {
            this.logger.trace("Compiling lang: [{}] type: [{}] script: {}", new Object[]{lang, scriptType, script});
        }
        if (scriptType == ScriptType.INDEXED) {
            if (this.client == null) {
                throw new ElasticsearchIllegalArgumentException("Got an indexed script with no Client registered.");
            }
            IndexedScript indexedScript = new IndexedScript(lang, script);
            if (lang != null && !lang.equals(indexedScript.lang)) {
                this.logger.trace("Overriding lang to " + indexedScript.lang, new Object[0]);
                lang = indexedScript.lang;
            }
            this.verifyDynamicScripting(indexedScript.lang);
            script = this.getScriptFromIndex(this.client, indexedScript.lang, indexedScript.id);
        } else if (scriptType == ScriptType.FILE) {
            CompiledScript compiled2 = (CompiledScript)this.staticCache.get(script);
            if (compiled2 != null) {
                return compiled2;
            }
            throw new ElasticsearchIllegalArgumentException("Unable to find on disk script " + script);
        }
        if (scriptType != ScriptType.INDEXED && (compiled = (CompiledScript)this.staticCache.get(script)) != null) {
            return compiled;
        }
        if (lang == null) {
            lang = defaultLang;
        }
        this.verifyDynamicScripting(lang);
        CacheKey cacheKey = new CacheKey(lang, script);
        compiled = this.cache.getIfPresent(cacheKey);
        if (compiled != null) {
            return compiled;
        }
        if (!this.dynamicScriptEnabled(lang)) {
            throw new ScriptException("dynamic scripting for [" + lang + "] disabled");
        }
        if (cacheKey == null) {
            cacheKey = new CacheKey(lang, script);
        }
        compiled = this.getCompiledScript(lang, script);
        this.cache.put(cacheKey, compiled);
        return compiled;
    }

    private CompiledScript getCompiledScript(String lang, String script) {
        ScriptEngineService service = this.scriptEngines.get(lang);
        if (service == null) {
            throw new ElasticsearchIllegalArgumentException("script_lang not supported [" + lang + "]");
        }
        CompiledScript compiled = new CompiledScript(lang, service.compile(script));
        return compiled;
    }

    private void verifyDynamicScripting(String lang) {
        if (!this.dynamicScriptEnabled(lang)) {
            throw new ScriptException("dynamic scripting for [" + lang + "] disabled");
        }
    }

    public GetResponse queryScriptIndex(Client client, String scriptLang, String id) {
        return this.queryScriptIndex(client, scriptLang, id, -3L, VersionType.INTERNAL);
    }

    public GetResponse queryScriptIndex(Client client, String scriptLang, String id, long version, VersionType versionType) {
        scriptLang = this.validateScriptLanguage(scriptLang);
        return (GetResponse)client.prepareGet(SCRIPT_INDEX, scriptLang, id).setVersion(version).setVersionType(versionType).get();
    }

    private String validateScriptLanguage(String scriptLang) {
        if (scriptLang == null) {
            scriptLang = defaultLang;
        } else if (!this.scriptEngines.containsKey(scriptLang)) {
            throw new ElasticsearchIllegalArgumentException("script_lang not supported [" + scriptLang + "]");
        }
        return scriptLang;
    }

    private String getScriptFromIndex(Client client, String scriptLang, String id) {
        GetResponse responseFields = this.queryScriptIndex(client, scriptLang, id);
        if (responseFields.isExists()) {
            return ScriptService.getScriptFromResponse(responseFields);
        }
        throw new ElasticsearchIllegalArgumentException("Unable to find script [.scripts/" + scriptLang + "/" + id + "]");
    }

    private void validate(BytesReference scriptBytes, String scriptLang) throws IOException {
        block4: {
            XContentParser parser = XContentFactory.xContent(scriptBytes).createParser(scriptBytes);
            TemplateQueryParser.TemplateContext context = TemplateQueryParser.parse(parser, "params", "script", "template");
            if (Strings.hasLength(context.template())) {
                try {
                    CompiledScript compiledScript = this.compile(scriptLang, context.template(), ScriptType.INLINE);
                    if (compiledScript == null) {
                        throw new ElasticsearchIllegalArgumentException("Unable to parse [" + context.template() + "] lang [" + scriptLang + "] (ScriptService.compile returned null)");
                    }
                    break block4;
                }
                catch (Exception e) {
                    throw new ElasticsearchIllegalArgumentException("Unable to parse [" + context.template() + "] lang [" + scriptLang + "]", e);
                }
            }
            throw new ElasticsearchIllegalArgumentException("Unable to find script in : " + scriptBytes.toUtf8());
        }
    }

    public void putScriptToIndex(Client client, BytesReference scriptBytes, @Nullable String scriptLang, String id, @Nullable TimeValue timeout, @Nullable String sOpType, long version, VersionType versionType, ActionListener<IndexResponse> listener) {
        try {
            scriptLang = this.validateScriptLanguage(scriptLang);
            this.validate(scriptBytes, scriptLang);
            IndexRequest indexRequest = new IndexRequest(SCRIPT_INDEX, scriptLang, id);
            indexRequest.listenerThreaded(false);
            indexRequest.operationThreaded(false);
            indexRequest.version(version);
            indexRequest.versionType(versionType);
            indexRequest.refresh(true);
            indexRequest.source(scriptBytes, true);
            if (timeout != null) {
                indexRequest.timeout(timeout);
            }
            if (sOpType != null) {
                indexRequest.opType(IndexRequest.OpType.fromString(sOpType));
            }
            client.index(indexRequest, listener);
        }
        catch (Throwable t) {
            listener.onFailure(t);
        }
    }

    public void deleteScriptFromIndex(Client client, @Nullable String scriptLang, String id, long version, ActionListener<DeleteResponse> listener) {
        scriptLang = this.validateScriptLanguage(scriptLang);
        client.delete(new DeleteRequest(SCRIPT_INDEX, scriptLang, id).refresh(true).version(version), listener);
    }

    public static String getScriptFromResponse(GetResponse responseFields) {
        Map<String, Object> source = responseFields.getSourceAsMap();
        if (source.containsKey("template")) {
            try {
                XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON);
                Object template = source.get("template");
                if (template instanceof Map) {
                    builder.map((Map)template);
                    return builder.string();
                }
                return template.toString();
            }
            catch (IOException | ClassCastException e) {
                throw new ElasticsearchIllegalStateException("Unable to parse " + responseFields.getSourceAsString() + " as json", e);
            }
        }
        if (source.containsKey("script")) {
            return source.get("script").toString();
        }
        try {
            XContentBuilder builder = XContentFactory.contentBuilder(XContentType.JSON);
            builder.map(responseFields.getSource());
            return builder.string();
        }
        catch (IOException | ClassCastException e) {
            throw new ElasticsearchIllegalStateException("Unable to parse " + responseFields.getSourceAsString() + " as json", e);
        }
    }

    public ExecutableScript executable(String lang, String script, ScriptType scriptType, Map vars) {
        return this.executable(this.compile(lang, script, scriptType), vars);
    }

    public ExecutableScript executable(CompiledScript compiledScript, Map vars) {
        return this.scriptEngines.get(compiledScript.lang()).executable(compiledScript.compiled(), vars);
    }

    public SearchScript search(CompiledScript compiledScript, SearchLookup lookup, @Nullable Map<String, Object> vars) {
        return this.scriptEngines.get(compiledScript.lang()).search(compiledScript.compiled(), lookup, vars);
    }

    public SearchScript search(SearchLookup lookup, String lang, String script, ScriptType scriptType, @Nullable Map<String, Object> vars) {
        return this.search(this.compile(lang, script, scriptType), lookup, vars);
    }

    public SearchScript search(MapperService mapperService, IndexFieldDataService fieldDataService, String lang, String script, ScriptType scriptType, @Nullable Map<String, Object> vars) {
        return this.search(this.compile(lang, script), new SearchLookup(mapperService, fieldDataService, null), vars);
    }

    public Object execute(CompiledScript compiledScript, Map vars) {
        return this.scriptEngines.get(compiledScript.lang()).execute(compiledScript.compiled(), vars);
    }

    public void clear() {
        this.cache.invalidateAll();
    }

    private boolean dynamicScriptEnabled(String lang) {
        ScriptEngineService service = this.scriptEngines.get(lang);
        if (service == null) {
            throw new ElasticsearchIllegalArgumentException("script_lang not supported [" + lang + "]");
        }
        if (this.dynamicScriptingDisabled == DynamicScriptDisabling.EVERYTHING_ALLOWED || "native".equals(lang) || "mustache".equals(lang)) {
            return true;
        }
        if (this.dynamicScriptingDisabled == DynamicScriptDisabling.ONLY_DISK_ALLOWED) {
            return false;
        }
        return service.sandboxed();
    }

    public static class DocScoreSearchScript
    extends AbstractFloatSearchScript {
        @Override
        public float runAsFloat() {
            try {
                return this.doc().score();
            }
            catch (IOException e) {
                return 0.0f;
            }
        }
    }

    public static class DocScoreNativeScriptFactory
    implements NativeScriptFactory {
        @Override
        public ExecutableScript newScript(@Nullable Map<String, Object> params) {
            return new DocScoreSearchScript();
        }
    }

    public static final class CacheKey {
        public final String lang;
        public final String script;

        public CacheKey(String lang, String script) {
            this.lang = lang;
            this.script = script;
        }

        public boolean equals(Object o) {
            CacheKey other = (CacheKey)o;
            return this.lang.equals(other.lang) && this.script.equals(other.script);
        }

        public int hashCode() {
            return this.lang.hashCode() + 31 * this.script.hashCode();
        }
    }

    private class ScriptChangesListener
    extends FileChangesListener {
        private ScriptChangesListener() {
        }

        private Tuple<String, String> scriptNameExt(File file) {
            String scriptPath = ScriptService.this.scriptsDirectory.toURI().relativize(file.toURI()).getPath();
            int extIndex = scriptPath.lastIndexOf(46);
            if (extIndex != -1) {
                String ext = scriptPath.substring(extIndex + 1);
                String scriptName = scriptPath.substring(0, extIndex).replace(File.separatorChar, '_');
                return new Tuple<String, String>(scriptName, ext);
            }
            return null;
        }

        @Override
        public void onFileInit(File file) {
            Tuple<String, String> scriptNameExt;
            if (ScriptService.this.logger.isTraceEnabled()) {
                ScriptService.this.logger.trace("Loading script file : [{}]", file);
            }
            if ((scriptNameExt = this.scriptNameExt(file)) != null) {
                boolean found = false;
                for (ScriptEngineService engineService : ScriptService.this.scriptEngines.values()) {
                    for (String s : engineService.extensions()) {
                        if (!s.equals(scriptNameExt.v2())) continue;
                        found = true;
                        try {
                            ScriptService.this.logger.info("compiling script file [{}]", file.getAbsolutePath());
                            String script = Streams.copyToString(new InputStreamReader((InputStream)new FileInputStream(file), Charsets.UTF_8));
                            ScriptService.this.staticCache.put(scriptNameExt.v1(), new CompiledScript(engineService.types()[0], engineService.compile(script)));
                        }
                        catch (Throwable e) {
                            ScriptService.this.logger.warn("failed to load/compile script [{}]", e, scriptNameExt.v1());
                        }
                        break;
                    }
                    if (!found) continue;
                    break;
                }
                if (!found) {
                    ScriptService.this.logger.warn("no script engine found for [{}]", scriptNameExt.v2());
                }
            }
        }

        @Override
        public void onFileCreated(File file) {
            this.onFileInit(file);
        }

        @Override
        public void onFileDeleted(File file) {
            Tuple<String, String> scriptNameExt = this.scriptNameExt(file);
            ScriptService.this.logger.info("removing script file [{}]", file.getAbsolutePath());
            ScriptService.this.staticCache.remove(scriptNameExt.v1());
        }

        @Override
        public void onFileChanged(File file) {
            this.onFileInit(file);
        }
    }

    static class IndexedScript {
        String lang;
        String id;

        IndexedScript(String lang, String script) {
            this.lang = lang;
            String[] parts = script.split("/");
            if (parts.length == 1) {
                this.id = script;
                if (this.lang == null) {
                    this.lang = defaultLang;
                }
            } else {
                if (parts.length != 3) {
                    throw new ElasticsearchIllegalArgumentException("Illegal index script format [" + script + "]" + " should be /lang/id");
                }
                if (!parts[1].equals(this.lang)) {
                    throw new ElasticsearchIllegalStateException("Conflicting script language, found [" + parts[1] + "] expected + [" + this.lang + "]");
                }
                this.id = parts[2];
            }
        }
    }

    public static enum ScriptType {
        INLINE,
        INDEXED,
        FILE;

        private static final int INLINE_VAL = 0;
        private static final int INDEXED_VAL = 1;
        private static final int FILE_VAL = 2;

        public static ScriptType readFrom(StreamInput in) throws IOException {
            int scriptTypeVal = in.readVInt();
            switch (scriptTypeVal) {
                case 1: {
                    return INDEXED;
                }
                case 0: {
                    return INLINE;
                }
                case 2: {
                    return FILE;
                }
            }
            throw new ElasticsearchIllegalArgumentException("Unexpected value read for ScriptType got [" + scriptTypeVal + "] expected one of [" + 0 + "," + 1 + "," + 2 + "]");
        }

        public static void writeTo(ScriptType scriptType, StreamOutput out) throws IOException {
            if (scriptType != null) {
                switch (scriptType) {
                    case INDEXED: {
                        out.writeVInt(1);
                        return;
                    }
                    case INLINE: {
                        out.writeVInt(0);
                        return;
                    }
                    case FILE: {
                        out.writeVInt(2);
                        return;
                    }
                }
                throw new ElasticsearchIllegalStateException("Unknown ScriptType " + (Object)((Object)scriptType));
            }
            out.writeVInt(0);
        }
    }

    static enum DynamicScriptDisabling {
        EVERYTHING_ALLOWED,
        ONLY_DISK_ALLOWED,
        SANDBOXED_ONLY;


        public static final DynamicScriptDisabling parse(String s) {
            switch (s.toLowerCase(Locale.ROOT)) {
                case "true": 
                case "all": {
                    return ONLY_DISK_ALLOWED;
                }
                case "false": 
                case "none": {
                    return EVERYTHING_ALLOWED;
                }
                case "sandbox": 
                case "sandboxed": {
                    return SANDBOXED_ONLY;
                }
            }
            throw new ElasticsearchIllegalArgumentException("Unrecognized script allowance setting: [" + s + "]");
        }
    }
}

