package com.xebialabs.xltest.domain;

import java.io.FileNotFoundException;
import java.util.*;
import java.util.concurrent.ExecutorService;
import java.util.regex.Matcher;
import java.util.regex.Pattern;
import javax.script.Bindings;
import javax.script.ScriptContext;
import javax.script.SimpleBindings;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import com.google.common.base.Strings;
import com.google.common.collect.Sets;
import com.samskivert.mustache.Mustache;
import com.samskivert.mustache.Template;

import com.xebialabs.deployit.plugin.api.reflect.PropertyDescriptor;
import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.overthere.CmdLine;
import com.xebialabs.overthere.RuntimeIOException;
import com.xebialabs.xltest.repository.ScriptExecutionException;
import com.xebialabs.xltest.repository.TestRunsRepository;
import com.xebialabs.xltest.repository.TestTools;
import com.xebialabs.xltest.view.ExecutionParameterRequest;

@Metadata(description = "A test specification with execution capabilities", root = Metadata.ConfigurationItemRoot.CONFIGURATION)
public class ExecutableTestSpecification extends TestSpecification implements Executable {

    private static final Logger LOG = LoggerFactory.getLogger(ExecutableTestSpecification.class);

    @Property(description = "Command line to execute test specification", required = false, category = BaseTestSpecification.CATEGORY_EXECUTION)
    private String commandLine;

    @Property(description = "The maximum number of minutes this test specification may execute", required = false, category = BaseTestSpecification.CATEGORY_EXECUTION)
    private Integer timeout;

    @Property(description = "download the test results itself (true) or expected test results to be delivered (false)", required = false, defaultValue = "true", hidden = false)
    private boolean importTestResults;

    @Property(description = "location of the script on the class path", required = false, defaultValue = "", hidden = true)
    private String scriptLocation;

    @Autowired
    private transient TestTools testTools;

    @Autowired
    private transient ExecutorService executorService;

    @Autowired
    private transient TestRunsRepository testRunsRepository;

    @Override
    public void execute(UUID testRunId, Map<String, Object> parameters, final EventHandler eventHandler) {
        TestTool testTool = testTools.findByName(getTestToolName());
        String resolvedCommandline = resolve(getCommandLine(), parameters);
        execute(testRunId, executorService, testTool, resolvedCommandline, parameters, new EventHandler() {
            @Override
            public void onReceive(Event event) throws Exception {
                eventHandler.onReceive(new Event(event, Event.props(Event.TEST_SPECIFICATION, getName())));
            }
        });

        if (isImportTestResults()) {
            final Qualifier qualifier = getQualifierIfAny(testRunId);

            doImport(testRunId, testTool, new EventHandler() {

                @Override
                public void onReceive(Event event) throws Exception {
                    if (qualifier != null) {
                        qualifier.update(event);
                    }
                    eventHandler.onReceive(event);
                }
            });

            try {
                eventHandler.onReceive(new Event(qualifier.getEvent(), Event.props(Event.TEST_SPECIFICATION, getName())));
            } catch (Exception e) {
                LOG.error("Unable to handle qualification event", e);
            }
        }
    }

    @Override
    public boolean updateExecutionProperties(String commandLine, Host host) {
        if (host != null) {
            LOG.debug("Updating host to {}", host);
            setHost(host);
        }
        if (commandLine != null) {
            LOG.debug("Updating commandline to {}", commandLine);
            this.commandLine = commandLine;
        }
        return isExecutable();
    }

    @Override
    public ExecutionParameterRequest toExecutionParameterRequest() {
        return new ExecutionParameterRequest(commandLine, getHost() != null ? getHost().getId() : null, getParameters());
    }

    private Map<String, Set<String>> getParameters() {
        // create a k,v map out of all properties in category called 'parameters'
        Map<String, Set<String>> parameterMap = new HashMap<>();
        Collection<PropertyDescriptor> propertyDescriptors = this.getType().getDescriptor().getPropertyDescriptors();
        for (PropertyDescriptor propertyDescriptor : propertyDescriptors) {
            if (BaseTestSpecification.CATEGORY_PARAMETERS.equalsIgnoreCase(propertyDescriptor.getCategory())) {
                String key = propertyDescriptor.getName();
                String value = (String) propertyDescriptor.get(this);
                parameterMap.put(key, Sets.newHashSet(value));
            }
        }
        return parameterMap;
    }

    private Qualifier getQualifierIfAny(UUID testRunId) {
        if (getQualification() == null) {
            return null;
        }
        return getQualification().getQualifier(testRunId, testRunsRepository);
    }

    protected void execute(UUID testRunId, ExecutorService executorService, TestTool testTool,
                           String resolvedCommandline, Map<String, Object> parameters, EventHandler eventHandler) {
        Bindings binding = new SimpleBindings();
        binding.put("testRunId", testRunId);
        binding.put("executorService", executorService);
        binding.put("self", this);
        binding.put("resolvedCommandLine", resolvedCommandline);
        binding.put("testTool", testTool);
        binding.put("eventHandler", eventHandler);
        binding.put("parameters", parameters == null ? new HashMap<String, Object>() : parameters);

        ScriptedConfigurationItem scriptedConfigurationItem = new ScriptedConfigurationItem();
        scriptedConfigurationItem.setScriptLocation(this.scriptLocation);
        ScriptContext context = scriptedConfigurationItem.newScriptContext();
        context.setBindings(binding, ScriptContext.ENGINE_SCOPE);
        try {
            scriptedConfigurationItem.execute(context, this);
        } catch (ScriptExecutionException e) {
            LOG.error("Unable to execute script", e);
        } catch (FileNotFoundException e) {
            throw new RuntimeIOException("No execution script found", e);
        }
    }

    protected String resolve(String commandLine, Map<String, Object> parameters) {
        if (commandLine == null || commandLine.equals("")) {
            return commandLine;
        }
        Template tmpl = Mustache.compiler().compile(commandLine);
        return tmpl.execute(parameters);
    }

    public static String[] parseCommandLine(String commandLine) {
        if (commandLine == null) {
            return new String[]{""};
        }
        List<String> result = new ArrayList<>();
        Pattern p = Pattern.compile("\"([^\"]*)\"|[\\S]+");
        Matcher m = p.matcher(commandLine);
        while (m.find()) {
            String token = (m.group(1) == null) ? m.group(0) : m.group(1);
            result.add(token);
        }
        return result.toArray(new String[result.size()]);
    }

    public static boolean isExecutable(String commandLine, Host host) {
        return !Strings.isNullOrEmpty(commandLine) && host != null;
    }

    @Override
    public boolean isExecutable() {
        return isExecutable(commandLine, getHost());
    }

    public String getCommandLine() {
        return commandLine;
    }

    public void setCommandLine(String commandLine) {
        this.commandLine = commandLine;
    }

    public int getTimeout() {
        return timeout;
    }

    public void setTimeout(int timeout) {
        this.timeout = timeout;
    }

    public void setTestTools(TestTools testTools) {
        this.testTools = testTools;
    }

    public void setExecutorService(ExecutorService executorService) {
        this.executorService = executorService;
    }

    public void setTestRunsRepository(TestRunsRepository testRunsRepository) {
        this.testRunsRepository = testRunsRepository;
    }

    public CmdLine getCmdLine() {
        return CmdLine.build(parseCommandLine(commandLine));
    }

    public CmdLine getCmdLine(String resolvedCommandLine) {
        return CmdLine.build(parseCommandLine(resolvedCommandLine));
    }

    public boolean isImportTestResults() {
        return importTestResults;
    }

    public void setImportTestResults(boolean importTestResults) {
        this.importTestResults = importTestResults;
    }
}
