package com.xebialabs.xltest.domain;

import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.beans.factory.config.AutowireCapableBeanFactory;

import com.xebialabs.deployit.plugin.api.udm.Metadata;
import com.xebialabs.deployit.plugin.api.udm.Property;
import com.xebialabs.deployit.plugin.overthere.Host;
import com.xebialabs.xltest.view.ExecutionParameterRequest;

import static com.xebialabs.xltest.domain.Event.CHILD_FINISHED;
import static com.xebialabs.xltest.domain.Event.CHILD_RUN_ID;
import static com.xebialabs.xltest.domain.Event.CHILD_STARTED;
import static java.util.Collections.emptyList;

@Metadata(description = "Grouping construct for test specification", root = Metadata.ConfigurationItemRoot.CONFIGURATION)
public class TestSpecificationSet extends BaseTestSpecification implements Executable {

    private static final Logger LOG = LoggerFactory.getLogger(TestSpecificationSet.class);

    @Property
    private List<BaseTestSpecification> testSpecifications = emptyList();

    @Property(description = "The report used to determine the outcome of a test run (success/fail)", required = false)
    private Qualification qualification;

    @Autowired
    private transient AutowireCapableBeanFactory beanFactory;

    @Override
    public boolean isExecutable() {
        return true;
    }

    @Override
    public void execute(UUID testRunId, final Map<String, Object> parameters, final EventHandler eventHandler) {
        try {
            eventHandler.onReceive(parentStartEvent(testRunId));
            final Qualifier qualifier = new CompositeExecutionQualifier(testRunId);
            for (BaseTestSpecification baseTestSpecification : testSpecifications) {
                if (baseTestSpecification instanceof Executable) {
                    Executable executable = (Executable) baseTestSpecification;
                    beanFactory.autowireBean(executable);
                    UUID childRunId = UUID.randomUUID();
                    eventHandler.onReceive(childStartEvent(testRunId, childRunId));
                    // catch all finish events to inspect the qualification result of the child
                    executable.execute(childRunId, parameters, new EventHandler() {
                        @Override
                        public void onReceive(Event event) throws Exception {
                            eventHandler.onReceive(event);
                            qualifier.update(event);
                        }
                    });
                    eventHandler.onReceive(childFinishedEvent(testRunId, childRunId));
                }
            }
            // TODO: This is possibly wrong, we're checking a composite executor, not a testSpec!
            eventHandler.onReceive(new Event(qualifier.getEvent(), Event.props(Event.TEST_SPECIFICATION, getName())));
        } catch (Exception e) {
            throw new RuntimeException(e);
        } finally {
            try {
                eventHandler.onReceive(parentFinishedEvent(testRunId));
            } catch (Exception e) {
                LOG.error("Unable to handle finish event", e);
            }
        }
    }

    @Override
    public boolean updateExecutionProperties(String commandLine, Host hostId) {
        // does nothing
        return false;
    }

    @Override
    public ExecutionParameterRequest toExecutionParameterRequest() {
        return new ExecutionParameterRequest("super", "super", getParameters());
    }

    public Map<String, Set<String>> getParameters() {
        Map<String, Set<String>> parameters = new HashMap<>();
        for (BaseTestSpecification eachChildTestSpec : getTestSpecifications()) {
            if (eachChildTestSpec instanceof Executable) {
                ExecutionParameterRequest executionParameterRequest = ((Executable) eachChildTestSpec).toExecutionParameterRequest();
                mergeParameters(parameters, executionParameterRequest.getParameters());
            }
        }
        return parameters;
    }

    private void mergeParameters(Map<String, Set<String>> parameters, Map<String, Set<String>> testSpecificationParameters) {
        for (Map.Entry<String, Set<String>> paramEntry : testSpecificationParameters.entrySet()) {
            if (parameters.containsKey(paramEntry.getKey())) {
                parameters.get(paramEntry.getKey()).addAll(paramEntry.getValue());
            } else {
                parameters.put(paramEntry.getKey(), paramEntry.getValue());
            }
        }
    }


    private Event childFinishedEvent(UUID testRunId, UUID childRunId) {
        return new Event(testRunId, CHILD_FINISHED, Event.props(
                CHILD_RUN_ID, childRunId.toString(),
                Event.TEST_SPECIFICATION, this.getName()));
    }

    private Event childStartEvent(UUID testRunId, UUID childRunId) {
        return new Event(testRunId, CHILD_STARTED, Event.props(
                CHILD_RUN_ID, childRunId.toString(),
                Event.TEST_SPECIFICATION, this.getName()));
    }

    private Event parentFinishedEvent(UUID testRunId) {
        return new Event(testRunId, Event.EXECUTION_FINISHED, Event.props(
                Event.TEST_SPECIFICATION, this.getName()));
    }

    private Event parentStartEvent(UUID testRunId) {
        return new Event(testRunId, Event.EXECUTION_STARTED, Event.props(
                Event.TEST_SPECIFICATION, this.getName()));
    }

    private static class CompositeExecutionQualifier implements Qualifier {

        private final UUID testRunId;
        Boolean qualification = true;

        public CompositeExecutionQualifier(UUID testRunId) {
            this.testRunId = testRunId;
        }

        @Override
        public void update(Event event) {
            if (qualification && Event.QUALIFICATION_COMPUTED.equals(event.getType())) {
                qualification = qualification && (Boolean) event.get("qualification");
            }
        }

        @Override
        public Boolean getQualificationResult() {
            return qualification;
        }

        @Override
        public Event getEvent() {
            return new Event(testRunId, Event.QUALIFICATION_COMPUTED,
                    Event.props(Event.QUALIFICATION, qualification));
        }
    }

    public List<BaseTestSpecification> getTestSpecifications() {
        return testSpecifications;
    }

    public void setTestSpecifications(List<BaseTestSpecification> testSpecifications) {
        this.testSpecifications = testSpecifications;
    }

    public Qualification getQualification() {
        return this.qualification;
    }

    public void setQualification(Qualification qualification) {
        this.qualification = qualification;
    }
}
