package com.xebialabs.xltest.service;

import java.io.*;
import java.util.List;
import java.util.Map;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.collect.Maps;

import com.xebialabs.deployit.plugin.api.reflect.Type;
import com.xebialabs.deployit.plugin.api.udm.ConfigurationItem;
import com.xebialabs.deployit.repository.*;
import com.xebialabs.deployit.repository.internal.Root;
import com.xebialabs.xltest.domain.TestRun;

import static com.google.common.collect.Lists.newArrayList;

public class InMemoryRepositoryService implements RepositoryService, Serializable {
    private Map<String, ConfigurationItem> store = Maps.newHashMap();

    private static Logger LOG = LoggerFactory.getLogger(InMemoryRepositoryService.class);

    public InMemoryRepositoryService() {
        store.put("Applications", Type.valueOf(Root.class).getDescriptor().newInstance("Applications"));
    }

    @Override
    public boolean exists(final String id) {
        return store.containsKey(id);
    }

    @SuppressWarnings("unchecked")
    @Override
    public <T extends ConfigurationItem> T read(final String id) {
        ConfigurationItem ci = store.get(id);
        return (T) defensiveCopy(ci);
    }

    @SuppressWarnings("unchecked")
    private <T extends ConfigurationItem> T defensiveCopy(T ci) {
        try {
            ByteArrayOutputStream bytes = new ByteArrayOutputStream();
            new ObjectOutputStream(bytes).writeObject(ci);
            T copiedObject = (T) new ObjectInputStream(new ByteArrayInputStream(bytes.toByteArray())).readObject();
            return copiedObject;
        } catch (Exception e) {
            e.printStackTrace();
            throw new RuntimeException(e);
        }
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, WorkDir workDir) {
        return this.<T>read(id);
    }

    @Override
    public <T extends ConfigurationItem> void create(final T... entities) {
        checkThatEntitiesDoNotExist(entities);
        storeEntities(entities);
    }

    private void checkThatEntitiesDoNotExist(final ConfigurationItem[] entities) {
        for (ConfigurationItem entity : entities) {
            if (store.containsKey(entity.getId())) {
                throw new IllegalStateException("Entity " + entity.getId() + " already exists and cannot be created twice.");
            }
        }
    }

    private void storeEntities(final ConfigurationItem[] entities) {
        for (ConfigurationItem entity : entities) {
            store.put(entity.getId(), defensiveCopy(entity));
        }
    }

    @Override
    public <T extends ConfigurationItem> void update(final T... entities) {
        for (T entity : entities) {
            T defensiveCopy = defensiveCopy(entity);
            if (entity instanceof TestRun) {
                TestRun tr = (TestRun) entity;
                TestRun copiedTr = (TestRun) defensiveCopy;
            }
            store.put(entity.getId(), defensiveCopy);
        }

    }

    @Override
    public <T extends ConfigurationItem> void move(String id, String newId) {
        ConfigurationItem remove = store.remove(id);
        remove.setId(newId);
        store.put(newId, defensiveCopy(remove));
    }

    @Override
    public <T extends ConfigurationItem> void copy(String id, String newId) {
        ConfigurationItem itemToCopy = store.get(id);
        ConfigurationItem copiedItem = defensiveCopy(itemToCopy);
        copiedItem.setId(newId);
        store.put(newId, copiedItem);
    }

    @Override
    public <T extends ConfigurationItem> void createOrUpdate(T... entities) {
        for (T entity : entities) {
            store.put(entity.getId(), defensiveCopy(entity));
        }
    }

    @Override
    public void rename(String id, String newName) {
        ConfigurationItem remove = store.remove(id);
        remove.setId(id.substring(0, id.lastIndexOf('/') + 1) + newName);
        store.put(remove.getId(), remove);
    }

    @Override
    public void delete(final String... ids) {
        for (String id : ids) {
            store.remove(id);
        }
    }

    @Override
    public void execute(ChangeSet batchUpdate) {
        if (!batchUpdate.getCreateCis().isEmpty()) {
            create(batchUpdate.getCreateCis().toArray(new ConfigurationItem[batchUpdate.getCreateCis().size()]));
        }

        if (!batchUpdate.getCreateOrUpdateCis().isEmpty()) {
            createOrUpdate(batchUpdate.getCreateOrUpdateCis().toArray(new ConfigurationItem[batchUpdate.getCreateOrUpdateCis().size()]));
        }

        if (!batchUpdate.getUpdateCis().isEmpty()) {
            update(batchUpdate.getUpdateCis().toArray(new ConfigurationItem[batchUpdate.getUpdateCis().size()]));
        }

        if (!batchUpdate.getDeleteCiIds().isEmpty()) {
            delete(batchUpdate.getDeleteCiIds().toArray(new String[batchUpdate.getDeleteCiIds().size()]));
        }
    }

    @Override
    public List<ConfigurationItemData> list(final SearchParameters criteria) {
        List<ConfigurationItemData> ids = newArrayList();
        if (getType(criteria) != null) {
            for (ConfigurationItem configurationItem : store.values()) {
                if (configurationItem.getType().equals(getType(criteria))) {
                    ids.add(new ConfigurationItemData(configurationItem.getId(), configurationItem.getType()));
                }
            }
            return ids;
        }
        throw new UnsupportedOperationException("Input error?");
    }

    /**
     * Limited search capabilities: only searches on type.
     * <p/>
     * !! It's checking on "instanceOf" type. Does this hold true for JCR searches?
     *
     * @param criteria
     * @param <T>
     * @return items in store for the search parameter type.
     */
    @Override
    public <T extends ConfigurationItem> List<T> listEntities(SearchParameters criteria) {
        List<T> ids = newArrayList();
        Type criteriaType = getType(criteria);
        if (criteriaType != null) {
            for (ConfigurationItem configurationItem : store.values()) {
                if (configurationItem.getType().instanceOf(criteriaType)) {
                    ids.add((T) configurationItem);
                }
            }
            return ids;
        }
        throw new UnsupportedOperationException("Input error?");
    }

    @Override
    public List<ConfigurationItemData> list(JcrQueryTemplate queryTemplate) {
        throw new UnsupportedOperationException("InMemoryRepositoryService.list() not implemented");
    }

    @Override
    public <T extends ConfigurationItem> List<T> listEntities(JcrQueryTemplate queryTemplate) {
        throw new UnsupportedOperationException("InMemoryRepositoryService.listEntities() not implemented");
    }

    @Override
    public void checkReferentialIntegrity(ChangeSet batchUpdate) throws ItemInUseException {
        //do nothing
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, final boolean useCache) {
        return this.<T>read(id);
    }

    @Override
    public <T extends ConfigurationItem> T read(final String id, final WorkDir workDir, final boolean useCache) {
        return this.<T>read(id);
    }

    /**
     * Work around scope in search parameters.
     *
     * @param criteria
     * @return
     */
    Type getType(SearchParameters criteria) {
        return criteria.getType();
    }

    @Override
    public <T extends ConfigurationItem> T read(String id, int depth) {
        // TODO Auto-generated method stub
        return null;
    }

    @Override
    public <T extends ConfigurationItem> T read(String id, int depth, WorkDir workDir, boolean useCache) {
        // TODO Auto-generated method stub
        return null;
    }
}
