package com.xebialabs.xltest.resources;

import java.io.File;
import java.io.IOException;
import java.io.InputStream;
import java.io.PrintWriter;
import java.util.Collections;
import java.util.HashMap;
import java.util.Map;
import java.util.UUID;
import java.util.zip.ZipEntry;
import java.util.zip.ZipInputStream;
import javax.servlet.ServletInputStream;
import javax.servlet.http.HttpServletRequest;
import javax.servlet.http.HttpServletResponse;
import javax.ws.rs.*;
import javax.ws.rs.core.Context;
import javax.ws.rs.core.Response;

import com.xebialabs.xltest.repository.NoTestRunFoundException;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.springframework.beans.factory.annotation.Autowired;
import org.springframework.stereotype.Controller;
import com.google.common.base.Splitter;
import com.google.common.io.ByteStreams;
import com.google.common.io.Files;

import com.xebialabs.deployit.repository.RepositoryService;
import com.xebialabs.overthere.local.LocalFile;
import com.xebialabs.xltest.domain.*;
import com.xebialabs.xltest.repository.TestRunsRepository;
import com.xebialabs.xltest.repository.TestSpecificationRepository;
import com.xebialabs.xltest.repository.TestTools;
import com.xebialabs.xltest.service.EventRepository;
import com.xebialabs.xltest.utils.TestToolCategory;

import static com.google.common.base.Strings.isNullOrEmpty;

@Controller
@Path("/import")
public class ImportController {
    private static final Logger LOG = LoggerFactory.getLogger(ImportController.class);
    public static final String REMOTE_TRACKING_ID = "XL_TEST_RUN_ID";

    private final TestTools testTools;
    private final RepositoryService repository;
    private final TestSpecificationRepository testSpecifications;
    private final EventRepository eventRepository;
    private final TestRunsRepository testRunsRepository;

    @Autowired
    public ImportController(TestTools testTools, RepositoryService repository, EventRepository eventRepository, TestRunsRepository testRunsRepository, TestSpecificationRepository testSpecificationRepository) {
        this.testTools = testTools;
        this.repository = repository;
        this.eventRepository = eventRepository;
        this.testRunsRepository = testRunsRepository;
        this.testSpecifications = testSpecificationRepository;
    }

    @POST
    @Path("/{jobName:.+}")
    public Response importTestResultsSentByJenkins(@PathParam("jobName") String jobName, @Context HttpServletRequest request) throws Exception {

        TestSpecification testSpecification = null;
        UUID testRunId = getTestRunId(request);

        if (testRunId != null) {
            try {
                TestRun testRun = testRunsRepository.getTestRun(testRunId);
                testSpecification = testSpecifications.get(testRun.getTestSpecificationName());
            } catch (NoTestRunFoundException e) {
                LOG.warn("A test run id ({}) was provided, but no run has been found in the database.");
            }
        } else {
            testRunId = UUID.randomUUID();
        }

        if (testSpecification == null) {
            //fall back on searching in database for a best-effort match
            testSpecification = testSpecifications.findOrCreateTestSpecificationWithJenkinsHost(
                    jobName,
                    request.getParameter("tool"),
                    request.getParameter("pattern"),
                    request.getParameter("jenkinsUrl"));
        }

        File localWorkspaceDirectory = extractZipInputStreamToTempLocalWorkspaceDirectory(request);
        final HashMap<String, Object> extraEventProperties = new HashMap<>(toSimpleMap(request.getParameterMap()));
        extraEventProperties.put("jobName", jobName);
        extraEventProperties.put(Event.TEST_SPECIFICATION, testSpecification.getName());

        // 1. do the import, so let testSpecification refer to the result location being the extracted zip
        TestTool testTool = testTools.findByName(testSpecification.getTestToolName());
        setDefaultQualificationIfAbsent(testSpecification, testTool.getCategory());
        final Qualifier qualifier = testSpecification.getQualification().getQualifier(testRunId, testRunsRepository);

        testSpecification.doImport(testRunId, testTool, LocalFile.valueOf(localWorkspaceDirectory),
                new EventHandler() {
                    @Override
                    public void onReceive(Event event) throws Exception {
                        Event eventWithSlaveInfo = new Event(event, extraEventProperties);
                        qualifier.update(eventWithSlaveInfo);
                        eventRepository.insert(eventWithSlaveInfo);
                    }
                });
        eventRepository.insert(new Event(qualifier.getEvent(), extraEventProperties));

        // TODO make this configurable localWorkspaceDirectory.delete();

        return Response.ok().build();
    }

    private Map<String, Object> toSimpleMap(Map<String, String[]> parameterMap) {
        Map<String, Object> resultMap = new HashMap<>(parameterMap.size());
        for (Map.Entry<String, String[]> entry : parameterMap.entrySet()) {
            resultMap.put(entry.getKey(), entry.getValue()[0]);
        }
        return resultMap;
    }

    private UUID getTestRunId(HttpServletRequest request) {
        // We try to use the tracking id used by Jenkins whenever possible.
        String parameterValue = request.getParameter(REMOTE_TRACKING_ID);
        if (!isNullOrEmpty(parameterValue)) {
            String result = Splitter.on('/').omitEmptyStrings().splitToList(parameterValue).get(0);
            return UUID.fromString(result);
        }
        return null;
    }

    private File extractZipInputStreamToTempLocalWorkspaceDirectory(HttpServletRequest request) throws IOException {
        ServletInputStream inputStream = request.getInputStream();
        File localWorkspaceDirectory = Files.createTempDir();
        extractZipInputStream(localWorkspaceDirectory, inputStream);
        inputStream.close();
        LOG.info("Extracted zip content into: " + localWorkspaceDirectory.getAbsolutePath());
        return localWorkspaceDirectory;
    }

    private void extractZipInputStream(File baseDir, InputStream is) throws IOException {
        final ZipInputStream zis = new ZipInputStream(is);
        ZipEntry entry;
        while ((entry = zis.getNextEntry()) != null) {
            if (!entry.isDirectory()) {
                final File file = new File(baseDir, entry.getName());
                LOG.debug("Copying {} into file {}", entry.getName(), file);
                Files.createParentDirs(file);
                Files.write(ByteStreams.toByteArray(zis), file);
            }
        }
    }

    private void setDefaultQualificationIfAbsent(TestSpecification testSpecification, String testToolCategory) {
        if (testSpecification.getQualification() == null) {
            String defaultQualificationId = null;
            if (testToolCategory.equals(TestToolCategory.FUNCTIONAL)) {
                defaultQualificationId = "Configuration/Reports/" + TestSpecification.DEFAULT_FUNCTIONAL_TEST_QUALIFIER;
            } else if (testToolCategory.equals(TestToolCategory.PERFORMANCE)) {
                defaultQualificationId = "Configuration/Reports/" + TestSpecification.DEFAULT_PERFORMANCE_TEST_QUALIFIER;
            }
            try {
                Qualification defaultQualification = repository.read(defaultQualificationId);
                testSpecification.setQualification(defaultQualification);
                repository.update(testSpecification);
                LOG.debug("testSpecification {} updated with default qualification {}", testSpecification.getId(), defaultQualificationId);
            } catch (Exception exception) {
                LOG.warn("Couldn't set default qualification {} on testSpecification {}", defaultQualificationId, testSpecification.getId());
            }
        }
    }

    @GET
    @Path("/{testSpecificationId:.+}")
    @Produces("text/event-stream")
    public void getTestRun(@PathParam("testSpecificationId") String testSpecificationId, @Context HttpServletResponse response) throws Exception {

        response.setContentType("text/event-stream");
        response.setCharacterEncoding("UTF-8");
        response.addHeader("Cache-Control", "no-cache");
        response.addHeader("Connection", "keep-alive");

        final PrintWriter writer = response.getWriter();

        final TestSpecification testSpecification;
        try {
            testSpecification = repository.read("Configuration/TestSpecifications/" + testSpecificationId);
        } catch (com.xebialabs.deployit.exception.NotFoundException e) {
            closeEventStream(writer, e.getMessage());
            return;
        }

        try {
            TestTool testTool = testTools.findByName(testSpecification.getTestToolName());
            setDefaultQualificationIfAbsent(testSpecification, testTool.getCategory());
            final Qualifier[] qualifierHolder = new Qualifier[1];
            qualifierHolder[0] = testSpecification.getQualification().getQualifier(UUID.randomUUID(), testRunsRepository);
            testSpecification.doImport(testTool, testSpecification.getRemotePath(),
                    new EventHandler() {
                        @Override
                        public void onReceive(Event event) throws Exception {
                            qualifierHolder[0].update(event);
                            eventRepository.insert(event);
                        }
                    }, new TestSpecification.ProgressCallback() {
                        @Override
                        public void update(UUID testRunId, int percentage) {
                            sendEvent(writer, percentage);
                            if (testRunId != null) {
                                eventRepository.insert(new Event(testRunId, qualifierHolder[0].getEvent(), Collections.<String, Object>emptyMap()));
                            }
                            qualifierHolder[0] = testSpecification.getQualification().getQualifier(UUID.randomUUID(), testRunsRepository);
                        }
                    });
        } finally {
            closeEventStream(writer);
        }
    }

    private void sendEvent(PrintWriter writer, int pct) {
        if (writer != null) {
            writer.write("event: event\n");
            writer.write("data: {\"percentage\":" + pct + "}\n\n");
            writer.flush();
        }
    }

    private void closeEventStream(PrintWriter writer) {
        if (writer != null) {
            writer.write("event: close\n");
            writer.write("data: {}\n\n");
            writer.close();
        }
    }

    private void closeEventStream(PrintWriter writer, String errorMessage) {
        if (writer != null) {
            writer.write("event: close\n");
            writer.write("data: {\"error\":\"" + errorMessage + "\"}\n\n");
            writer.close();
        }
    }

}
