package com.xebialabs.xltest.testtools.fitnesse;

import java.io.File;
import java.io.FileNotFoundException;
import java.util.*;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import com.google.common.base.Splitter;

import com.xebialabs.xlt.plugin.api.resultparser.ImportException;
import com.xebialabs.xlt.plugin.api.resultparser.ImportFailedException;
import com.xebialabs.xlt.plugin.api.testrun.Event;

import fitnesse.FitNesseContext;
import fitnesse.reporting.history.*;
import fitnesse.wiki.*;
import fitnesse.wiki.fs.FileSystemPage;
import fitnesse.wiki.fs.FileSystemPageFactory;
import util.FileUtil;

import static java.lang.String.format;

public class FitnesseSuiteImporterFactory {
    private static final Logger LOG = LoggerFactory.getLogger(FitnesseSuiteImporterFactory.class);

    private final File resultsDirectory;
    private final FileSystemPage root;
    private final TestHistory testHistory = new TestHistory();

    public FitnesseSuiteImporterFactory(File fitNesseRoot) {
        resultsDirectory = new File(fitNesseRoot, "files/" + FitNesseContext.testResultsDirectoryName);
        root = new FileSystemPageFactory().makePage(fitNesseRoot, "FitNesseRoot", null, new SystemVariableSource());
        testHistory.readHistoryDirectory(resultsDirectory);
    }

    public List<FitNesseSuiteImporter> create() throws ImportFailedException {
        List<FitNesseSuiteImporter> toImport = new ArrayList<>();

        File[] resultFiles = resultsDirectory.listFiles();
        if (resultFiles == null) {
            return Collections.emptyList();
        }

        try {
            for (File pageNamedFile : resultFiles) {
                if (!pageNamedFile.isDirectory()) {
                    continue;
                }

                // check if it contains test results or suite results
                String pageName = pageNamedFile.getName();
                LOG.trace("Will try to make report for " + pageName);

                PageHistory pageHistory = getPageHistory(resultsDirectory, pageName);

                if (pageHistoryIsSuite(pageHistory)) {
                    for (Date date : pageHistory.datesInChronologicalOrder()) {
                        ExecutionReport executionReport = getExecutionReport(pageHistory, date);
                        if (executionReport instanceof SuiteExecutionReport) {
                            FitNesseSuiteImporter fitNesseSuiteImporter = new FitNesseSuiteImporter(this, (SuiteExecutionReport) executionReport, pageName);
                            toImport.add(fitNesseSuiteImporter);
                        }
                    }
                }
            }
        } catch (Exception e) {
            throw new ImportFailedException(e);
        }

        /* NOTE: the getLastModified() of fitnesse is in the local timezone of the machine running this code. */
        Collections.sort(toImport, new Comparator<FitNesseSuiteImporter>() {
            @Override
            public int compare(FitNesseSuiteImporter o1, FitNesseSuiteImporter o2) {
                int cmp = Long.valueOf(o1.getLastModified()).compareTo(o2.getLastModified());
                if (cmp != 0) {
                    return cmp;
                }
                return o1.getPageName().compareTo(o2.getPageName());
            }
        });

        return toImport;
    }

    public List<Event> produceEventsForReport(String testRunId, SuiteExecutionReport suiteExecutionReport) throws ImportException {
        List<Event> result = new ArrayList<>();
        for (SuiteExecutionReport.PageHistoryReference reference : suiteExecutionReport.getPageHistoryReferences()) {
            Set<String> tags = findTags(reference);
            String firstErrorMessage = findFirstErrorMessage(reference);

            TestSummary testSummary = new TestSummary(
                extractHierarchy(reference.getPageName(), '.'),
                tags,
                reference.getTime(),
                (int) reference.getRunTimeInMillis(),
                reference.getTestSummary().getRight(),
                reference.getTestSummary().getWrong(),
                reference.getTestSummary().getExceptions(),
                firstErrorMessage);

            Event event = new Event(Event.TYPE_FUNCTIONAL_RESULT, testSummary.toMap()).setTestRunId(testRunId);
            result.add(event);
        }
        return result;
    }

    public static List<String> extractHierarchy(String eventName, char separator) {
        return Splitter.on(separator).splitToList(eventName);
    }

    private Set<String> findTags(SuiteExecutionReport.PageHistoryReference reference) {
        WikiPagePath wikiPagePath = PathParser.parse(reference.getPageName());
        PageCrawler pageCrawler = root.getPageCrawler();
        Set<String> tags = set();
        if (pageCrawler.pageExists(wikiPagePath)) {
            WikiPage thePage = pageCrawler.getPage(wikiPagePath);
            tags = findTags(thePage);
        }
        return tags;
    }

    /**
     * Find tags on wiki page, so we can add tags on parent pages as well.
     *
     * @param page
     * @return a set of tags
     */
    private Set<String> findTags(WikiPage page) {
        if (page == null || page.isRoot()) {
            return set();
        }

        String tags = page.getData().getAttribute(PageData.PropertySUITES);
        Set<String> tagSet = isNotBlank(tags) ? set(tags.split(",")) : set();
        tagSet.addAll(findTags(page.getParent()));
        return tagSet;
    }

    private String findFirstErrorMessage(SuiteExecutionReport.PageHistoryReference reference) {
        String firstErrorMessage = null;
        try {
            PageHistory pageHistory = getPageHistory(resultsDirectory, reference.getPageName());
            if (pageHistory == null) {
                return "No test results found for page: " + reference.getPageName();
            }
            TestExecutionReport testExecutionReport = getExecutionReport(pageHistory, new Date(reference.getTime()));
            firstErrorMessage = findFirstErrorMessage(testExecutionReport);
        } catch (Exception e) {
            LOG.warn("Unable to find test result file for test page {}", reference.getPageName());
            LOG.debug("Could not determine first error message", e);
        }
        return firstErrorMessage;
    }

    private String findFirstErrorMessage(TestExecutionReport testExecutionReport) {
        TestExecutionReport.TestResult testResult = testExecutionReport.getResults().get(0);

        for (TestExecutionReport.InstructionResult instructionResult : testResult.getInstructions()) {
            for (TestExecutionReport.Expectation expectation : instructionResult.getExpectations()) {
                if ("fail".equals(expectation.getStatus())) {
                    return format("Actual: '%s'; Expected: '%s'", expectation.getActual(), expectation.getExpected());
                }
                if ("error".equals(expectation.getStatus())) {
                    return expectation.getEvaluationMessage();
                }
            }
        }
        return null;
    }

    @SuppressWarnings("unchecked")
    private <T extends ExecutionReport> T getExecutionReport(PageHistory pageHistory, Date date) throws Exception {
        TestResultRecord testResultRecord = pageHistory.get(date);
        return (T) ExecutionReport.makeReport(FileUtil.getFileContent(testResultRecord.getFile()));
    }

    private PageHistory getPageHistory(File resultsDirectory, String pageName) throws FileNotFoundException {
        if (!resultsDirectory.canRead()) {
            throw new FileNotFoundException("Could not find " + resultsDirectory);
        }
        return testHistory.getPageHistory(pageName);
    }

    private boolean pageHistoryIsSuite(PageHistory pageHistory) throws Exception {
        if (pageHistory == null) {
            return false;
        }
        ExecutionReport report = getExecutionReport(pageHistory, pageHistory.getLatestDate());
        return report instanceof SuiteExecutionReport;
    }

    public static Set<String> set(String... tags) {
        Set<String> tagSet = new TreeSet<>();
        for (String tag : tags) {
            String trimmed = tag.trim();
            if (!"".equals(trimmed)) {
                tagSet.add(trimmed);
            }
        }
        return tagSet;
    }

    private boolean isNotBlank(String tags) {
        return tags != null && !"".equals(tags);
    }

}
