/*
 * Decompiled with CFR 0.152.
 */
package org.elasticsearch.indices.store;

import java.io.Closeable;
import java.io.IOException;
import java.util.ArrayList;
import java.util.EnumSet;
import java.util.concurrent.atomic.AtomicInteger;
import org.apache.lucene.store.StoreRateLimiting;
import org.elasticsearch.cluster.ClusterChangedEvent;
import org.elasticsearch.cluster.ClusterName;
import org.elasticsearch.cluster.ClusterService;
import org.elasticsearch.cluster.ClusterState;
import org.elasticsearch.cluster.ClusterStateListener;
import org.elasticsearch.cluster.ClusterStateNonMasterUpdateTask;
import org.elasticsearch.cluster.metadata.IndexMetaData;
import org.elasticsearch.cluster.node.DiscoveryNode;
import org.elasticsearch.cluster.routing.IndexRoutingTable;
import org.elasticsearch.cluster.routing.IndexShardRoutingTable;
import org.elasticsearch.cluster.routing.ShardRouting;
import org.elasticsearch.common.collect.Tuple;
import org.elasticsearch.common.component.AbstractComponent;
import org.elasticsearch.common.inject.Inject;
import org.elasticsearch.common.io.stream.StreamInput;
import org.elasticsearch.common.io.stream.StreamOutput;
import org.elasticsearch.common.settings.ImmutableSettings;
import org.elasticsearch.common.settings.Settings;
import org.elasticsearch.common.unit.ByteSizeUnit;
import org.elasticsearch.common.unit.ByteSizeValue;
import org.elasticsearch.env.NodeEnvironment;
import org.elasticsearch.index.IndexService;
import org.elasticsearch.index.shard.IndexShard;
import org.elasticsearch.index.shard.IndexShardState;
import org.elasticsearch.index.shard.ShardId;
import org.elasticsearch.indices.IndicesService;
import org.elasticsearch.node.settings.NodeSettingsService;
import org.elasticsearch.transport.BaseTransportRequestHandler;
import org.elasticsearch.transport.TransportChannel;
import org.elasticsearch.transport.TransportException;
import org.elasticsearch.transport.TransportRequest;
import org.elasticsearch.transport.TransportResponse;
import org.elasticsearch.transport.TransportResponseHandler;
import org.elasticsearch.transport.TransportService;

public class IndicesStore
extends AbstractComponent
implements ClusterStateListener,
Closeable {
    public static final String INDICES_STORE_THROTTLE_TYPE = "indices.store.throttle.type";
    public static final String INDICES_STORE_THROTTLE_MAX_BYTES_PER_SEC = "indices.store.throttle.max_bytes_per_sec";
    public static final String ACTION_SHARD_EXISTS = "internal:index/shard/exists";
    private static final EnumSet<IndexShardState> ACTIVE_STATES = EnumSet.of(IndexShardState.STARTED, IndexShardState.RELOCATED);
    private final NodeEnvironment nodeEnv;
    private final NodeSettingsService nodeSettingsService;
    private final IndicesService indicesService;
    private final ClusterService clusterService;
    private final TransportService transportService;
    private volatile String rateLimitingType;
    private volatile ByteSizeValue rateLimitingThrottle;
    private final StoreRateLimiting rateLimiting = new StoreRateLimiting();
    private final ApplySettings applySettings = new ApplySettings();

    @Inject
    public IndicesStore(Settings settings, NodeEnvironment nodeEnv, NodeSettingsService nodeSettingsService, IndicesService indicesService, ClusterService clusterService, TransportService transportService) {
        super(settings);
        this.nodeEnv = nodeEnv;
        this.nodeSettingsService = nodeSettingsService;
        this.indicesService = indicesService;
        this.clusterService = clusterService;
        this.transportService = transportService;
        transportService.registerHandler(ACTION_SHARD_EXISTS, new ShardActiveRequestHandler());
        this.rateLimitingType = this.componentSettings.get("throttle.type", StoreRateLimiting.Type.MERGE.name());
        this.rateLimiting.setType(this.rateLimitingType);
        this.rateLimitingThrottle = this.componentSettings.getAsBytesSize("throttle.max_bytes_per_sec", new ByteSizeValue(20L, ByteSizeUnit.MB));
        this.rateLimiting.setMaxRate(this.rateLimitingThrottle);
        this.logger.debug("using indices.store.throttle.type [{}], with index.store.throttle.max_bytes_per_sec [{}]", this.rateLimitingType, this.rateLimitingThrottle);
        nodeSettingsService.addListener(this.applySettings);
        clusterService.addLast(this);
    }

    IndicesStore() {
        super(ImmutableSettings.EMPTY);
        this.nodeEnv = null;
        this.nodeSettingsService = null;
        this.indicesService = null;
        this.clusterService = null;
        this.transportService = null;
    }

    public StoreRateLimiting rateLimiting() {
        return this.rateLimiting;
    }

    @Override
    public void close() {
        this.nodeSettingsService.removeListener(this.applySettings);
        this.clusterService.remove(this);
    }

    @Override
    public void clusterChanged(ClusterChangedEvent event) {
        if (!event.routingTableChanged()) {
            return;
        }
        if (event.state().blocks().disableStatePersistence()) {
            return;
        }
        for (IndexRoutingTable indexRoutingTable : event.state().routingTable()) {
            for (IndexShardRoutingTable indexShardRoutingTable : indexRoutingTable) {
                ShardId shardId;
                if (!this.shardCanBeDeleted(event.state(), indexShardRoutingTable) || !this.indicesService.canDeleteShardContent(shardId = indexShardRoutingTable.shardId(), event.state().getMetaData().index(shardId.getIndex()))) continue;
                this.deleteShardIfExistElseWhere(event.state(), indexShardRoutingTable);
            }
        }
    }

    boolean shardCanBeDeleted(ClusterState state, IndexShardRoutingTable indexShardRoutingTable) {
        if (indexShardRoutingTable.size() == 0) {
            return false;
        }
        for (ShardRouting shardRouting : indexShardRoutingTable) {
            if (!shardRouting.started()) {
                return false;
            }
            DiscoveryNode node = state.nodes().get(shardRouting.currentNodeId());
            if (node == null) {
                return false;
            }
            if (shardRouting.relocatingNodeId() != null && (node = state.nodes().get(shardRouting.relocatingNodeId())) == null) {
                return false;
            }
            String localNodeId = state.getNodes().localNode().id();
            if (!localNodeId.equals(shardRouting.currentNodeId()) && !localNodeId.equals(shardRouting.relocatingNodeId())) continue;
            return false;
        }
        return true;
    }

    private void deleteShardIfExistElseWhere(ClusterState state, IndexShardRoutingTable indexShardRoutingTable) {
        ArrayList<Tuple<DiscoveryNode, ShardActiveRequest>> requests = new ArrayList<Tuple<DiscoveryNode, ShardActiveRequest>>(indexShardRoutingTable.size());
        String indexUUID = state.getMetaData().index(indexShardRoutingTable.shardId().getIndex()).getUUID();
        ClusterName clusterName = state.getClusterName();
        for (ShardRouting shardRouting : indexShardRoutingTable) {
            DiscoveryNode discoveryNode = state.nodes().get(shardRouting.currentNodeId());
            assert (discoveryNode != null);
            requests.add(new Tuple<DiscoveryNode, ShardActiveRequest>(discoveryNode, new ShardActiveRequest(clusterName, indexUUID, shardRouting.shardId())));
            if (shardRouting.relocatingNodeId() == null) continue;
            DiscoveryNode relocatingNode = state.nodes().get(shardRouting.relocatingNodeId());
            assert (relocatingNode != null);
            requests.add(new Tuple<DiscoveryNode, ShardActiveRequest>(relocatingNode, new ShardActiveRequest(clusterName, indexUUID, shardRouting.shardId())));
        }
        ShardActiveResponseHandler responseHandler = new ShardActiveResponseHandler(indexShardRoutingTable.shardId(), state, requests.size());
        for (Tuple tuple : requests) {
            this.logger.trace("{} sending shard active check to {}", ((ShardActiveRequest)tuple.v2()).shardId, tuple.v1());
            this.transportService.sendRequest((DiscoveryNode)tuple.v1(), ACTION_SHARD_EXISTS, (TransportRequest)tuple.v2(), responseHandler);
        }
    }

    private static class ShardActiveResponse
    extends TransportResponse {
        private boolean shardActive;
        private DiscoveryNode node;

        ShardActiveResponse() {
        }

        ShardActiveResponse(boolean shardActive, DiscoveryNode node) {
            this.shardActive = shardActive;
            this.node = node;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.shardActive = in.readBoolean();
            this.node = DiscoveryNode.readNode(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            out.writeBoolean(this.shardActive);
            this.node.writeTo(out);
        }
    }

    private static class ShardActiveRequest
    extends TransportRequest {
        private ClusterName clusterName;
        private String indexUUID;
        private ShardId shardId;

        ShardActiveRequest() {
        }

        ShardActiveRequest(ClusterName clusterName, String indexUUID, ShardId shardId) {
            this.shardId = shardId;
            this.indexUUID = indexUUID;
            this.clusterName = clusterName;
        }

        @Override
        public void readFrom(StreamInput in) throws IOException {
            super.readFrom(in);
            this.clusterName = ClusterName.readClusterName(in);
            this.indexUUID = in.readString();
            this.shardId = ShardId.readShardId(in);
        }

        @Override
        public void writeTo(StreamOutput out) throws IOException {
            super.writeTo(out);
            this.clusterName.writeTo(out);
            out.writeString(this.indexUUID);
            this.shardId.writeTo(out);
        }
    }

    private class ShardActiveRequestHandler
    extends BaseTransportRequestHandler<ShardActiveRequest> {
        private ShardActiveRequestHandler() {
        }

        @Override
        public ShardActiveRequest newInstance() {
            return new ShardActiveRequest();
        }

        @Override
        public String executor() {
            return "same";
        }

        @Override
        public void messageReceived(ShardActiveRequest request, TransportChannel channel) throws Exception {
            channel.sendResponse(new ShardActiveResponse(this.shardActive(request), IndicesStore.this.clusterService.localNode()));
        }

        private boolean shardActive(ShardActiveRequest request) {
            IndexShard indexShard;
            ClusterName thisClusterName = IndicesStore.this.clusterService.state().getClusterName();
            if (!thisClusterName.equals(request.clusterName)) {
                IndicesStore.this.logger.trace("shard exists request meant for cluster[{}], but this is cluster[{}], ignoring request", request.clusterName, thisClusterName);
                return false;
            }
            ShardId shardId = request.shardId;
            IndexService indexService = IndicesStore.this.indicesService.indexService(shardId.index().getName());
            if (indexService != null && indexService.indexUUID().equals(request.indexUUID) && (indexShard = indexService.shard(shardId.getId())) != null) {
                return ACTIVE_STATES.contains((Object)indexShard.state());
            }
            return false;
        }
    }

    private class ShardActiveResponseHandler
    implements TransportResponseHandler<ShardActiveResponse> {
        private final ShardId shardId;
        private final int expectedActiveCopies;
        private final ClusterState clusterState;
        private final AtomicInteger awaitingResponses;
        private final AtomicInteger activeCopies;

        public ShardActiveResponseHandler(ShardId shardId, ClusterState clusterState, int expectedActiveCopies) {
            this.shardId = shardId;
            this.expectedActiveCopies = expectedActiveCopies;
            this.clusterState = clusterState;
            this.awaitingResponses = new AtomicInteger(expectedActiveCopies);
            this.activeCopies = new AtomicInteger();
        }

        @Override
        public ShardActiveResponse newInstance() {
            return new ShardActiveResponse();
        }

        @Override
        public void handleResponse(ShardActiveResponse response) {
            IndicesStore.this.logger.trace("{} is {}active on node {}", this.shardId, response.shardActive ? "" : "not ", response.node);
            if (response.shardActive) {
                this.activeCopies.incrementAndGet();
            }
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        @Override
        public void handleException(TransportException exp) {
            IndicesStore.this.logger.debug("shards active request failed for {}", exp, this.shardId);
            if (this.awaitingResponses.decrementAndGet() == 0) {
                this.allNodesResponded();
            }
        }

        @Override
        public String executor() {
            return "same";
        }

        private void allNodesResponded() {
            if (this.activeCopies.get() != this.expectedActiveCopies) {
                IndicesStore.this.logger.trace("not deleting shard {}, expected {} active copies, but only {} found active copies", this.shardId, this.expectedActiveCopies, this.activeCopies.get());
                return;
            }
            ClusterState latestClusterState = IndicesStore.this.clusterService.state();
            if (this.clusterState.getVersion() != latestClusterState.getVersion()) {
                IndicesStore.this.logger.trace("not deleting shard {}, the latest cluster state version[{}] is not equal to cluster state before shard active api call [{}]", this.shardId, latestClusterState.getVersion(), this.clusterState.getVersion());
                return;
            }
            IndicesStore.this.clusterService.submitStateUpdateTask("indices_store", new ClusterStateNonMasterUpdateTask(){

                @Override
                public ClusterState execute(ClusterState currentState) throws Exception {
                    if (ShardActiveResponseHandler.this.clusterState.getVersion() != currentState.getVersion()) {
                        IndicesStore.this.logger.trace("not deleting shard {}, the update task state version[{}] is not equal to cluster state before shard active api call [{}]", ShardActiveResponseHandler.this.shardId, currentState.getVersion(), ShardActiveResponseHandler.this.clusterState.getVersion());
                        return currentState;
                    }
                    IndexMetaData indexMeta = ShardActiveResponseHandler.this.clusterState.getMetaData().indices().get(ShardActiveResponseHandler.this.shardId.getIndex());
                    try {
                        IndicesStore.this.indicesService.deleteShardStore("no longer used", ShardActiveResponseHandler.this.shardId, indexMeta);
                    }
                    catch (Throwable ex) {
                        IndicesStore.this.logger.debug("{} failed to delete unallocated shard, ignoring", ex, ShardActiveResponseHandler.this.shardId);
                    }
                    if (!IndicesStore.this.indicesService.hasIndex(ShardActiveResponseHandler.this.shardId.getIndex()) && !currentState.nodes().localNode().masterNode()) {
                        try {
                            IndicesStore.this.indicesService.deleteIndexStore("no longer used", indexMeta, currentState);
                        }
                        catch (Throwable ex) {
                            IndicesStore.this.logger.debug("{} failed to delete unallocated index, ignoring", ex, ShardActiveResponseHandler.this.shardId.getIndex());
                        }
                    }
                    return currentState;
                }

                @Override
                public void onFailure(String source, Throwable t) {
                    IndicesStore.this.logger.error("{} unexpected error during deletion of unallocated shard", t, ShardActiveResponseHandler.this.shardId);
                }
            });
        }
    }

    class ApplySettings
    implements NodeSettingsService.Listener {
        ApplySettings() {
        }

        @Override
        public void onRefreshSettings(Settings settings) {
            ByteSizeValue rateLimitingThrottle;
            String rateLimitingType = settings.get(IndicesStore.INDICES_STORE_THROTTLE_TYPE, IndicesStore.this.rateLimitingType);
            StoreRateLimiting.Type.fromString(rateLimitingType);
            if (!rateLimitingType.equals(IndicesStore.this.rateLimitingType)) {
                IndicesStore.this.logger.info("updating indices.store.throttle.type from [{}] to [{}]", IndicesStore.this.rateLimitingType, rateLimitingType);
                IndicesStore.this.rateLimitingType = rateLimitingType;
                IndicesStore.this.rateLimiting.setType(rateLimitingType);
            }
            if (!(rateLimitingThrottle = settings.getAsBytesSize(IndicesStore.INDICES_STORE_THROTTLE_MAX_BYTES_PER_SEC, IndicesStore.this.rateLimitingThrottle)).equals(IndicesStore.this.rateLimitingThrottle)) {
                IndicesStore.this.logger.info("updating indices.store.throttle.max_bytes_per_sec from [{}] to [{}], note, type is [{}]", IndicesStore.this.rateLimitingThrottle, rateLimitingThrottle, IndicesStore.this.rateLimitingType);
                IndicesStore.this.rateLimitingThrottle = rateLimitingThrottle;
                IndicesStore.this.rateLimiting.setMaxRate(rateLimitingThrottle);
            }
        }
    }
}

