/*
 * Decompiled with CFR 0.152.
 */
package org.modeshape.jcr.clustering;

import java.io.ByteArrayInputStream;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.ObjectInputStream;
import java.io.ObjectOutputStream;
import java.io.ObjectStreamClass;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.concurrent.CopyOnWriteArraySet;
import java.util.concurrent.TimeUnit;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicInteger;
import javax.jcr.RepositoryException;
import org.jgroups.Address;
import org.jgroups.Channel;
import org.jgroups.ChannelListener;
import org.jgroups.JChannel;
import org.jgroups.MergeView;
import org.jgroups.Message;
import org.jgroups.ReceiverAdapter;
import org.jgroups.View;
import org.jgroups.blocks.locking.LockService;
import org.jgroups.conf.ProtocolStackConfigurator;
import org.jgroups.conf.XmlConfigurator;
import org.jgroups.fork.ForkChannel;
import org.jgroups.protocols.CENTRAL_LOCK;
import org.jgroups.protocols.FORK;
import org.jgroups.stack.Protocol;
import org.jgroups.stack.ProtocolStack;
import org.modeshape.common.SystemFailureException;
import org.modeshape.common.annotation.ThreadSafe;
import org.modeshape.common.i18n.I18nResource;
import org.modeshape.common.logging.Logger;
import org.modeshape.common.util.StringUtil;
import org.modeshape.jcr.clustering.ClusteringI18n;
import org.modeshape.jcr.clustering.MessageConsumer;

@ThreadSafe
public abstract class ClusteringService {
    protected static final Logger LOGGER = Logger.getLogger(ClusteringService.class);
    private static final long DEFAULT_MAX_CLOCK_DELAY_CLUSTER_MILLIS = TimeUnit.MINUTES.toMillis(10L);
    private static final String GLOBAL_LOCK = "modeshape-global-lock";
    protected final Listener listener;
    protected final Receiver receiver;
    protected final String clusterName;
    protected JChannel channel;
    protected LockService lockService;
    private final long maxAllowedClockDelayMillis;
    private final AtomicInteger membersInCluster;
    private final AtomicBoolean isOpen;
    private final Set<MessageConsumer<Serializable>> consumers;

    protected ClusteringService(String clusterName) {
        assert (clusterName != null);
        this.clusterName = clusterName;
        this.listener = new Listener();
        this.receiver = new Receiver();
        this.isOpen = new AtomicBoolean(false);
        this.membersInCluster = new AtomicInteger(1);
        this.maxAllowedClockDelayMillis = DEFAULT_MAX_CLOCK_DELAY_CLUSTER_MILLIS;
        this.consumers = new CopyOnWriteArraySet<MessageConsumer<Serializable>>();
    }

    public void restart() throws Exception {
        this.shutdown();
        this.init();
    }

    public synchronized void addConsumer(MessageConsumer<? extends Serializable> consumer) {
        this.consumers.add(consumer);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public synchronized boolean shutdown() {
        if (this.channel == null) {
            return false;
        }
        LOGGER.debug("Shutting down clustering service...", new Object[0]);
        this.consumers.clear();
        this.isOpen.set(false);
        this.lockService.unlockAll();
        try {
            this.channel.removeChannelListener((ChannelListener)this.listener);
            this.channel.setReceiver(null);
            this.channel.close();
            LOGGER.debug("Successfully closed main channel", new Object[0]);
        }
        finally {
            this.channel = null;
        }
        this.membersInCluster.set(1);
        return true;
    }

    public boolean tryLock(long time, TimeUnit unit) {
        try {
            return this.lockService.getLock(GLOBAL_LOCK).tryLock(time, unit);
        }
        catch (InterruptedException e) {
            LOGGER.debug("Thread " + Thread.currentThread().getName() + " received interrupt request while waiting to acquire lock '{0}'", new Object[]{GLOBAL_LOCK});
            Thread.interrupted();
            return false;
        }
    }

    public void unlock() {
        this.lockService.getLock(GLOBAL_LOCK).unlock();
    }

    public boolean isOpen() {
        return this.isOpen.get();
    }

    public boolean multipleMembersInCluster() {
        return this.membersInCluster.get() > 1;
    }

    public int membersInCluster() {
        return this.membersInCluster.get();
    }

    public String clusterName() {
        return this.channel.getClusterName();
    }

    public long getMaxAllowedClockDelayMillis() {
        return this.maxAllowedClockDelayMillis;
    }

    public boolean sendMessage(Serializable payload) {
        if (!this.isOpen() || !this.multipleMembersInCluster()) {
            return false;
        }
        if (LOGGER.isDebugEnabled()) {
            LOGGER.debug("Sending payload {0} in cluster {1} ", new Object[]{payload, this.clusterName()});
        }
        try {
            byte[] messageData = this.toByteArray(payload);
            Message jgMessage = new Message(null, null, messageData);
            this.channel.send(jgMessage);
            return true;
        }
        catch (Exception e) {
            throw new SystemFailureException(ClusteringI18n.errorSendingMessage.text(new Object[]{this.clusterName()}), (Throwable)e);
        }
    }

    public static ClusteringService startStandalone(String clusterName, String jgroupsConfig) {
        StandaloneClusteringService clusteringService = new StandaloneClusteringService(clusterName, jgroupsConfig);
        ((ClusteringService)clusteringService).init();
        return clusteringService;
    }

    public static ClusteringService startForked(String forkStackId, Channel mainChannel) {
        ForkedClusteringService clusteringService = new ForkedClusteringService(forkStackId, mainChannel);
        ((ClusteringService)clusteringService).init();
        return clusteringService;
    }

    private byte[] toByteArray(Object payload) throws IOException {
        ByteArrayOutputStream output = new ByteArrayOutputStream();
        try (ObjectOutputStream stream = new ObjectOutputStream(output);){
            stream.writeObject(payload);
        }
        return output.toByteArray();
    }

    protected Serializable fromByteArray(byte[] data, ClassLoader classLoader) throws IOException, ClassNotFoundException {
        if (classLoader == null) {
            classLoader = ClusteringService.class.getClassLoader();
        }
        try (ObjectInputStreamWithClassLoader input = new ObjectInputStreamWithClassLoader(new ByteArrayInputStream(data), classLoader);){
            Serializable serializable = (Serializable)input.readObject();
            return serializable;
        }
    }

    protected JChannel getChannel() {
        return this.channel;
    }

    protected abstract void init();

    private static class ForkedClusteringService
    extends ClusteringService {
        private static final String FORK_CHANNEL_NAME = "modeshape-fork-channel";
        private static final Map<String, List<String>> FORK_STACKS_BY_CHANNEL_NAME = new HashMap<String, List<String>>();
        private final Channel mainChannel;

        protected ForkedClusteringService(String forkStackId, Channel mainChannel) {
            super(forkStackId);
            this.mainChannel = mainChannel;
        }

        @Override
        protected void init() {
            try {
                Protocol topProtocol = this.mainChannel.getProtocolStack().getTopProtocol();
                String forkStackId = this.clusterName;
                boolean alreadyHasForkProtocol = this.mainChannel.getProtocolStack().findProtocol(FORK.class) != null;
                this.channel = new ForkChannel(this.mainChannel, forkStackId, FORK_CHANNEL_NAME, true, 1, topProtocol.getClass(), new Protocol[]{new CENTRAL_LOCK()});
                this.lockService = new LockService(this.channel);
                this.channel.addChannelListener((ChannelListener)this.listener);
                this.channel.setReceiver((org.jgroups.Receiver)this.receiver);
                this.channel.connect(FORK_CHANNEL_NAME);
                if (!alreadyHasForkProtocol) {
                    String mainChannelName = this.mainChannel.getName();
                    List<String> existingForksForChannel = FORK_STACKS_BY_CHANNEL_NAME.get(mainChannelName);
                    if (existingForksForChannel == null) {
                        existingForksForChannel = new ArrayList<String>();
                        FORK_STACKS_BY_CHANNEL_NAME.put(mainChannelName, existingForksForChannel);
                    }
                    existingForksForChannel.add(forkStackId);
                }
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        @Override
        public synchronized boolean shutdown() {
            if (super.shutdown()) {
                String mainChannelName = this.mainChannel.getName();
                List<String> forksForChannel = FORK_STACKS_BY_CHANNEL_NAME.get(mainChannelName);
                if (forksForChannel != null) {
                    forksForChannel.remove(this.clusterName);
                    if (forksForChannel.isEmpty()) {
                        FORK_STACKS_BY_CHANNEL_NAME.remove(mainChannelName);
                        Protocol removed = this.mainChannel.getProtocolStack().removeProtocol(FORK.class);
                        if (removed != null) {
                            LOGGER.debug("FORK protocol removed from original channel stack for channel {0}", new Object[]{mainChannelName});
                        } else {
                            LOGGER.debug("FORK protocol not found in original channel stack for channel {0}", new Object[]{mainChannelName});
                        }
                    }
                }
                return true;
            }
            return false;
        }
    }

    private static class StandaloneClusteringService
    extends ClusteringService {
        private final String jgroupsConfig;

        protected StandaloneClusteringService(String clusterName, String jgroupsConfig) {
            super(clusterName);
            this.jgroupsConfig = jgroupsConfig;
        }

        @Override
        protected void init() {
            try {
                this.channel = this.newChannel(this.jgroupsConfig);
                ProtocolStack protocolStack = this.channel.getProtocolStack();
                Protocol centralLock = protocolStack.findProtocol(CENTRAL_LOCK.class);
                if (centralLock == null) {
                    CENTRAL_LOCK lockingProtocol = new CENTRAL_LOCK();
                    lockingProtocol.init();
                    protocolStack.addProtocol((Protocol)lockingProtocol);
                }
                this.lockService = new LockService(this.channel);
                this.channel.addChannelListener((ChannelListener)this.listener);
                this.channel.setReceiver((org.jgroups.Receiver)this.receiver);
                this.channel.connect(this.clusterName);
            }
            catch (Exception e) {
                throw new RuntimeException(e);
            }
        }

        /*
         * WARNING - Removed try catching itself - possible behaviour change.
         */
        private JChannel newChannel(String jgroupsConfig) throws Exception {
            if (StringUtil.isBlank((String)jgroupsConfig)) {
                return new JChannel();
            }
            XmlConfigurator configurator = null;
            InputStream stream = ClusteringService.class.getClassLoader().getResourceAsStream(jgroupsConfig);
            try {
                configurator = XmlConfigurator.getInstance((InputStream)stream);
            }
            catch (IOException e) {
                LOGGER.debug((Throwable)e, "Channel configuration is not a classpath resource", new Object[0]);
                stream = new ByteArrayInputStream(jgroupsConfig.getBytes());
                try {
                    configurator = XmlConfigurator.getInstance((InputStream)stream);
                }
                catch (IOException e1) {
                    LOGGER.debug((Throwable)e, "Channel configuration is not valid XML content", new Object[0]);
                }
            }
            finally {
                if (stream != null) {
                    try {
                        stream.close();
                    }
                    catch (IOException e) {}
                }
            }
            if (configurator == null) {
                throw new RepositoryException(ClusteringI18n.channelConfigurationError.text(new Object[]{jgroupsConfig}));
            }
            return new JChannel((ProtocolStackConfigurator)configurator);
        }
    }

    private static class ObjectInputStreamWithClassLoader
    extends ObjectInputStream {
        private ClassLoader cl;

        public ObjectInputStreamWithClassLoader(InputStream in, ClassLoader cl) throws IOException {
            super(in);
            this.cl = cl;
        }

        @Override
        protected Class<?> resolveClass(ObjectStreamClass desc) throws IOException, ClassNotFoundException {
            if (this.cl == null) {
                return super.resolveClass(desc);
            }
            try {
                return Class.forName(desc.getName(), false, this.cl);
            }
            catch (ClassNotFoundException ex) {
                return super.resolveClass(desc);
            }
        }

        @Override
        public void close() throws IOException {
            super.close();
            this.cl = null;
        }
    }

    protected class Listener
    implements ChannelListener {
        protected Listener() {
        }

        public void channelClosed(Channel channel) {
            ClusteringService.this.isOpen.set(false);
        }

        public void channelConnected(Channel channel) {
            ClusteringService.this.isOpen.set(true);
        }

        public void channelDisconnected(Channel channel) {
            ClusteringService.this.isOpen.set(false);
        }
    }

    protected final class Receiver
    extends ReceiverAdapter {
        protected Receiver() {
        }

        public void block() {
            ClusteringService.this.isOpen.set(false);
        }

        public void receive(Message message) {
            try {
                Serializable payload = ClusteringService.this.fromByteArray(message.getBuffer(), ((Object)((Object)this)).getClass().getClassLoader());
                if (LOGGER.isDebugEnabled()) {
                    LOGGER.debug("Cluster {0} received payload {1}", new Object[]{ClusteringService.this.clusterName(), payload});
                }
                for (MessageConsumer consumer : ClusteringService.this.consumers) {
                    if (!consumer.getPayloadType().isAssignableFrom(payload.getClass())) continue;
                    consumer.consume(payload);
                }
            }
            catch (Exception e) {
                String msg = ClusteringI18n.errorReceivingMessage.text(new Object[]{ClusteringService.this.clusterName()});
                throw new SystemFailureException(msg, (Throwable)e);
            }
        }

        public void suspect(Address suspectedMbr) {
            LOGGER.error((I18nResource)ClusteringI18n.memberOfClusterIsSuspect, new Object[]{ClusteringService.this.clusterName(), suspectedMbr});
        }

        public void viewAccepted(View newView) {
            LOGGER.trace("Members of '{0}' cluster have changed: {1}, total count: {2}", new Object[]{ClusteringService.this.clusterName(), newView, newView.getMembers().size()});
            if (newView instanceof MergeView) {
                LOGGER.trace("Received a merged view in cluster {0}. Releasing all locks...", new Object[]{ClusteringService.this.clusterName()});
                ClusteringService.this.lockService.unlockAll();
            }
            ClusteringService.this.membersInCluster.set(newView.getMembers().size());
            if (ClusteringService.this.membersInCluster.get() > 1) {
                LOGGER.debug("There are now multiple members of cluster '{0}'; changes will be propagated throughout the cluster", new Object[]{ClusteringService.this.clusterName()});
            } else if (ClusteringService.this.membersInCluster.get() == 1) {
                LOGGER.debug("There is only one member of cluster '{0}'; changes will be propagated locally only", new Object[]{ClusteringService.this.clusterName()});
            }
        }
    }
}

