package de.jaggl.sqlbuilder.conditions;

import static de.jaggl.sqlbuilder.domain.ConcatenationType.AND;
import static de.jaggl.sqlbuilder.domain.ConcatenationType.OR;
import static de.jaggl.sqlbuilder.domain.ConditionType.WHERE;
import static de.jaggl.sqlbuilder.domain.ConditionType.WHERE_NOT;

import de.jaggl.sqlbuilder.domain.BuildingContext;
import de.jaggl.sqlbuilder.domain.ConcatenationType;
import de.jaggl.sqlbuilder.domain.ConditionType;
import de.jaggl.sqlbuilder.utils.Indentation;
import lombok.Getter;
import lombok.Setter;
import lombok.ToString;

/**
 * @author Martin Schumacher
 *
 * @since 2.0.0
 */
@Getter
@Setter
@ToString
public abstract class Condition
{
    private ConditionType type;
    private ConcatenationType concatenation;

    protected abstract String doBuild(BuildingContext context, Indentation indentation);

    public String build(BuildingContext context, boolean concatenate, Indentation indentation)
    {
        var builder = new StringBuilder();
        if (concatenate)
        {
            builder.append(concatenation == AND ? context.getDialect().getLabels().getAnd() : context.getDialect().getLabels().getOr());
        }
        if (type == WHERE_NOT)
        {
            builder.append(" ").append(context.getDialect().getLabels().getNot());
        }
        if (concatenate)
        {
            if (indentation.isEnabled() && CombinedCondition.class.isAssignableFrom(getClass()))
            {
                builder.append(context.getDelimiter());
            }
            else
            {
                builder.append(" ");
            }
        }
        builder.append(doBuild(context, indentation));
        return builder.toString();
    }

    public Condition and(Condition other)
    {
        return append(other, AND, WHERE);
    }

    public Condition or(Condition other)
    {
        return append(other, OR, WHERE);
    }

    public Condition andNot(Condition other)
    {
        return append(other, AND, WHERE_NOT);
    }

    public Condition orNot(Condition other)
    {
        return append(other, OR, WHERE_NOT);
    }

    public Condition append(Condition other, ConcatenationType concatenationType, ConditionType conditionType)
    {
        other.setConcatenation(concatenationType);
        other.setType(conditionType);
        CombinedCondition combined;
        if (!CombinedCondition.class.isAssignableFrom(getClass()))
        {
            combined = new CombinedCondition(this);
        }
        else
        {
            combined = (CombinedCondition) this;
        }
        combined.append(other);
        return combined;
    }

    public static Condition plain(String plainCondition)
    {
        return new PlainCondition(plainCondition);
    }
}
