/*
 * Decompiled with CFR 0.152.
 */
package io.grpc.internal;

import com.google.common.annotations.VisibleForTesting;
import com.google.common.base.Preconditions;
import com.google.common.util.concurrent.ThreadFactoryBuilder;
import io.grpc.Metadata;
import io.grpc.Status;
import io.grpc.internal.SharedResourceHolder;
import java.net.URI;
import java.net.URISyntaxException;
import java.util.EnumSet;
import java.util.Set;
import java.util.concurrent.ExecutorService;
import java.util.concurrent.Executors;
import java.util.concurrent.ScheduledExecutorService;
import javax.annotation.Nullable;

public final class GrpcUtil {
    public static final Metadata.Key<Long> TIMEOUT_KEY = Metadata.Key.of("grpc-timeout", new TimeoutMarshaller());
    public static final Metadata.Key<String> MESSAGE_ENCODING_KEY = Metadata.Key.of("grpc-encoding", Metadata.ASCII_STRING_MARSHALLER);
    public static final Metadata.Key<String> AUTHORITY_KEY = Metadata.Key.of("grpc-authority", Metadata.ASCII_STRING_MARSHALLER);
    public static final Metadata.Key<String> CONTENT_TYPE_KEY = Metadata.Key.of("content-type", Metadata.ASCII_STRING_MARSHALLER);
    public static final Metadata.Key<String> USER_AGENT_KEY = Metadata.Key.of("user-agent", Metadata.ASCII_STRING_MARSHALLER);
    public static final String CONTENT_TYPE_GRPC = "application/grpc";
    public static final String HTTP_METHOD = "POST";
    public static final String TE_TRAILERS = "trailers";
    public static final String TIMEOUT = "grpc-timeout";
    public static final String MESSAGE_ENCODING = "grpc-encoding";
    public static final int DEFAULT_MAX_MESSAGE_SIZE = 0x6400000;
    public static final Set<Status.Code> CANCEL_REASONS = EnumSet.of(Status.Code.CANCELLED, Status.Code.DEADLINE_EXCEEDED, Status.Code.INTERNAL, Status.Code.UNKNOWN);
    static final SharedResourceHolder.Resource<ExecutorService> SHARED_CHANNEL_EXECUTOR = new SharedResourceHolder.Resource<ExecutorService>(){
        private static final String name = "grpc-default-executor";

        @Override
        public ExecutorService create() {
            return Executors.newCachedThreadPool(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("grpc-default-executor-%d").build());
        }

        @Override
        public void close(ExecutorService instance) {
            instance.shutdown();
        }

        public String toString() {
            return name;
        }
    };
    public static final SharedResourceHolder.Resource<ScheduledExecutorService> TIMER_SERVICE = new SharedResourceHolder.Resource<ScheduledExecutorService>(){

        @Override
        public ScheduledExecutorService create() {
            return Executors.newSingleThreadScheduledExecutor(new ThreadFactoryBuilder().setDaemon(true).setNameFormat("grpc-timer-%d").build());
        }

        @Override
        public void close(ScheduledExecutorService instance) {
            instance.shutdown();
        }
    };

    public static Status httpStatusToGrpcStatus(int httpStatusCode) {
        switch (httpStatusCode) {
            case 401: {
                return Status.UNAUTHENTICATED;
            }
            case 403: {
                return Status.PERMISSION_DENIED;
            }
        }
        if (httpStatusCode < 100) {
            return Status.UNKNOWN;
        }
        if (httpStatusCode < 200) {
            return Status.INTERNAL;
        }
        if (httpStatusCode < 300) {
            return Status.OK;
        }
        return Status.UNKNOWN;
    }

    public static boolean isGrpcContentType(String contentType) {
        if (contentType == null) {
            return false;
        }
        if (CONTENT_TYPE_GRPC.length() > contentType.length()) {
            return false;
        }
        if (!(contentType = contentType.toLowerCase()).startsWith(CONTENT_TYPE_GRPC)) {
            return false;
        }
        if (contentType.length() == CONTENT_TYPE_GRPC.length()) {
            return true;
        }
        char nextChar = contentType.charAt(CONTENT_TYPE_GRPC.length());
        return nextChar == '+' || nextChar == ';';
    }

    public static String getGrpcUserAgent(String transportName, @Nullable String applicationUserAgent) {
        StringBuilder builder = new StringBuilder();
        if (applicationUserAgent != null) {
            builder.append(applicationUserAgent);
            builder.append(' ');
        }
        builder.append("grpc-java-");
        builder.append(transportName);
        String version = GrpcUtil.class.getPackage().getImplementationVersion();
        if (version != null) {
            builder.append("/");
            builder.append(version);
        }
        return builder.toString();
    }

    public static URI authorityToUri(String authority) {
        URI uri;
        Preconditions.checkNotNull((Object)authority, (Object)"authority");
        try {
            uri = new URI(null, authority, null, null, null);
        }
        catch (URISyntaxException ex) {
            throw new IllegalArgumentException("Invalid authority: " + authority, ex);
        }
        if (uri.getUserInfo() != null) {
            throw new IllegalArgumentException("Userinfo must not be present on authority: " + authority);
        }
        return uri;
    }

    public static String checkAuthority(String authority) {
        GrpcUtil.authorityToUri(authority);
        return authority;
    }

    public static String authorityFromHostAndPort(String host, int port) {
        try {
            return new URI(null, null, host, port, null, null, null).getAuthority();
        }
        catch (URISyntaxException ex) {
            throw new IllegalArgumentException("Invalid host or port: " + host + " " + port, ex);
        }
    }

    private GrpcUtil() {
    }

    @VisibleForTesting
    static class TimeoutMarshaller
    implements Metadata.AsciiMarshaller<Long> {
        TimeoutMarshaller() {
        }

        @Override
        public String toAsciiString(Long timeoutMicros) {
            String timeoutUnit;
            long timeout;
            Preconditions.checkArgument((timeoutMicros >= 0L ? 1 : 0) != 0, (Object)"Negative timeout");
            int cutoff = 100000000;
            if (timeoutMicros < (long)cutoff) {
                timeout = timeoutMicros;
                timeoutUnit = "u";
            } else if (timeoutMicros / 1000L < (long)cutoff) {
                timeout = timeoutMicros / 1000L;
                timeoutUnit = "m";
            } else if (timeoutMicros / 1000000L < (long)cutoff) {
                timeout = timeoutMicros / 1000000L;
                timeoutUnit = "S";
            } else if (timeoutMicros / 60000000L < (long)cutoff) {
                timeout = timeoutMicros / 60000000L;
                timeoutUnit = "M";
            } else if (timeoutMicros / 3600000000L < (long)cutoff) {
                timeout = timeoutMicros / 3600000000L;
                timeoutUnit = "H";
            } else {
                throw new IllegalArgumentException("Timeout too large");
            }
            return Long.toString(timeout) + timeoutUnit;
        }

        @Override
        public Long parseAsciiString(String serialized) {
            long factor;
            String valuePart = serialized.substring(0, serialized.length() - 1);
            char unit = serialized.charAt(serialized.length() - 1);
            switch (unit) {
                case 'u': {
                    factor = 1L;
                    break;
                }
                case 'm': {
                    factor = 1000L;
                    break;
                }
                case 'S': {
                    factor = 1000000L;
                    break;
                }
                case 'M': {
                    factor = 60000000L;
                    break;
                }
                case 'H': {
                    factor = 3600000000L;
                    break;
                }
                default: {
                    throw new IllegalArgumentException(String.format("Invalid timeout unit: %s", Character.valueOf(unit)));
                }
            }
            return Long.parseLong(valuePart) * factor;
        }
    }

    public static enum Http2Error {
        NO_ERROR(0, Status.INTERNAL),
        PROTOCOL_ERROR(1, Status.INTERNAL),
        INTERNAL_ERROR(2, Status.INTERNAL),
        FLOW_CONTROL_ERROR(3, Status.INTERNAL),
        SETTINGS_TIMEOUT(4, Status.INTERNAL),
        STREAM_CLOSED(5, Status.INTERNAL),
        FRAME_SIZE_ERROR(6, Status.INTERNAL),
        REFUSED_STREAM(7, Status.UNAVAILABLE),
        CANCEL(8, Status.CANCELLED),
        COMPRESSION_ERROR(9, Status.INTERNAL),
        CONNECT_ERROR(10, Status.INTERNAL),
        ENHANCE_YOUR_CALM(11, Status.RESOURCE_EXHAUSTED.withDescription("Bandwidth exhausted")),
        INADEQUATE_SECURITY(12, Status.PERMISSION_DENIED.withDescription("Permission denied as protocol is not secure enough to call")),
        HTTP_1_1_REQUIRED(13, Status.UNKNOWN);

        private static final Http2Error[] codeMap;
        private final int code;
        private final Status status;

        private Http2Error(int code, Status status) {
            this.code = code;
            this.status = status.augmentDescription("HTTP/2 error code: " + this.name());
        }

        public long code() {
            return this.code;
        }

        public Status status() {
            return this.status;
        }

        public static Http2Error forCode(long code) {
            if (code >= (long)codeMap.length || code < 0L) {
                return null;
            }
            return codeMap[(int)code];
        }

        public static Status statusForCode(long code) {
            Http2Error error = Http2Error.forCode(code);
            if (error == null) {
                Status.Code statusCode = INTERNAL_ERROR.status().getCode();
                return Status.fromCodeValue(statusCode.value()).withDescription("Unrecognized HTTP/2 error code: " + code);
            }
            return error.status();
        }

        static {
            Http2Error[] errors = Http2Error.values();
            int size = (int)errors[errors.length - 1].code() + 1;
            codeMap = new Http2Error[size];
            for (Http2Error error : errors) {
                int index = (int)error.code();
                Http2Error.codeMap[index] = error;
            }
        }
    }
}

