/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.examples;

import webwork.action.ActionSupport;
import webwork.action.CommandDriven;

import java.io.BufferedInputStream;
import java.io.InputStream;
import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Collections;
import java.util.List;

/**
 * Load content from a given URL
 *
 * @author Rickard \u00D6berg (rickard@dreambean.com)
 * @version $Revision: 1.4 $
 */
public class HitUrl
        extends ActionSupport
        implements CommandDriven
{
    // Constants -----------------------------------------------------
    public static final int EXCEPTION = -1;
    public static final int NO_CONNECT = -2;

    // Static --------------------------------------------------------
    static List history = new ArrayList();

    // Attributes ----------------------------------------------------
    String urlString = "";
    int count = 1;
    int threadCount = 1;

    URL url;
    int size;
    long total;
    long averageTime;
    long meanTime;
    long fastest = 0;
    long slowest = 0;
    Long[] times = new Long[0];
    int errors = 0;
    int connectErrors = 0;
    int done = 0;

    int currentIndex;

    // Public --------------------------------------------------------
    public void setUrl(String url)
    {
        this.urlString = url;
        synchronized (history)
        {
            if (!history.contains(url))
            {
                history.add(url);
                if (history.size() > 10)
                {
                    history.remove(0);
                }
            }
        }
    }

    public String getUrl()
    {
        return urlString;
    }

    public List getHistory()
    {
        return history;
    }

    public void setCount(int count)
    {
        this.count = count;
    }

    public int getCount()
    {
        return count;
    }

    public void setThreadCount(int count)
    {
        this.threadCount = count;
    }

    public int getThreadCount()
    {
        return threadCount;
    }

    public int getSize()
    {
        return size;
    }

    public long getTotal()
    {
        return total;
    }

    public long getAverageTime()
    {
        return averageTime;
    }

    public long getMeanTime()
    {
        return meanTime;
    }

    public long getFastestTime()
    {
        return fastest;
    }

    public long getSlowestTime()
    {
        return slowest;
    }

    public int getNrOfErrors()
    {
        return errors;
    }

    public int getNrOfConnectErrors()
    {
        return connectErrors;
    }

    public Long[] getTimes()
    {
        return times;
    }

    // Action implementation -----------------------------------------
    public void doValidation()
    {
        try
        {
            url = new URL(urlString);
        }
        catch (MalformedURLException e)
        {
            addError("url", "Invalid URL");
        }
    }

    public String doExecute()
            throws Exception
    {
        // Hit URL
        times = new Long[count];
        long start = System.currentTimeMillis();
        for (int i = 0; i < threadCount; i++)
        {
            // Start a bunch of threads
            new Thread(new Runnable()
            {
                public void run()
                {
                    int idx;
                    while ((idx = getNextIndex()) < count)
                    {
                        int tries = 3; // Try three times before giving up
                        while (--tries > 0)
                        {
                            try
                            {
                                // Get data
                                long current = System.currentTimeMillis();
                                InputStream in = new BufferedInputStream(url.openStream());

                                byte[] data = new byte[2000];
                                int sz;
                                while ((sz = in.read(data)) != -1)
                                {
                                    size += sz;
                                }

                                long now = System.currentTimeMillis();
                                addTime(idx, now - current);
                                break;
                            }
                            catch (java.net.ConnectException e)
                            {
                                // Retry
                                addConnectError();
                            }
                            catch (Exception e)
                            {
                                System.out.println("Error at:" + idx);
                                e.printStackTrace();
                                addTime(idx, EXCEPTION);
                                addError();
                                break;
                            }
                        }
                        // Couldn't get a connection
                        if (tries == 0)
                        {
                            addTime(idx, NO_CONNECT);
                        }
                    }
                }
            }).start();
        }

        // Wait for threads to finish
        synchronized (this)
        {
            while (done < count)
            {
                this.wait();
            }
        }

        // Calc stats
        ArrayList meanTimes = new ArrayList();
        for (int i = 0; i < count; i++)
        {
//	      System.out.println(i+"="+times[i]);

            if (times[i].longValue() >= 0) // Check that it's not an error
            {
                if (times[i].longValue() < fastest || fastest == 0)
                {
                    fastest = times[i].longValue();
                }
                if (times[i].longValue() > slowest)
                {
                    slowest = times[i].longValue();
                }
                total += times[i].longValue();
                meanTimes.add(times[i]);
            }
        }

        if (meanTimes.size() > 0)
        {
            Collections.sort(meanTimes);
            meanTime = ((Long) meanTimes.get(meanTimes.size() / 2)).longValue();
            size /= count;
            averageTime = total / count;
        }

        return SUCCESS;
    }

    private synchronized int getNextIndex()
    {
        return currentIndex++;
    }

    private synchronized void addTime(int idx, long value)
    {
//		System.out.println("Set "+idx+" to "+value);
        times[idx] = new Long(value);
        done++;
        this.notify();
    }

    private synchronized void addError()
    {
        errors++;
    }

    private synchronized void addConnectError()
    {
        connectErrors++;
    }
}
