/*
 * WebWork, Web Application Framework
 *
 * Distributable under Apache license.
 * See terms of license at opensource.org
 */
package webwork.view.taglib;

import webwork.util.BeanUtil;
import webwork.util.injection.ObjectFactory;

import javax.servlet.jsp.JspException;
import javax.servlet.jsp.PageContext;

/**
 * Instantiate a JavaBean.
 * <p/>
 * The bean may be an action, in which it is executed before used. It is lazily executed, which means that you can set
 * parameters by using the "param" tag.
 *
 * @author Rickard Őberg (rickard@dreambean.com)
 * @version $Revision: 1.11 $
 */
public class BeanTag
        extends WebWorkTagSupport
        implements ParamTag.Parametric
{
    // Attributes ----------------------------------------------------
    protected String nameAttr;
    protected Object bean;

    // Public --------------------------------------------------------
    /**
     * Set name of JavaBean class.
     *
     * @param aName class name
     */
    public void setName(String aName)
    {
        nameAttr = aName;
    }

    /**
     * Set property in bean
     */
    public void addParameter(String name, Object value)
    {
        BeanUtil.setProperty(name, value, bean);
    }

    // BodyTag implementation ----------------------------------------
    public int doStartTag() throws JspException
    {
        try
        {
            String name = (String) findValue(nameAttr);
            if (name == null)
            {
                throw new JspException("No bean named " + nameAttr + " found. Try putting quotes around name");
            }
            bean = loadBean(name);
        }
        catch (Exception e)
        {
            try
            {
                pageContext.handlePageException(e);
            }
            catch (Exception ex)
            {
                // Ignore
            }
        }

        getStack().pushValue(bean);

        // Store as attribute?
        if (getId() != null)
        {
            pageContext.setAttribute(getId(), bean);
            pageContext.setAttribute(getId(), bean, PageContext.REQUEST_SCOPE);
        }

        return EVAL_BODY_INCLUDE;
    }

    public int doEndTag() throws JspException
    {
        getStack().popValue();
        bean = null;      // clear bean at the end of the request - WW-797
        super.doEndTag();
        return EVAL_PAGE;
    }

    protected Object loadBean(String name) throws Exception
    {
        return ObjectFactory.instantiateBean(Thread.currentThread().getContextClassLoader(), name);
    }
}
