package webwork.view.velocity;

import java.io.InputStream;
import javax.servlet.ServletContext;

import org.apache.velocity.runtime.resource.loader.ResourceLoader;
import org.apache.velocity.runtime.resource.Resource;
import org.apache.velocity.exception.ResourceNotFoundException;
import org.apache.commons.collections.ExtendedProperties;

/**
 * Resource loader that uses the ServletContext of a webapp to load Velocity templates.  (it's much easier to use with
 * servlets than the standard FileResourceLoader, in particular the use of war files is transparent).
 * <p/>
 * The default search path is '/' (relative to the webapp root), but you can change this behaviour by specifying one or
 * more paths by mean of as many webapp.resource.loader.path properties as needed in the velocity.properties file.
 * <p/>
 * All paths must be relative to the root of the webapp.
 * <p/>
 * NOTE: Shamelessly stolen from velocity-tools sources. We include it because it's a lot more civilised that lugging in
 * all of velo-tools just for a resource loader.
 *
 * @author <a href="mailto:geirm@optonline.net">Geir Magnusson Jr.</a>
 * @author <a href="mailto:nathan@esha.com">Nathan Bubna</a>
 * @author <a href="mailto:claude@savoirweb.com">Claude Brisson</a>
 * @version $Id: WebAppLoader.java,v 1.2 2003/11/16 04:54:53 fate Exp $
 */
public class WebAppLoader extends ResourceLoader
{

    /**
     * The root paths for templates (relative to webapp's root).
     */
    protected String[] paths = null;

    protected ServletContext servletContext = null;

    /**
     * This is abstract in the base class, so we need it. <br> NOTE: this expects that the ServletContext has already
     * been placed in the runtime's application attributes under its full class name (i.e.
     * "javax.servlet.ServletContext").
     *
     * @param configuration the {@link org.apache.commons.collections.ExtendedProperties} associated with this resource
     *                      loader.
     */
    public void init(ExtendedProperties configuration)
    {
        rsvc.info("WebappLoader : initialization starting.");

        /* get configured paths */
        paths = configuration.getStringArray("path");
        if (paths == null || paths.length == 0)
        {
            paths = new String[1];
            paths[0] = "/";
        }
        else
        {
            /* make sure the paths end with a '/' */
            for (int i = 0; i < paths.length; i++)
            {
                if (!paths[i].endsWith("/"))
                {
                    paths[i] += '/';
                }
                rsvc.info("WebappLoader : added template path - '" + paths[i] + "'");
            }
        }

        /* get the ServletContext */
        Object obj = rsvc.getApplicationAttribute(ServletContext.class.getName());
        if (obj instanceof ServletContext)
        {
            servletContext = (ServletContext) obj;
        }
        else
        {
            rsvc.error("WebappLoader : unable to retrieve ServletContext");
        }

        rsvc.info("WebappLoader : initialization complete.");
    }

    /**
     * Get an InputStream so that the Runtime can build a template with it.
     *
     * @param name name of template to get
     *
     * @return InputStream containing the template
     *
     * @throws org.apache.velocity.exception.ResourceNotFoundException
     *          if template not found in  classpath.
     */
    public synchronized InputStream getResourceStream(String name) throws ResourceNotFoundException
    {
        InputStream result = null;
        if (name == null || name.length() == 0)
        {
            throw new ResourceNotFoundException("WebappLoader : No template name provided");
        }

        /* since the paths always ends in '/',
* make sure the name never ends in one */
        while (name.startsWith("/"))
        {
            name = name.substring(1);
        }

        Exception exception = null;
        for (int i = 0; i < paths.length; i++)
        {
            try
            {
                result = servletContext.getResourceAsStream(paths[i] + name);

                /* exit the loop if we found the template */
                if (result != null)
                {
                    break;
                }
            }
            catch (Exception e)
            {
                /* only save the first one for later throwing */
                if (exception == null)
                {
                    exception = e;
                }
            }
        }

        /* if we never found the template */
        if (result == null)
        {
            String msg;
            if (exception == null)
            {
                msg = "WebappLoader : Resource '" + name + "' not found.";
            }
            else
            {
                msg = exception.getMessage();
            }
            /* convert to a general Velocity ResourceNotFoundException */
            throw new ResourceNotFoundException(msg);
        }

        return result;
    }

    /**
     * Defaults to return false.
     */
    public boolean isSourceModified(Resource resource)
    {
        return false;
    }

    /**
     * Defaults to return 0
     */
    public long getLastModified(Resource resource)
    {
        return 0;
    }
}
