package webwork.view.xslt;

import org.apache.commons.logging.*;
import org.w3c.dom.Attr;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;
import webwork.util.ValueStack;
import webwork.util.InjectionUtils;

import java.beans.IntrospectionException;
import java.beans.Introspector;
import java.beans.PropertyDescriptor;
import java.lang.reflect.Array;
import java.lang.reflect.Method;
import java.util.*;

public class XMLResultAdapter
{
    Map classes; // Method cache
    Map pdCache; // Property Descriptor Cache

    public XMLResultAdapter()
    {
        classes = new HashMap();
        pdCache = new HashMap();
    }

    public void markupObject(StringBuffer buff, Object o, String propertyName)
    {
        markupObject(buff, new Stack(), o, propertyName);
    }

    protected void markupObject(StringBuffer buff, Stack visitedObjects, Object o, String propertyName)
    {
        //FIX boolean[] types!
        if (o == null)
        {
            buff.append("null");

        }
        else if (isBasicType(o))
        {
            markupBasicType(buff, visitedObjects, o, propertyName);
        }
        else if (o instanceof String)
        {
            markupString(buff, visitedObjects, (String) o, propertyName);
        }
        else if (o instanceof Properties)
        {
            markupProperties(buff, visitedObjects, (Properties) o, propertyName);
        }
        else if (o instanceof Map)
        {
            markupMap(buff, visitedObjects, (Map) o, propertyName);
        }
        else if (o instanceof Collection)
        {
            markupCollection(buff, visitedObjects, (Collection) o, propertyName);
        }
        else if (o instanceof Node)
        {
            markupNode(buff, visitedObjects, (Node) o, propertyName);
        }
        else if (o.getClass().isArray())
        {
            markupArray(buff, visitedObjects, o, propertyName);
        }
        else
        {
            markupBean(buff, visitedObjects, o, propertyName);
        }
    }

    protected void markupBean(StringBuffer buff, Stack visitedObjects, Object object, String propertyName)
    {
        visitedObjects.push(object);
        try
        {
            PropertyDescriptor[] pd = getPropertyDescriptors(object);
            if (pd.length > 0)
            {
                //This is a bean
                buff.append("<");
                buff.append(propertyName);
                buff.append(">");
                properties:
                for (int i = 0; i < pd.length; ++i)
                {
                    try
                    {
                        Method readMethod = pd[i].getReadMethod();
                        // Test if property is readeable (yes, there are write-only props, sometimes)
                        if (readMethod != null)
                        {
                            Object value = InjectionUtils.invoke(readMethod, object, null);
                            while (value instanceof ValueStack.ValueHolder)
                            {
                                value = ((ValueStack.ValueHolder) value).getValue();
                            }

                            if (visitedObjects.contains(value))
                            {
                                LogFactory.getLog(this.getClass()).debug("Skipping duplicate" + value + "[" + value.getClass() + "]");
                                continue;
                            }
                            markupObject(buff, visitedObjects, value, pd[i].getName());
                        }
                    }
                    catch (java.lang.reflect.InvocationTargetException e)
                    {
                        LogFactory.getLog(this.getClass()).error(e.getTargetException());
                    }
                    catch (java.lang.IllegalAccessException e)
                    {
                        LogFactory.getLog(this.getClass()).error(e);

                    }

                }

                buff.append("</");

                buff.append(propertyName);

                buff.append(">");

            }
            else
            {

                markupAny(buff, visitedObjects, object, propertyName);

            }

        }
        catch (java.beans.IntrospectionException e)
        {

            LogFactory.getLog(this.getClass()).error(e.getMessage());

        }


        visitedObjects.pop();

    }


    protected void markupBasicType(StringBuffer buff, Stack visitedObjects, Object object, String propertyName)
    {

        buff.append("<");

        buff.append(propertyName);

        buff.append(">");

        buff.append(object.toString());

        buff.append("</");

        buff.append(propertyName);

        buff.append(">");

    }


    protected void markupString(StringBuffer buff, Stack visitedObjects, String object, String propertyName)
    {

        buff.append("<");

        buff.append(propertyName);

        buff.append(">");

        buff.append(object);

        buff.append("</");

        buff.append(propertyName);

        buff.append(">");

    }


    protected void markupAny(StringBuffer buff, Stack visitedObjects, Object object, String propertyName)
    {

        buff.append("<");

        buff.append(propertyName);

        buff.append(">");

        buff.append(object.toString());

        buff.append("</");

        buff.append(propertyName);

        buff.append(">");

    }


    protected void markupMap(StringBuffer buff, Stack visitedObjects, Map map, String propertyName)
    {

        buff.append("<");

        buff.append(propertyName);

        buff.append(">");

        for (Iterator i = map.entrySet().iterator(); i.hasNext();)
        {

            Map.Entry e = (Map.Entry) i.next();

            buff.append("<entry>");

            markupObject(buff, visitedObjects, e.getKey(), "key");

            markupObject(buff, visitedObjects, e.getValue(), "value");

            buff.append("</entry>");

        }

        buff.append("</");

        buff.append(propertyName);

        buff.append(">");

    }


    protected void markupProperties(StringBuffer buff, Stack visitedObjects, Properties props, String propertyName)
    {

        buff.append("<");

        buff.append(propertyName);

        buff.append(">");

        for (Enumeration e = props.propertyNames(); e.hasMoreElements();)
        {

            String key = (String) e.nextElement();

            buff.append("<entry>");

            markupObject(buff, visitedObjects, key, "key");

            markupObject(buff, visitedObjects, props.getProperty(key), "value");

            buff.append("</entry>");

        }

        buff.append("</");

        buff.append(propertyName);

        buff.append(">");

    }


    protected void markupCollection(StringBuffer buff, Stack visitedObjects, Collection col, String propertyName)
    {

        buff.append("<");

        buff.append(propertyName);

        buff.append(">");

        for (Iterator i = col.iterator(); i.hasNext();)
        {

            markupObject(buff, visitedObjects, i.next(), "item");

        }

        buff.append("</");

        buff.append(propertyName);

        buff.append(">");

    }


    protected void markupArray(StringBuffer buff, Stack visitedObjects, Object o, String propertyName)
    {

        buff.append("<");

        buff.append(propertyName);

        buff.append(">");

        for (int i = 0; i < Array.getLength(o); i++)
        {

            markupObject(buff, visitedObjects, Array.get(o, i), "item");

        }

        buff.append("</");

        buff.append(propertyName);

        buff.append(">");

    }


    protected void markupNode(StringBuffer buff, Stack visitedObjects, Node node, String propertyName)
    {

        if (propertyName != null)
        {

            buff.append("<");

            buff.append(propertyName);

            buff.append(">");

        }

        // is there anything to do?
        if (node == null)
        {

            return;

        }


        int type = node.getNodeType();

        boolean canonical = false;

        switch (type)
        {

            // print document
            case Node.DOCUMENT_NODE:

            {


                NodeList children = node.getChildNodes();

                for (int iChild = 0; iChild < children.getLength(); iChild++)
                {

                    markupNode(buff, visitedObjects, children.item(iChild), null);

                }

                break;

            }

            // print element with attributes
            case Node.ELEMENT_NODE:

            {

                buff.append('<');

                buff.append(node.getNodeName());

                Attr attrs[] = sortAttributes(node.getAttributes());

                for (int i = 0; i < attrs.length; i++)
                {

                    Attr attr = attrs[i];

                    buff.append(' ');

                    buff.append(attr.getNodeName());

                    buff.append("=\"");

                    buff.append(normalize(attr.getNodeValue()));

                    buff.append('"');

                }

                buff.append('>');

                NodeList children = node.getChildNodes();

                if (children != null)
                {

                    int len = children.getLength();

                    for (int i = 0; i < len; i++)
                    {

                        markupNode(buff, visitedObjects, children.item(i), null);

                    }

                }

                break;

            }

            // handle entity reference nodes
            case Node.ENTITY_REFERENCE_NODE:

            {

                if (canonical)
                {

                    NodeList children = node.getChildNodes();

                    if (children != null)
                    {

                        int len = children.getLength();

                        for (int i = 0; i < len; i++)
                        {

                            markupNode(buff, visitedObjects, children.item(i), null);

                        }

                    }

                }
                else
                {

                    buff.append('&');

                    buff.append(node.getNodeName());

                    buff.append(';');

                }

                break;

            }

            // print cdata sections
            case Node.CDATA_SECTION_NODE:

            {

                if (canonical)
                {

                    buff.append(normalize(node.getNodeValue()));

                }
                else
                {

                    buff.append("<![CDATA[");

                    buff.append(node.getNodeValue());

                    buff.append("]]>");

                }

                break;

            }

            // print text
            case Node.TEXT_NODE:

            {

                buff.append(normalize(node.getNodeValue()));

                break;

            }

            // print processing instruction
            case Node.PROCESSING_INSTRUCTION_NODE:

            {

                buff.append("<?");

                buff.append(node.getNodeName());

                String data = node.getNodeValue();

                if (data != null && data.length() > 0)
                {

                    buff.append(' ');

                    buff.append(data);

                }

                buff.append("?>");

                break;

            }

        }


        if (type == Node.ELEMENT_NODE)
        {

            buff.append("</");

            buff.append(node.getNodeName());

            buff.append('>');

        }


        if (propertyName != null)
        {

            buff.append("<");

            buff.append(propertyName);

            buff.append(">");

        }


    }


    protected Attr[] sortAttributes(NamedNodeMap attrs)
    {


        int len = (attrs != null) ? attrs.getLength() : 0;

        Attr array[] = new Attr[len];

        for (int i = 0; i < len; i++)
        {

            array[i] = (Attr) attrs.item(i);

        }

        for (int i = 0; i < len - 1; i++)
        {

            String name = array[i].getNodeName();

            int index = i;

            for (int j = i + 1; j < len; j++)
            {

                String curName = array[j].getNodeName();

                if (curName.compareTo(name) < 0)
                {

                    name = curName;

                    index = j;

                }

            }

            if (index != i)
            {

                Attr temp = array[i];

                array[i] = array[index];

                array[index] = temp;

            }

        }


        return (array);


    } // sortAttributes(NamedNodeMap):Attr[]


    /**
     * Normalizes the given string.
     */

    protected String normalize(String s)
    {

        StringBuffer str = new StringBuffer();


        int len = (s != null) ? s.length() : 0;

        for (int i = 0; i < len; i++)
        {

            char ch = s.charAt(i);

            switch (ch)
            {

                case '<':

                {

                    str.append("&lt;");

                    break;

                }

                case '>':

                {

                    str.append("&gt;");

                    break;

                }

                case '&':

                {

                    str.append("&amp;");

                    break;

                }

                case '"':

                {

                    str.append("&quot;");

                    break;

                }

                case '\'':

                {

                    str.append("&apos;");

                    break;

                }

                case '\r':

                case '\n':

                {

                    str.append("&#");

                    str.append(Integer.toString(ch));

                    str.append(';');

                    break;
                    // else, default append char

                }

                default:

                {

                    str.append(ch);

                }

            }

        }


        return (str.toString());


    } // normalize(String):String


    protected boolean isBasicType(Object o)
    {

        Class type = o.getClass();

        return ((type.equals(Integer.TYPE) ||

                type.equals(Long.TYPE) ||

                type.equals(Short.TYPE) ||

                type.equals(Byte.TYPE) ||

                type.equals(Boolean.TYPE) ||

                type.equals(Character.TYPE) ||

                type.equals(Float.TYPE) ||

                type.equals(Double.TYPE) ||

                Integer.class.isAssignableFrom(type) ||

                Long.class.isAssignableFrom(type) ||

                Short.class.isAssignableFrom(type) ||

                Byte.class.isAssignableFrom(type) ||

                Boolean.class.isAssignableFrom(type) ||

                Float.class.isAssignableFrom(type) ||

                Character.class.isAssignableFrom(type) ||

                Double.class.isAssignableFrom(type)));

    }


    private PropertyDescriptor[] getPropertyDescriptors(Object o) throws IntrospectionException
    {

        PropertyDescriptor[] pd = (PropertyDescriptor[]) pdCache.get(o.getClass());

        if (pd == null)
        {

            pd = Introspector.getBeanInfo(o.getClass(), Object.class).getPropertyDescriptors();

            pdCache.put(o.getClass(), pd);

        }

        return pd;

    }

}
