/*
 * Decompiled with CFR 0.152.
 */
package org.apache.kafka.tools;

import com.fasterxml.jackson.core.JsonProcessingException;
import com.fasterxml.jackson.databind.ObjectMapper;
import java.text.DateFormat;
import java.text.SimpleDateFormat;
import java.time.Duration;
import java.util.Collection;
import java.util.Collections;
import java.util.Date;
import java.util.HashMap;
import java.util.Map;
import java.util.Properties;
import java.util.Random;
import java.util.concurrent.atomic.AtomicBoolean;
import java.util.concurrent.atomic.AtomicLong;
import net.sourceforge.argparse4j.ArgumentParsers;
import net.sourceforge.argparse4j.impl.Arguments;
import net.sourceforge.argparse4j.inf.ArgumentAction;
import net.sourceforge.argparse4j.inf.ArgumentParser;
import net.sourceforge.argparse4j.inf.Namespace;
import org.apache.kafka.clients.consumer.ConsumerRebalanceListener;
import org.apache.kafka.clients.consumer.ConsumerRecord;
import org.apache.kafka.clients.consumer.ConsumerRecords;
import org.apache.kafka.clients.consumer.KafkaConsumer;
import org.apache.kafka.clients.consumer.OffsetAndMetadata;
import org.apache.kafka.clients.producer.KafkaProducer;
import org.apache.kafka.clients.producer.ProducerRecord;
import org.apache.kafka.common.KafkaException;
import org.apache.kafka.common.TopicPartition;
import org.apache.kafka.common.errors.OutOfOrderSequenceException;
import org.apache.kafka.common.errors.ProducerFencedException;
import org.apache.kafka.common.utils.Exit;

public class TransactionalMessageCopier {
    private static final DateFormat FORMAT = new SimpleDateFormat("yyyy/MM/dd HH:mm:ss:SSS");

    private static ArgumentParser argParser() {
        ArgumentParser parser = ArgumentParsers.newArgumentParser((String)"transactional-message-copier").defaultHelp(true).description("This tool copies messages transactionally from an input partition to an output topic, committing the consumed offsets along with the output messages");
        parser.addArgument(new String[]{"--input-topic"}).action((ArgumentAction)Arguments.store()).required(true).type(String.class).metavar(new String[]{"INPUT-TOPIC"}).dest("inputTopic").help("Consume messages from this topic");
        parser.addArgument(new String[]{"--input-partition"}).action((ArgumentAction)Arguments.store()).required(true).type(Integer.class).metavar(new String[]{"INPUT-PARTITION"}).dest("inputPartition").help("Consume messages from this partition of the input topic.");
        parser.addArgument(new String[]{"--output-topic"}).action((ArgumentAction)Arguments.store()).required(true).type(String.class).metavar(new String[]{"OUTPUT-TOPIC"}).dest("outputTopic").help("Produce messages to this topic");
        parser.addArgument(new String[]{"--broker-list"}).action((ArgumentAction)Arguments.store()).required(true).type(String.class).metavar(new String[]{"HOST1:PORT1[,HOST2:PORT2[...]]"}).dest("brokerList").help("Comma-separated list of Kafka brokers in the form HOST1:PORT1,HOST2:PORT2,...");
        parser.addArgument(new String[]{"--max-messages"}).action((ArgumentAction)Arguments.store()).required(false).setDefault((Object)-1).type(Integer.class).metavar(new String[]{"MAX-MESSAGES"}).dest("maxMessages").help("Process these many messages upto the end offset at the time this program was launched. If set to -1 we will just read to the end offset of the input partition (as of the time the program was launched).");
        parser.addArgument(new String[]{"--consumer-group"}).action((ArgumentAction)Arguments.store()).required(false).setDefault((Object)-1).type(String.class).metavar(new String[]{"CONSUMER-GROUP"}).dest("consumerGroup").help("The consumer group id to use for storing the consumer offsets.");
        parser.addArgument(new String[]{"--transaction-size"}).action((ArgumentAction)Arguments.store()).required(false).setDefault((Object)200).type(Integer.class).metavar(new String[]{"TRANSACTION-SIZE"}).dest("messagesPerTransaction").help("The number of messages to put in each transaction. Default is 200.");
        parser.addArgument(new String[]{"--transaction-timeout"}).action((ArgumentAction)Arguments.store()).required(false).setDefault((Object)60000).type(Integer.class).metavar(new String[]{"TRANSACTION-TIMEOUT"}).dest("transactionTimeout").help("The transaction timeout in milliseconds. Default is 60000(1 minute).");
        parser.addArgument(new String[]{"--transactional-id"}).action((ArgumentAction)Arguments.store()).required(true).type(String.class).metavar(new String[]{"TRANSACTIONAL-ID"}).dest("transactionalId").help("The transactionalId to assign to the producer");
        parser.addArgument(new String[]{"--enable-random-aborts"}).action((ArgumentAction)Arguments.storeTrue()).type(Boolean.class).metavar(new String[]{"ENABLE-RANDOM-ABORTS"}).dest("enableRandomAborts").help("Whether or not to enable random transaction aborts (for system testing)");
        parser.addArgument(new String[]{"--group-mode"}).action((ArgumentAction)Arguments.storeTrue()).type(Boolean.class).metavar(new String[]{"GROUP-MODE"}).dest("groupMode").help("Whether to let consumer subscribe to the input topic or do manual assign. If we do subscription based consumption, the input partition shall be ignored");
        parser.addArgument(new String[]{"--use-group-metadata"}).action((ArgumentAction)Arguments.storeTrue()).type(Boolean.class).metavar(new String[]{"USE-GROUP-METADATA"}).dest("useGroupMetadata").help("Whether to use the new transactional commit API with group metadata");
        return parser;
    }

    private static KafkaProducer<String, String> createProducer(Namespace parsedArgs) {
        Properties props = new Properties();
        props.put("bootstrap.servers", parsedArgs.getString("brokerList"));
        props.put("transactional.id", parsedArgs.getString("transactionalId"));
        props.put("key.serializer", "org.apache.kafka.common.serialization.StringSerializer");
        props.put("value.serializer", "org.apache.kafka.common.serialization.StringSerializer");
        props.put("batch.size", "512");
        props.put("max.in.flight.requests.per.connection", "5");
        props.put("transaction.timeout.ms", parsedArgs.getInt("transactionTimeout"));
        return new KafkaProducer(props);
    }

    private static KafkaConsumer<String, String> createConsumer(Namespace parsedArgs) {
        String consumerGroup = parsedArgs.getString("consumerGroup");
        String brokerList = parsedArgs.getString("brokerList");
        Integer numMessagesPerTransaction = parsedArgs.getInt("messagesPerTransaction");
        Properties props = new Properties();
        props.put("group.id", consumerGroup);
        props.put("bootstrap.servers", brokerList);
        props.put("isolation.level", "read_committed");
        props.put("max.poll.records", numMessagesPerTransaction.toString());
        props.put("enable.auto.commit", "false");
        props.put("session.timeout.ms", "10000");
        props.put("max.poll.interval.ms", "180000");
        props.put("heartbeat.interval.ms", "3000");
        props.put("auto.offset.reset", "earliest");
        props.put("key.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        props.put("value.deserializer", "org.apache.kafka.common.serialization.StringDeserializer");
        return new KafkaConsumer(props);
    }

    private static ProducerRecord<String, String> producerRecordFromConsumerRecord(String topic, ConsumerRecord<String, String> record) {
        return new ProducerRecord(topic, Integer.valueOf(record.partition()), record.key(), record.value());
    }

    private static Map<TopicPartition, OffsetAndMetadata> consumerPositions(KafkaConsumer<String, String> consumer) {
        HashMap<TopicPartition, OffsetAndMetadata> positions = new HashMap<TopicPartition, OffsetAndMetadata>();
        for (TopicPartition topicPartition : consumer.assignment()) {
            positions.put(topicPartition, new OffsetAndMetadata(consumer.position(topicPartition), null));
        }
        return positions;
    }

    private static void resetToLastCommittedPositions(KafkaConsumer<String, String> consumer) {
        Map committed = consumer.committed(consumer.assignment());
        consumer.assignment().forEach(tp -> {
            OffsetAndMetadata offsetAndMetadata = (OffsetAndMetadata)committed.get(tp);
            if (offsetAndMetadata != null) {
                consumer.seek(tp, offsetAndMetadata.offset());
            } else {
                consumer.seekToBeginning(Collections.singleton(tp));
            }
        });
    }

    private static long messagesRemaining(KafkaConsumer<String, String> consumer, TopicPartition partition) {
        long currentPosition = consumer.position(partition);
        Map endOffsets = consumer.endOffsets(Collections.singleton(partition));
        if (endOffsets.containsKey(partition)) {
            return (Long)endOffsets.get(partition) - currentPosition;
        }
        return 0L;
    }

    private static String toJsonString(Map<String, Object> data) {
        String json;
        try {
            ObjectMapper mapper = new ObjectMapper();
            json = mapper.writeValueAsString(data);
        }
        catch (JsonProcessingException e) {
            json = "Bad data can't be written as json: " + e.getMessage();
        }
        return json;
    }

    private static synchronized String statusAsJson(long totalProcessed, long consumedSinceLastRebalanced, long remaining, String transactionalId, String stage) {
        HashMap<String, Object> statusData = new HashMap<String, Object>();
        statusData.put("progress", transactionalId);
        statusData.put("totalProcessed", totalProcessed);
        statusData.put("consumed", consumedSinceLastRebalanced);
        statusData.put("remaining", remaining);
        statusData.put("time", FORMAT.format(new Date()));
        statusData.put("stage", stage);
        return TransactionalMessageCopier.toJsonString(statusData);
    }

    private static synchronized String shutDownString(long totalProcessed, long consumedSinceLastRebalanced, long remaining, String transactionalId) {
        HashMap<String, Object> shutdownData = new HashMap<String, Object>();
        shutdownData.put("shutdown_complete", transactionalId);
        shutdownData.put("totalProcessed", totalProcessed);
        shutdownData.put("consumed", consumedSinceLastRebalanced);
        shutdownData.put("remaining", remaining);
        shutdownData.put("time", FORMAT.format(new Date()));
        return TransactionalMessageCopier.toJsonString(shutdownData);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    public static void main(String[] args) {
        Namespace parsedArgs = TransactionalMessageCopier.argParser().parseArgsOrFail(args);
        final String transactionalId = parsedArgs.getString("transactionalId");
        String outputTopic = parsedArgs.getString("outputTopic");
        String consumerGroup = parsedArgs.getString("consumerGroup");
        KafkaProducer<String, String> producer = TransactionalMessageCopier.createProducer(parsedArgs);
        final KafkaConsumer<String, String> consumer = TransactionalMessageCopier.createConsumer(parsedArgs);
        final AtomicLong remainingMessages = new AtomicLong(parsedArgs.getInt("maxMessages") == -1 ? Long.MAX_VALUE : (long)parsedArgs.getInt("maxMessages").intValue());
        boolean groupMode = parsedArgs.getBoolean("groupMode");
        String topicName = parsedArgs.getString("inputTopic");
        final AtomicLong numMessagesProcessedSinceLastRebalance = new AtomicLong(0L);
        final AtomicLong totalMessageProcessed = new AtomicLong(0L);
        if (groupMode) {
            consumer.subscribe(Collections.singleton(topicName), new ConsumerRebalanceListener(){

                public void onPartitionsRevoked(Collection<TopicPartition> partitions) {
                }

                public void onPartitionsAssigned(Collection<TopicPartition> partitions) {
                    remainingMessages.set(partitions.stream().mapToLong(partition -> TransactionalMessageCopier.messagesRemaining((KafkaConsumer<String, String>)consumer, partition)).sum());
                    numMessagesProcessedSinceLastRebalance.set(0L);
                    System.out.println(TransactionalMessageCopier.statusAsJson(totalMessageProcessed.get(), numMessagesProcessedSinceLastRebalance.get(), remainingMessages.get(), transactionalId, "RebalanceComplete"));
                }
            });
        } else {
            TopicPartition inputPartition = new TopicPartition(topicName, parsedArgs.getInt("inputPartition").intValue());
            consumer.assign(Collections.singleton(inputPartition));
            remainingMessages.set(Math.min(TransactionalMessageCopier.messagesRemaining(consumer, inputPartition), remainingMessages.get()));
        }
        boolean enableRandomAborts = parsedArgs.getBoolean("enableRandomAborts");
        producer.initTransactions();
        AtomicBoolean isShuttingDown = new AtomicBoolean(false);
        Exit.addShutdownHook((String)"transactional-message-copier-shutdown-hook", () -> {
            isShuttingDown.set(true);
            producer.close();
            KafkaConsumer kafkaConsumer = consumer;
            synchronized (kafkaConsumer) {
                consumer.close();
            }
            System.out.println(TransactionalMessageCopier.shutDownString(totalMessageProcessed.get(), numMessagesProcessedSinceLastRebalance.get(), remainingMessages.get(), transactionalId));
        });
        boolean useGroupMetadata = parsedArgs.getBoolean("useGroupMetadata");
        try {
            Random random = new Random();
            while (remainingMessages.get() > 0L) {
                System.out.println(TransactionalMessageCopier.statusAsJson(totalMessageProcessed.get(), numMessagesProcessedSinceLastRebalance.get(), remainingMessages.get(), transactionalId, "ProcessLoop"));
                if (isShuttingDown.get()) {
                    break;
                }
                ConsumerRecords records = consumer.poll(Duration.ofMillis(200L));
                if (records.count() <= 0) continue;
                try {
                    producer.beginTransaction();
                    for (ConsumerRecord record : records) {
                        producer.send(TransactionalMessageCopier.producerRecordFromConsumerRecord(outputTopic, (ConsumerRecord<String, String>)record));
                    }
                    long messagesSentWithinCurrentTxn = records.count();
                    if (useGroupMetadata) {
                        producer.sendOffsetsToTransaction(TransactionalMessageCopier.consumerPositions(consumer), consumer.groupMetadata());
                    } else {
                        producer.sendOffsetsToTransaction(TransactionalMessageCopier.consumerPositions(consumer), consumerGroup);
                    }
                    if (enableRandomAborts && random.nextInt() % 3 == 0) {
                        throw new KafkaException("Aborting transaction");
                    }
                    producer.commitTransaction();
                    remainingMessages.getAndAdd(-messagesSentWithinCurrentTxn);
                    numMessagesProcessedSinceLastRebalance.getAndAdd(messagesSentWithinCurrentTxn);
                    totalMessageProcessed.getAndAdd(messagesSentWithinCurrentTxn);
                }
                catch (OutOfOrderSequenceException | ProducerFencedException e) {
                    throw e;
                }
                catch (KafkaException e) {
                    producer.abortTransaction();
                    TransactionalMessageCopier.resetToLastCommittedPositions(consumer);
                }
            }
        }
        finally {
            producer.close();
            KafkaConsumer<String, String> kafkaConsumer = consumer;
            synchronized (kafkaConsumer) {
                consumer.close();
            }
        }
        System.exit(0);
    }
}

