/*
 * Decompiled with CFR 0.152.
 */
package org.sonar.batch.scan.filesystem;

import com.google.common.base.Joiner;
import com.google.common.collect.Lists;
import com.google.common.collect.Maps;
import java.text.MessageFormat;
import java.util.List;
import java.util.Map;
import javax.annotation.CheckForNull;
import org.apache.commons.lang.StringUtils;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;
import org.sonar.api.batch.fs.InputFile;
import org.sonar.api.batch.fs.internal.PathPattern;
import org.sonar.api.config.Settings;
import org.sonar.api.utils.MessageException;
import org.sonar.batch.languages.Language;
import org.sonar.batch.languages.LanguagesReferential;

class LanguageDetection {
    private static final Logger LOG = LoggerFactory.getLogger(LanguageDetection.class);
    private final Map<String, PathPattern[]> patternsByLanguage = Maps.newLinkedHashMap();
    private final List<String> languagesToConsider = Lists.newArrayList();
    private final String forcedLanguage;

    LanguageDetection(Settings settings, LanguagesReferential languages) {
        for (Language language : languages.all()) {
            String[] filePatterns = settings.getStringArray(this.getFileLangPatternPropKey(language.key()));
            PathPattern[] pathPatterns = PathPattern.create((String[])filePatterns);
            if (pathPatterns.length > 0) {
                this.patternsByLanguage.put(language.key(), pathPatterns);
                continue;
            }
            String[] patterns = language.fileSuffixes().toArray(new String[language.fileSuffixes().size()]);
            for (int i = 0; i < patterns.length; ++i) {
                String suffix = patterns[i];
                String extension = LanguageDetection.sanitizeExtension(suffix);
                patterns[i] = "**/*." + extension;
            }
            PathPattern[] defaultLanguagePatterns = PathPattern.create((String[])patterns);
            this.patternsByLanguage.put(language.key(), defaultLanguagePatterns);
            LOG.debug("Declared extensions of language " + language + " were converted to " + this.getDetails(language.key()));
        }
        this.forcedLanguage = StringUtils.defaultIfBlank((String)settings.getString("sonar.language"), null);
        if (this.forcedLanguage != null) {
            if (!this.patternsByLanguage.containsKey(this.forcedLanguage)) {
                throw MessageException.of((String)("No language is installed with key '" + this.forcedLanguage + "'. Please update property '" + "sonar.language" + "'"));
            }
            this.languagesToConsider.add(this.forcedLanguage);
        } else {
            this.languagesToConsider.addAll(this.patternsByLanguage.keySet());
        }
    }

    Map<String, PathPattern[]> patternsByLanguage() {
        return this.patternsByLanguage;
    }

    @CheckForNull
    String language(InputFile inputFile) {
        String detectedLanguage = null;
        for (String languageKey : this.languagesToConsider) {
            if (!this.isCandidateForLanguage(inputFile, languageKey)) continue;
            if (detectedLanguage == null) {
                detectedLanguage = languageKey;
                continue;
            }
            throw MessageException.of((String)MessageFormat.format("Language of file ''{0}'' can not be decided as the file matches patterns of both {1} and {2}", inputFile.relativePath(), this.getDetails(detectedLanguage), this.getDetails(languageKey)));
        }
        if (detectedLanguage != null) {
            LOG.debug(String.format("Language of file '%s' is detected to be '%s'", inputFile.relativePath(), detectedLanguage));
            return detectedLanguage;
        }
        if (this.forcedLanguage != null && this.patternsByLanguage.get(this.forcedLanguage).length == 0) {
            return this.forcedLanguage;
        }
        return null;
    }

    private boolean isCandidateForLanguage(InputFile inputFile, String languageKey) {
        PathPattern[] patterns = this.patternsByLanguage.get(languageKey);
        if (patterns != null) {
            for (PathPattern pathPattern : patterns) {
                if (!pathPattern.match(inputFile, false)) continue;
                return true;
            }
        }
        return false;
    }

    private String getFileLangPatternPropKey(String languageKey) {
        return "sonar.lang.patterns." + languageKey;
    }

    private String getDetails(String detectedLanguage) {
        return this.getFileLangPatternPropKey(detectedLanguage) + " : " + Joiner.on((String)",").join((Object[])this.patternsByLanguage.get(detectedLanguage));
    }

    static String sanitizeExtension(String suffix) {
        return StringUtils.lowerCase((String)StringUtils.removeStart((String)suffix, (String)"."));
    }
}

