/*
 * (C) Copyright 2013 Kurento (http://kurento.org/)
 *
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the GNU Lesser General Public License
 * (LGPL) version 2.1 which accompanies this distribution, and is available at
 * http://www.gnu.org/licenses/lgpl-2.1.html
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
 * Lesser General Public License for more details.
 *
 */

package org.kurento.repository;

import java.util.List;

/**
 * This interface represents the main interface to interact with a repository of items. In the
 * contexto of Kurento Media Framework, the repository is intended to store media files, but can be
 * used to store any kind of binary content.
 *
 * The items are represented as instances of {@link RepositoryItem} obtained with methods of this
 * interface. This interface offers several CRUD operations to manage repository items.
 *
 * @author Micael Gallego (micael.gallego@gmail.com)
 */
public interface Repository {

  /**
   * Creates a new repository item with an autogenerated unique id.
   * 
   * @return the new repository item.
   */
  RepositoryItem createRepositoryItem();

  /**
   * Creates a new repository item with the specified id. This id must be unique. If the id is used
   * for another repository item a {@link DuplicateItemException} is thrown.
   * 
   * @param id
   *          the identifier of the new repository item.
   * 
   * @return the new repository item.
   */
  RepositoryItem createRepositoryItem(String id);

  /**
   * Returns the repository item identified by the specified id. The id is unique between repository
   * items.
   * 
   * @param id
   *          The id of the repository item that can be returned
   * @return the repository item with the indicated id
   */
  RepositoryItem findRepositoryItemById(String id);

  /**
   * Returns the repository items with the specified value in the specified attribute of metadata's
   * repository item.
   * 
   * @param attribute
   *          attribute name of the metadata
   * @param value
   *          expected value in the attribute
   * @return repository items with the specified value in the attribute
   */
  List<RepositoryItem> findRepositoryItemsByAttValue(String attribute, String value);

  /**
   * Returns the repository items with value that matches the specified regular expression in the
   * specified attribute of metadata's repository item. The regular expression must be formatted
   * with "Perl compatible regular expressions (PCRE)".
   * 
   * Concrete repository implementations can have specific constraints in the regular expressions.
   * Please consult the documentation for more details.
   * 
   * @param attribute
   *          attribute name of the metadata
   * @param regex
   *          regular expression to match the attribute's value
   * @return repository items with an attribute's value that matches the regular expression
   */
  List<RepositoryItem> findRepositoryItemsByAttRegex(String attribute, String regex);

  /**
   * Removes the specified repository item.
   * 
   * @param item
   *          that will be removed.
   */
  void remove(RepositoryItem item);

}
